#!/usr/bin/env python3
"""
TimCash - Script de Démarrage Rapide
Facilite le lancement et la configuration du projet
"""

import os
import sys
import subprocess
import json
import time
from pathlib import Path

class TimCashStarter:
    def __init__(self):
        self.project_root = Path(__file__).parent
        self.backend_dir = self.project_root / "backend"
        
    def print_banner(self):
        """Affiche la bannière TimCash"""
        banner = """
╔══════════════════════════════════════════════════════════════╗
║                                                              ║
║  ████████╗██╗███╗   ███╗ ██████╗ █████╗ ███████╗██╗  ██╗    ║
║  ╚══██╔══╝██║████╗ ████║██╔════╝██╔══██╗██╔════╝██║  ██║    ║
║     ██║   ██║██╔████╔██║██║     ███████║███████╗███████║    ║
║     ██║   ██║██║╚██╔╝██║██║     ██╔══██║╚════██║██╔══██║    ║
║     ██║   ██║██║ ╚═╝ ██║╚██████╗██║  ██║███████║██║  ██║    ║
║     ╚═╝   ╚═╝╚═╝     ╚═╝ ╚═════╝╚═╝  ╚═╝╚══════╝╚═╝  ╚═╝    ║
║                                                              ║
║              Modern Financial Platform v0.1.0               ║
║                                                              ║
╚══════════════════════════════════════════════════════════════╝
        """
        print(banner)
        print("🚀 Démarrage de TimCash - Plateforme Financière Moderne")
        print("=" * 60)
    
    def check_dependencies(self):
        """Vérifie les dépendances système"""
        print("🔍 Vérification des dépendances...")
        
        # Vérifier Python
        if sys.version_info < (3, 11):
            print("❌ Python 3.11+ requis")
            return False
        print("✅ Python", sys.version.split()[0])
        
        # Vérifier pip
        try:
            subprocess.run(["pip", "--version"], check=True, capture_output=True)
            print("✅ pip disponible")
        except:
            print("❌ pip non trouvé")
            return False
        
        # Vérifier Node.js
        try:
            result = subprocess.run(["node", "--version"], check=True, capture_output=True, text=True)
            print(f"✅ Node.js {result.stdout.strip()}")
        except:
            print("⚠️  Node.js non trouvé (optionnel pour backend seul)")
        
        # Vérifier Docker
        try:
            subprocess.run(["docker", "--version"], check=True, capture_output=True)
            print("✅ Docker disponible")
        except:
            print("⚠️  Docker non trouvé (optionnel)")
        
        return True
    
    def install_backend_dependencies(self):
        """Installe les dépendances Python"""
        print("\n📦 Installation des dépendances backend...")
        
        requirements_file = self.backend_dir / "requirements.txt"
        if not requirements_file.exists():
            print("❌ Fichier requirements.txt non trouvé")
            return False
        
        try:
            subprocess.run([
                sys.executable, "-m", "pip", "install", "-r", str(requirements_file)
            ], check=True, cwd=self.backend_dir)
            print("✅ Dépendances backend installées")
            return True
        except subprocess.CalledProcessError:
            print("❌ Erreur lors de l'installation des dépendances")
            return False
    
    def install_frontend_dependencies(self):
        """Installe les dépendances Node.js"""
        print("\n📦 Installation des dépendances frontend...")
        
        package_json = self.project_root / "package.json"
        if not package_json.exists():
            print("⚠️  package.json non trouvé, skip frontend")
            return True
        
        try:
            subprocess.run(["npm", "install"], check=True, cwd=self.project_root)
            print("✅ Dépendances frontend installées")
            return True
        except subprocess.CalledProcessError:
            print("❌ Erreur lors de l'installation npm")
            return False
        except FileNotFoundError:
            print("⚠️  npm non trouvé, skip frontend")
            return True
    
    def setup_environment(self):
        """Configure l'environnement"""
        print("\n⚙️  Configuration de l'environnement...")
        
        env_file = self.project_root / ".env"
        if not env_file.exists():
            env_content = """# TimCash Environment Configuration
SECRET_KEY=dev-secret-key-change-in-production
CINETPAY_API_KEY=your-cinetpay-api-key
CINETPAY_SITE_ID=your-cinetpay-site-id
CINETPAY_SECRET_KEY=your-cinetpay-secret-key
REDIS_URL=redis://localhost:6379/0
ENVIRONMENT=development
"""
            env_file.write_text(env_content)
            print("✅ Fichier .env créé")
        else:
            print("✅ Fichier .env existe")
        
        return True
    
    def start_backend(self):
        """Démarre le serveur backend"""
        print("\n🚀 Démarrage du serveur backend...")
        print("📍 URL: http://localhost:8000")
        print("📚 Documentation: http://localhost:8000/docs")
        print("🛑 Ctrl+C pour arrêter")
        print("-" * 50)
        
        try:
            os.chdir(self.backend_dir)
            subprocess.run([sys.executable, "main.py"])
        except KeyboardInterrupt:
            print("\n👋 Arrêt du serveur backend")
        except Exception as e:
            print(f"❌ Erreur: {e}")
    
    def start_frontend(self):
        """Démarre le serveur frontend"""
        print("\n🚀 Démarrage du serveur frontend...")
        print("📍 URL: http://localhost:3000")
        print("🛑 Ctrl+C pour arrêter")
        print("-" * 50)
        
        try:
            os.chdir(self.project_root)
            subprocess.run(["npm", "run", "dev"])
        except KeyboardInterrupt:
            print("\n👋 Arrêt du serveur frontend")
        except Exception as e:
            print(f"❌ Erreur: {e}")
    
    def start_docker(self):
        """Démarre avec Docker Compose"""
        print("\n🐳 Démarrage avec Docker Compose...")
        print("📍 Frontend: http://localhost:3000")
        print("📍 Backend: http://localhost:8000")
        print("📍 Redis: localhost:6379")
        print("🛑 Ctrl+C pour arrêter")
        print("-" * 50)
        
        try:
            subprocess.run(["docker-compose", "up", "--build"], cwd=self.project_root)
        except KeyboardInterrupt:
            print("\n👋 Arrêt des conteneurs Docker")
            subprocess.run(["docker-compose", "down"], cwd=self.project_root)
        except Exception as e:
            print(f"❌ Erreur Docker: {e}")
    
    def run_tests(self):
        """Exécute les tests"""
        print("\n🧪 Exécution des tests...")
        
        # Tests backend (si pytest disponible)
        try:
            subprocess.run([sys.executable, "-m", "pytest", "-v"], 
                         cwd=self.backend_dir, check=True)
            print("✅ Tests backend réussis")
        except subprocess.CalledProcessError:
            print("❌ Échec des tests backend")
        except FileNotFoundError:
            print("⚠️  pytest non installé")
        
        # Tests frontend (si disponible)
        try:
            subprocess.run(["npm", "test"], cwd=self.project_root, check=True)
            print("✅ Tests frontend réussis")
        except subprocess.CalledProcessError:
            print("❌ Échec des tests frontend")
        except FileNotFoundError:
            print("⚠️  npm non disponible")
    
    def show_menu(self):
        """Affiche le menu principal"""
        print("\n📋 Options disponibles:")
        print("1. 🚀 Démarrer backend seul")
        print("2. 🌐 Démarrer frontend seul")
        print("3. 🐳 Démarrer avec Docker")
        print("4. 🧪 Exécuter les tests")
        print("5. 📚 Ouvrir la documentation")
        print("6. ⚙️  Réinstaller les dépendances")
        print("0. 👋 Quitter")
        print("-" * 50)
        
        choice = input("Votre choix: ").strip()
        return choice
    
    def open_documentation(self):
        """Ouvre la documentation"""
        import webbrowser
        
        print("\n📚 Ouverture de la documentation...")
        
        docs = [
            ("API Documentation", "http://localhost:8000/docs"),
            ("Technical Guide", str(self.project_root / "MISE_A_JOUR_COMPTE_STOCK_ADMIN.md")),
            ("Testing Guide", str(self.project_root / "LOAN_ADMIN_STOCK_TEST.md")),
            ("API Examples", str(self.project_root / "API_USAGE_EXAMPLES.md")),
            ("Roadmap", str(self.project_root / "NEXT_STEPS_ROADMAP.md"))
        ]
        
        for name, url in docs:
            print(f"📖 {name}: {url}")
            if url.startswith("http"):
                try:
                    webbrowser.open(url)
                except:
                    pass
    
    def run(self):
        """Point d'entrée principal"""
        self.print_banner()
        
        if not self.check_dependencies():
            print("\n❌ Dépendances manquantes. Installation requise.")
            return
        
        # Installation automatique au premier lancement
        self.install_backend_dependencies()
        self.install_frontend_dependencies()
        self.setup_environment()
        
        while True:
            choice = self.show_menu()
            
            if choice == "1":
                self.start_backend()
            elif choice == "2":
                self.start_frontend()
            elif choice == "3":
                self.start_docker()
            elif choice == "4":
                self.run_tests()
            elif choice == "5":
                self.open_documentation()
            elif choice == "6":
                self.install_backend_dependencies()
                self.install_frontend_dependencies()
            elif choice == "0":
                print("\n👋 Au revoir!")
                break
            else:
                print("❌ Choix invalide")

if __name__ == "__main__":
    starter = TimCashStarter()
    starter.run()
