export interface User {
  id: string;
  email: string;
  firstName?: string;
  lastName?: string;
  full_name: string; // Nom complet utilisé par le backend
  phone: string;
  role: 'admin' | 'manager' | 'user';
  timType?: 'TIM_MINI' | 'TIM_MAXI' | 'TIM_BUSINESS'; // Type de compte mobile
  isVerified: boolean;
  isBlocked: boolean;
  createdAt: string;
  updatedAt: string;
}

export interface Wallet {
  id: string;
  userId: string;
  balance: number;
  currency: string;
  timType: 'TIM_MINI' | 'TIM_MAXI' | 'TIM_BUSINESS';
  maxBalance?: number; // Limite selon le type TIM
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

export interface Transaction {
  id: string;
  walletId: string;
  type: 'credit' | 'debit' | 'transfer' | 'loan' | 'repayment' | 
        'nfc_deposit' | 'nfc_withdrawal' | 'recharge_cinetpay' | 
        'pret_mobile' | 'remboursement_auto' | 'recharge_by_business' | 
        'virement_bancaire' | 'mobile_transfer';
  amount: number;
  currency: string;
  description: string;
  status: 'pending' | 'completed' | 'failed' | 'cancelled';
  referenceId?: string;
  fromWalletType?: 'TIM_MINI' | 'TIM_MAXI' | 'TIM_BUSINESS';
  toWalletType?: 'TIM_MINI' | 'TIM_MAXI' | 'TIM_BUSINESS';
  nfcEnabled?: boolean; // Pour les transactions NFC
  rechargeMethod?: 'cinetpay' | 'tim_business'; // Méthode de rechargement
  businessUserId?: string; // ID du TIM BUSINESS qui effectue la recharge
  commissionAmount?: number; // Montant de la commission appliquée
  commissionRate?: number; // Taux de commission utilisé
  commissionRateId?: string; // ID du taux de commission appliqué
  createdAt: string;
  updatedAt: string;
}

export interface Loan {
  id: string;
  userId: string;
  userTimType: 'TIM_MAXI' | 'TIM_BUSINESS'; // Seuls MAXI et BUSINESS peuvent emprunter
  amount: number;
  interestRate: number;
  penaltyRate: number; // 15% de pénalité
  duration: number; // in months
  status: 'pending' | 'approved' | 'rejected' | 'active' | 'completed' | 'overdue';
  monthlyPayment: number;
  totalRepayment: number;
  remainingBalance: number;
  nextPaymentDate: string;
  autoRepaymentEnabled: boolean; // Remboursement automatique lors des recharges
  createdAt: string;
  updatedAt: string;
}

export interface DashboardStats {
  totalUsers: number;
  totalWallets: number;
  totalTransactions: number;
  totalLoans: number;
  totalRevenue: number;
  activeLoans: number;
  pendingTransactions: number;
  recentTransactions: Transaction[];
}

export interface AdminStockAccount {
  id: string;
  balance: number;
  totalCredits: number;
  totalDebits: number;
  totalCommissions: number;
  lastUpdated: string;
}

export interface AuthState {
  user: User | null;
  token: string | null;
  isAuthenticated: boolean;
}

// Types spécifiques TIM CASH
export interface TimAccountLimits {
  TIM_MINI: {
    maxBalance: number;
    features: string[];
  };
  TIM_MAXI: {
    maxBalance: number;
    maxLoanAmount: number;
    features: string[];
  };
  TIM_BUSINESS: {
    maxBalance: number;
    maxLoanAmount: number;
    features: string[];
  };
}

export interface TimUserCapabilities {
  canUseNFC: boolean;
  canRequestLoan: boolean;
  canRechargeOthers: boolean;
  canTransferToBank: boolean;
  canReceiveRecharges: boolean;
}

export interface NFCTransaction {
  id: string;
  userId: string;
  type: 'nfc_deposit' | 'nfc_withdrawal';
  amount: number;
  location?: string;
  deviceId?: string;
  merchantId?: string;
  createdAt: string;
}

export interface RechargeTransaction {
  id: string;
  fromUserId: string; // TIM_BUSINESS qui recharge
  toUserId: string;   // TIM_MINI/MAXI qui reçoit
  amount: number;
  commission?: number;
  method: 'cinetpay' | 'tim_business';
  status: 'pending' | 'completed' | 'failed';
  location?: string; // Lieu de la recharge (pour TIM_BUSINESS)
  createdAt: string;
}

// Gestion des commissions
export interface CommissionRate {
  id: string;
  name: string;
  type: 'percentage' | 'fixed';
  value: number; // Pourcentage (0-100) ou montant fixe en FCFA
  category: 'recharge_cinetpay' | 'recharge_business' | 'nfc_transaction' | 
           'loan_interest' | 'loan_penalty' | 'withdrawal' | 'transfer' | 
           'bank_transfer';
  description: string;
  minAmount?: number; // Montant minimum pour appliquer la commission
  maxAmount?: number; // Montant maximum pour la commission
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  updatedBy: string; // ID de l'admin qui a modifié
}

export interface CommissionHistory {
  id: string;
  commissionRateId: string;
  oldValue: number;
  newValue: number;
  changedBy: string;
  reason?: string;
  createdAt: string;
}

export interface CommissionSettings {
  id: string;
  globalCommissionEnabled: boolean;
  defaultCurrency: string;
  commissionRates: CommissionRate[];
  lastUpdated: string;
}