/**
 * Service de gestion des alertes avec SweetAlert2
 * Remplace les notifications push par des alertes élégantes
 */

import Swal from 'sweetalert2';

class SweetAlertService {
  /**
   * Configuration par défaut
   */
  private defaultConfig = {
    confirmButtonColor: '#D97706', // Amber-600
    cancelButtonColor: '#6B7280', // Gray-500
    background: '#FFFFFF',
    color: '#1F2937',
    iconColor: '#D97706',
    customClass: {
      popup: 'rounded-lg shadow-xl',
      title: 'text-xl font-bold',
      confirmButton: 'px-6 py-2 rounded-lg font-medium',
      cancelButton: 'px-6 py-2 rounded-lg font-medium'
    }
  };

  /**
   * Alerte de succès
   */
  success(title: string, message?: string, options?: any) {
    return Swal.fire({
      icon: 'success',
      title,
      text: message,
      ...this.defaultConfig,
      ...options
    });
  }

  /**
   * Alerte d'erreur
   */
  error(title: string, message?: string, options?: any) {
    return Swal.fire({
      icon: 'error',
      title,
      text: message,
      ...this.defaultConfig,
      confirmButtonColor: '#EF4444', // Red-500
      ...options
    });
  }

  /**
   * Alerte d'avertissement
   */
  warning(title: string, message?: string, options?: any) {
    return Swal.fire({
      icon: 'warning',
      title,
      text: message,
      ...this.defaultConfig,
      confirmButtonColor: '#F59E0B', // Amber-500
      ...options
    });
  }

  /**
   * Alerte d'information
   */
  info(title: string, message?: string, options?: any) {
    return Swal.fire({
      icon: 'info',
      title,
      text: message,
      ...this.defaultConfig,
      confirmButtonColor: '#3B82F6', // Blue-500
      ...options
    });
  }

  /**
   * Alerte de confirmation
   */
  confirm(title: string, message?: string, options?: any) {
    return Swal.fire({
      icon: 'question',
      title,
      text: message,
      showCancelButton: true,
      confirmButtonText: 'Oui',
      cancelButtonText: 'Non',
      ...this.defaultConfig,
      ...options
    });
  }

  /**
   * Toast notification (coin de l'écran)
   */
  toast(title: string, icon: 'success' | 'error' | 'warning' | 'info' = 'success', position: any = 'top-end') {
    return Swal.fire({
      toast: true,
      position,
      icon,
      title,
      showConfirmButton: false,
      timer: 3000,
      timerProgressBar: true,
      didOpen: (toast) => {
        toast.addEventListener('mouseenter', Swal.stopTimer);
        toast.addEventListener('mouseleave', Swal.resumeTimer);
      }
    });
  }

  /**
   * Alerte de chargement
   */
  loading(title: string = 'Chargement...', message?: string) {
    return Swal.fire({
      title,
      text: message,
      allowOutsideClick: false,
      allowEscapeKey: false,
      showConfirmButton: false,
      didOpen: () => {
        Swal.showLoading();
      }
    });
  }

  /**
   * Fermer l'alerte en cours
   */
  close() {
    Swal.close();
  }

  // ==================== NOTIFICATIONS SPÉCIFIQUES ====================

  /**
   * Notification d'expiration de session
   */
  sessionExpired() {
    return Swal.fire({
      icon: 'warning',
      title: '🔒 Session Expirée',
      text: 'Votre session a expiré. Veuillez vous reconnecter.',
      confirmButtonText: 'Se reconnecter',
      allowOutsideClick: false,
      allowEscapeKey: false,
      ...this.defaultConfig,
      confirmButtonColor: '#D97706'
    }).then((result) => {
      if (result.isConfirmed) {
        window.location.href = '/login';
      }
    });
  }

  /**
   * Notification de transaction
   */
  transaction(type: string, amount: number, status: 'completed' | 'pending' | 'failed') {
    const icons = {
      completed: 'success' as const,
      pending: 'info' as const,
      failed: 'error' as const
    };

    const titles = {
      completed: '✅ Transaction Réussie',
      pending: '⏳ Transaction En Cours',
      failed: '❌ Transaction Échouée'
    };

    return this.toast(
      `${titles[status]}: ${type} - ${amount.toLocaleString('fr-FR')} FCFA`,
      icons[status],
      'top-end'
    );
  }

  /**
   * Notification de prêt
   */
  loan(loanStatus: 'approved' | 'rejected' | 'due_soon', amount: number, dueDate?: string) {
    if (loanStatus === 'approved') {
      return Swal.fire({
        icon: 'success',
        title: '✅ Prêt Approuvé !',
        html: `
          <p class="text-lg mb-2">Félicitations !</p>
          <p class="text-2xl font-bold text-amber-600">${amount.toLocaleString('fr-FR')} FCFA</p>
          <p class="text-sm text-gray-600 mt-2">Votre prêt a été approuvé et sera crédité dans quelques instants.</p>
        `,
        confirmButtonText: 'Super !',
        ...this.defaultConfig
      });
    } else if (loanStatus === 'rejected') {
      return Swal.fire({
        icon: 'error',
        title: '❌ Prêt Refusé',
        html: `
          <p class="text-lg mb-2">Désolé</p>
          <p class="text-xl font-bold text-red-600">${amount.toLocaleString('fr-FR')} FCFA</p>
          <p class="text-sm text-gray-600 mt-2">Votre demande de prêt a été refusée. Veuillez réessayer plus tard.</p>
        `,
        confirmButtonText: 'Compris',
        ...this.defaultConfig,
        confirmButtonColor: '#EF4444'
      });
    } else {
      return Swal.fire({
        icon: 'warning',
        title: '⏰ Échéance de Prêt',
        html: `
          <p class="text-lg mb-2">Rappel Important</p>
          <p class="text-xl font-bold text-amber-600">${amount.toLocaleString('fr-FR')} FCFA</p>
          <p class="text-sm text-gray-600 mt-2">Votre prêt arrive à échéance le <strong>${dueDate}</strong></p>
        `,
        confirmButtonText: 'Rembourser',
        showCancelButton: true,
        cancelButtonText: 'Plus tard',
        ...this.defaultConfig
      });
    }
  }

  /**
   * Notification de paiement NFC
   */
  nfcPayment(merchantName: string, amount: number, location?: string) {
    const locationText = location ? `<p class="text-sm text-gray-600">📍 ${location}</p>` : '';
    
    return Swal.fire({
      icon: 'success',
      title: '💳 Paiement NFC Effectué',
      html: `
        <p class="text-2xl font-bold text-amber-600 mb-2">${amount.toLocaleString('fr-FR')} FCFA</p>
        <p class="text-lg mb-1">Chez ${merchantName}</p>
        ${locationText}
      `,
      timer: 4000,
      timerProgressBar: true,
      showConfirmButton: false,
      ...this.defaultConfig
    });
  }

  /**
   * Notification de solde faible
   */
  lowBalance(balance: number, threshold: number = 1000) {
    return Swal.fire({
      icon: 'warning',
      title: '⚠️ Solde Faible',
      html: `
        <p class="text-lg mb-2">Votre solde est faible</p>
        <p class="text-2xl font-bold text-red-600">${balance.toLocaleString('fr-FR')} FCFA</p>
        <p class="text-sm text-gray-600 mt-2">Pensez à recharger votre compte pour continuer à utiliser TIM CASH.</p>
      `,
      confirmButtonText: 'Recharger',
      showCancelButton: true,
      cancelButtonText: 'Plus tard',
      ...this.defaultConfig,
      confirmButtonColor: '#F59E0B'
    }).then((result) => {
      if (result.isConfirmed) {
        window.location.href = '/wallet/reload';
      }
    });
  }

  /**
   * Notification de vérification de compte
   */
  accountVerification(status: 'verified' | 'rejected') {
    if (status === 'verified') {
      return Swal.fire({
        icon: 'success',
        title: '✅ Compte Vérifié !',
        html: `
          <p class="text-lg mb-2">Félicitations !</p>
          <p class="text-sm text-gray-600">Votre compte TIM CASH a été vérifié avec succès.</p>
          <p class="text-sm text-gray-600 mt-2">Vous pouvez maintenant profiter de toutes les fonctionnalités.</p>
        `,
        confirmButtonText: 'Génial !',
        ...this.defaultConfig
      });
    } else {
      return Swal.fire({
        icon: 'error',
        title: '❌ Vérification Échouée',
        html: `
          <p class="text-lg mb-2">Désolé</p>
          <p class="text-sm text-gray-600">La vérification de votre compte a échoué.</p>
          <p class="text-sm text-gray-600 mt-2">Veuillez vérifier vos documents et réessayer.</p>
        `,
        confirmButtonText: 'Réessayer',
        ...this.defaultConfig,
        confirmButtonColor: '#EF4444'
      });
    }
  }

  /**
   * Notification de connexion réussie
   */
  loginSuccess(userName: string) {
    return this.toast(
      `👋 Bienvenue ${userName} !`,
      'success',
      'top-end'
    );
  }

  /**
   * Notification de déconnexion
   */
  logoutSuccess() {
    return this.toast(
      '👋 À bientôt !',
      'info',
      'top-end'
    );
  }

  /**
   * Notification d'erreur réseau
   */
  networkError() {
    return Swal.fire({
      icon: 'error',
      title: '🌐 Erreur de Connexion',
      text: 'Impossible de se connecter au serveur. Vérifiez votre connexion internet.',
      confirmButtonText: 'Réessayer',
      ...this.defaultConfig,
      confirmButtonColor: '#EF4444'
    });
  }

  /**
   * Notification de mise à jour disponible
   */
  updateAvailable() {
    return Swal.fire({
      icon: 'info',
      title: '🔄 Mise à Jour Disponible',
      text: 'Une nouvelle version de TIM CASH est disponible.',
      confirmButtonText: 'Mettre à jour',
      showCancelButton: true,
      cancelButtonText: 'Plus tard',
      ...this.defaultConfig,
      confirmButtonColor: '#3B82F6'
    });
  }

  /**
   * Alerte personnalisée avec HTML
   */
  custom(config: any) {
    return Swal.fire({
      ...this.defaultConfig,
      ...config
    });
  }
}

// Instance globale
export const sweetAlert = new SweetAlertService();
