/**
 * Service de gestion des notifications push
 * Remplace les notifications web par des push notifications natives
 */

import { apiService } from './apiService';

class NotificationService {
  private fcmToken: string | null = null;

  /**
   * Initialiser le service de notifications
   */
  async initialize(): Promise<void> {
    try {
      // Vérifier si les notifications sont supportées
      if (!('Notification' in window)) {
        console.warn('⚠️ Les notifications ne sont pas supportées par ce navigateur');
        return;
      }

      // Demander la permission
      const permission = await Notification.requestPermission();
      
      if (permission === 'granted') {
        console.log('✅ Permission de notification accordée');
        
        // Charger Firebase dynamiquement
        await this.setupFirebase();
      } else {
        console.log('❌ Permission de notification refusée');
      }
    } catch (error) {
      console.error('❌ Erreur initialisation notifications:', error);
    }
  }

  /**
   * Configurer Firebase (à implémenter après configuration Firebase)
   */
  private async setupFirebase(): Promise<void> {
    // TODO: Implémenter après configuration Firebase
    console.log('🔧 Configuration Firebase à faire');
    
    // Exemple de ce qui sera fait :
    // const { requestNotificationPermission } = await import('./firebaseService');
    // this.fcmToken = await requestNotificationPermission();
    // 
    // if (this.fcmToken) {
    //   await this.registerDevice(this.fcmToken);
    // }
  }

  /**
   * Enregistrer l'appareil dans le backend
   */
  async registerDevice(token: string): Promise<void> {
    try {
      await apiService.post('/notifications/register-device', {
        device_token: token,
        device_type: 'web',
        device_name: this.getDeviceName()
      });
      
      console.log('✅ Appareil enregistré pour les push notifications');
      this.fcmToken = token;
    } catch (error) {
      console.error('❌ Erreur enregistrement appareil:', error);
    }
  }

  /**
   * Désenregistrer l'appareil
   */
  async unregisterDevice(): Promise<void> {
    if (!this.fcmToken) return;

    try {
      await apiService.delete(`/notifications/unregister-device/${this.fcmToken}`);
      console.log('✅ Appareil désenregistré');
      this.fcmToken = null;
    } catch (error) {
      console.error('❌ Erreur désenregistrement appareil:', error);
    }
  }

  /**
   * Envoyer une notification de test
   */
  async sendTestNotification(): Promise<void> {
    try {
      const result = await apiService.post('/notifications/send-test');
      console.log('✅ Notification de test envoyée:', result);
    } catch (error) {
      console.error('❌ Erreur envoi notification test:', error);
    }
  }

  /**
   * Afficher une notification locale (fallback si Firebase n'est pas configuré)
   */
  showLocalNotification(title: string, body: string, data?: any): void {
    if (Notification.permission === 'granted') {
      const notification = new Notification(title, {
        body,
        icon: '/logo192.png',
        badge: '/badge.png',
        tag: data?.type || 'default',
        requireInteraction: data?.priority === 'high'
      });

      notification.onclick = () => {
        window.focus();
        notification.close();
        
        // Gérer les actions selon le type
        if (data?.action === 'redirect_to_login') {
          window.location.href = '/login';
        }
      };
    }
  }

  /**
   * Notification d'expiration de session
   */
  notifySessionExpired(): void {
    this.showLocalNotification(
      '🔒 Session Expirée',
      'Votre session a expiré. Veuillez vous reconnecter.',
      {
        type: 'session_expired',
        action: 'redirect_to_login',
        priority: 'high'
      }
    );
  }

  /**
   * Notification de transaction
   */
  notifyTransaction(type: string, amount: number, status: string): void {
    const emoji = status === 'completed' ? '✅' : status === 'pending' ? '⏳' : '❌';
    
    this.showLocalNotification(
      `${emoji} Transaction ${status}`,
      `${type}: ${amount.toLocaleString('fr-FR')} FCFA`,
      {
        type: 'transaction',
        transaction_type: type,
        amount,
        status
      }
    );
  }

  /**
   * Notification de solde faible
   */
  notifyLowBalance(balance: number): void {
    this.showLocalNotification(
      '⚠️ Solde Faible',
      `Votre solde est de ${balance.toLocaleString('fr-FR')} FCFA. Pensez à recharger.`,
      {
        type: 'low_balance',
        balance,
        priority: 'high'
      }
    );
  }

  /**
   * Obtenir le nom de l'appareil
   */
  private getDeviceName(): string {
    const ua = navigator.userAgent;
    let browser = 'Unknown';
    let os = 'Unknown';

    // Détecter le navigateur
    if (ua.indexOf('Chrome') > -1) browser = 'Chrome';
    else if (ua.indexOf('Safari') > -1) browser = 'Safari';
    else if (ua.indexOf('Firefox') > -1) browser = 'Firefox';
    else if (ua.indexOf('Edge') > -1) browser = 'Edge';

    // Détecter l'OS
    if (ua.indexOf('Windows') > -1) os = 'Windows';
    else if (ua.indexOf('Mac') > -1) os = 'macOS';
    else if (ua.indexOf('Linux') > -1) os = 'Linux';
    else if (ua.indexOf('Android') > -1) os = 'Android';
    else if (ua.indexOf('iOS') > -1) os = 'iOS';

    return `${browser} - ${os}`;
  }
}

// Instance globale
export const notificationService = new NotificationService();
