/**
 * Service API pour TIM CASH
 * Communication avec le backend SQLAlchemy
 */
import axios, { AxiosInstance, AxiosResponse } from 'axios';
import { API_CONFIG, API_ENDPOINTS, ERROR_MESSAGES } from '../config/api';

// Types
export interface LoginRequest {
  email: string;
  password: string;
}

export interface LoginResponse {
  access_token: string;
  refresh_token: string;
  token_type: string;
}

export interface RegisterRequest {
  username: string;
  email: string;
  password: string;
  full_name: string;
  phone?: string;
  tim_account_type?: string;
}

export interface User {
  id: string;
  username: string;
  email: string;
  full_name: string;
  phone?: string;
  is_admin: boolean;
  tim_account_type: string;
  wallet: {
    id: string;
    balance: number;
    max_balance?: number;
    is_active: boolean;
    nfc_enabled: boolean;
  };
  created_at: string;
}

export interface WalletBalance {
  balance: number;
  max_balance?: number;
  currency: string;
  account_type: string;
  usage_percentage: number;
}

export interface Transaction {
  id: string;
  type: string;
  amount: number;
  commission_amount: number;
  status: string;
  description: string;
  reference: string;
  merchant_name?: string;
  location?: string;
  created_at: string;
  completed_at?: string;
}

export interface Loan {
  id: string;
  amount: number;
  interest_rate: number;
  penalty_rate: number;
  remaining_balance: number;
  status: string;
  due_date: string;
  days_overdue: number;
  created_at: string;
}

export interface AdminStock {
  balance: number;
  total_credits: number;
  total_debits: number;
  total_commissions: number;
  last_updated: string;
  currency: string;
}

export interface DashboardStats {
  total_users: number;
  total_transactions: number;
  total_loans: number;
  active_loans: number;
  total_volume: number;
  admin_stock_balance: number;
  total_commissions: number;
  nfc_transactions: number;
}

class ApiService {
  private api: AxiosInstance;
  private accessToken: string | null = null;
  private refreshToken: string | null = null;

  constructor() {
    this.api = axios.create({
      baseURL: API_CONFIG.BASE_URL,
      timeout: API_CONFIG.TIMEOUT,
      headers: API_CONFIG.HEADERS,
    });

    // Intercepteur pour ajouter le token d'authentification
    this.api.interceptors.request.use(
      (config) => {
        if (this.accessToken) {
          config.headers.Authorization = `Bearer ${this.accessToken}`;
        }
        return config;
      },
      (error) => Promise.reject(error)
    );

    // Intercepteur pour gérer les erreurs et le refresh token
    this.api.interceptors.response.use(
      (response) => response,
      async (error) => {
        if (error.response?.status === 401 && this.refreshToken) {
          try {
            await this.refreshAccessToken();
            // Retry la requête originale
            return this.api.request(error.config);
          } catch (refreshError) {
            this.logout();
            throw refreshError;
          }
        }
        return Promise.reject(this.handleError(error));
      }
    );

    // Charger les tokens depuis le localStorage
    this.loadTokensFromStorage();
  }

  private loadTokensFromStorage(): void {
    this.accessToken = localStorage.getItem('tim_access_token');
    this.refreshToken = localStorage.getItem('tim_refresh_token');
  }

  private saveTokensToStorage(accessToken: string, refreshToken: string): void {
    this.accessToken = accessToken;
    this.refreshToken = refreshToken;
    localStorage.setItem('tim_access_token', accessToken);
    localStorage.setItem('tim_refresh_token', refreshToken);
  }

  private clearTokensFromStorage(): void {
    this.accessToken = null;
    this.refreshToken = null;
    localStorage.removeItem('tim_access_token');
    localStorage.removeItem('tim_refresh_token');
    
    // Afficher une alerte SweetAlert d'expiration de session
    import('./sweetAlertService').then(({ sweetAlert }) => {
      sweetAlert.sessionExpired();
    });
  }

  private handleError(error: any): Error {
    if (error.response) {
      const status = error.response.status;
      const message = error.response.data?.detail || error.response.data?.message;
      
      switch (status) {
        case 401:
          return new Error(message || ERROR_MESSAGES.UNAUTHORIZED);
        case 403:
          return new Error(message || ERROR_MESSAGES.FORBIDDEN);
        case 404:
          return new Error(message || ERROR_MESSAGES.NOT_FOUND);
        case 422:
          // Pour les erreurs de validation, essayer d'extraire le détail
          const detail = error.response?.data?.detail || message || ERROR_MESSAGES.VALIDATION_ERROR;
          return new Error(detail);
        case 500:
          return new Error(message || ERROR_MESSAGES.SERVER_ERROR);
        default:
          return new Error(message || `Erreur HTTP ${status}`);
      }
    } else if (error.request) {
      // Gestion spéciale pour les erreurs de CSP
      if (error.message?.includes('Content Security Policy')) {
        return new Error('Erreur de sécurité: Connexion bloquée par la politique de sécurité. Veuillez contacter l\'administrateur.');
      }
      return new Error(ERROR_MESSAGES.NETWORK_ERROR);
    } else {
      return new Error(error.message || 'Erreur inconnue');
    }
  }

  // Authentification
  async login(credentials: LoginRequest): Promise<LoginResponse> {
    const response: AxiosResponse<LoginResponse> = await this.api.post(
      API_ENDPOINTS.AUTH.LOGIN,
      credentials
    );
    
    this.saveTokensToStorage(response.data.access_token, response.data.refresh_token);
    return response.data;
  }

  async register(userData: RegisterRequest): Promise<LoginResponse> {
    const response: AxiosResponse<LoginResponse> = await this.api.post(
      API_ENDPOINTS.AUTH.REGISTER,
      userData
    );
    
    this.saveTokensToStorage(response.data.access_token, response.data.refresh_token);
    return response.data;
  }

  async getCurrentUser(): Promise<User> {
    const response: AxiosResponse<User> = await this.api.get(API_ENDPOINTS.AUTH.ME);
    return response.data;
  }

  async refreshAccessToken(): Promise<void> {
    if (!this.refreshToken) {
      throw new Error('No refresh token available');
    }

    const response: AxiosResponse<LoginResponse> = await this.api.post(
      API_ENDPOINTS.AUTH.REFRESH,
      { refresh_token: this.refreshToken }
    );
    
    this.saveTokensToStorage(response.data.access_token, response.data.refresh_token);
  }

  async logout(): Promise<void> {
    try {
      if (this.refreshToken) {
        await this.api.post(API_ENDPOINTS.AUTH.LOGOUT, {
          refresh_token: this.refreshToken,
        });
      }
    } catch (error) {
      console.error('Logout error:', error);
    } finally {
      this.clearTokensFromStorage();
    }
  }

  // Portefeuille
  async getWalletBalance(): Promise<WalletBalance> {
    const response: AxiosResponse<WalletBalance> = await this.api.get(
      API_ENDPOINTS.WALLET.BALANCE
    );
    return response.data;
  }

  async reloadWallet(amount: number, paymentMethod: string = 'cinetpay'): Promise<any> {
    const response = await this.api.post(API_ENDPOINTS.WALLET.RELOAD, {
      amount,
      payment_method: paymentMethod,
    });
    return response.data;
  }

  async transferFunds(recipientEmail: string, amount: number, description?: string): Promise<any> {
    const response = await this.api.post(API_ENDPOINTS.WALLET.TRANSFER, {
      recipient_email: recipientEmail,
      amount,
      description,
    });
    return response.data;
  }

  // Transactions
  async getTransactions(limit: number = 50, offset: number = 0): Promise<{
    transactions: Transaction[];
    total_count: number;
    limit: number;
    offset: number;
  }> {
    const response = await this.api.get(
      `${API_ENDPOINTS.TRANSACTIONS.LIST}?limit=${limit}&offset=${offset}`
    );
    return response.data;
  }

  // Prêts
  async createLoan(amount: number, purpose?: string): Promise<any> {
    const response = await this.api.post(API_ENDPOINTS.LOANS.CREATE, {
      amount,
      purpose,
    });
    return response.data;
  }

  async getActiveLoans(): Promise<{
    loans: Loan[];
    total_count: number;
  }> {
    const response = await this.api.get(API_ENDPOINTS.LOANS.ACTIVE);
    return response.data;
  }

  // Administration
  async getAdminStock(): Promise<AdminStock> {
    const response: AxiosResponse<AdminStock> = await this.api.get(
      API_ENDPOINTS.ADMIN.STOCK
    );
    return response.data;
  }

  async getAllUsers(): Promise<{
    users: any[];
    total_count: number;
  }> {
    const response = await this.api.get(API_ENDPOINTS.ADMIN.USERS);
    return response.data;
  }

  async getUserDetails(userId: string): Promise<any> {
    const response = await this.api.get(`${API_ENDPOINTS.ADMIN.USERS}/${userId}`);
    return response.data;
  }

  async updateUserStatus(userId: string, isActive: boolean): Promise<any> {
    const response = await this.api.put(`${API_ENDPOINTS.ADMIN.USERS}/${userId}/status`, null, {
      params: { is_active: isActive }
    });
    return response.data;
  }

  async updateUserRole(userId: string, isAdmin: boolean): Promise<any> {
    const response = await this.api.put(`${API_ENDPOINTS.ADMIN.USERS}/${userId}/role`, null, {
      params: { is_admin: isAdmin }
    });
    return response.data;
  }

  async deleteUser(userId: string): Promise<any> {
    const response = await this.api.delete(`${API_ENDPOINTS.ADMIN.USERS}/${userId}`);
    return response.data;
  }

  async getDashboardStats(): Promise<DashboardStats> {
    const response: AxiosResponse<DashboardStats> = await this.api.get(
      API_ENDPOINTS.ADMIN.DASHBOARD_STATS
    );
    return response.data;
  }

  async getCommissionRates(): Promise<any> {
    const response = await this.api.get(API_ENDPOINTS.ADMIN.COMMISSION_RATES);
    return response.data;
  }

  async getDeviceAnalytics(): Promise<any> {
    const response = await this.api.get(API_ENDPOINTS.ADMIN.DEVICE_ANALYTICS);
    return response.data;
  }

  async getAllTransactionsAdmin(
    limit: number = 100,
    offset: number = 0,
    statusFilter?: string,
    transactionType?: string
  ): Promise<{
    transactions: any[];
    total_count: number;
  }> {
    let url = `${API_ENDPOINTS.ADMIN.TRANSACTIONS}?limit=${limit}&offset=${offset}`;

    if (statusFilter) {
      url += `&status=${statusFilter}`;
    }

    if (transactionType) {
      url += `&transaction_type=${transactionType}`;
    }

    const response = await this.api.get(url);
    return {
      transactions: response.data.transactions || [],
      total_count: response.data.total_count || 0
    };
  }

  async getTransactionsStatsAdmin(): Promise<any> {
    const response = await this.api.get(API_ENDPOINTS.ADMIN.TRANSACTIONS_STATS);
    return response.data;
  }

  async activateCommissionsMobile(): Promise<any> {
    const response = await this.api.post(API_ENDPOINTS.ADMIN.COMMISSION_ACTIVATE_MOBILE);
    return response.data;
  }

  async getMobileCommissionStatus(): Promise<any> {
    const response = await this.api.get(API_ENDPOINTS.ADMIN.COMMISSION_MOBILE_STATUS);
    return response.data;
  }

  async activateAllCommissions(): Promise<any> {
    const response = await this.api.post(API_ENDPOINTS.ADMIN.COMMISSION_ACTIVATE_ALL);
    return response.data;
  }

  async getRechargeCommissionBreakdown(amount: number): Promise<any> {
    const response = await this.api.get(`${API_ENDPOINTS.ADMIN.COMMISSION_RECHARGE_BREAKDOWN}?amount=${amount}`);
    return response.data;
  }

  // Géographie
  async getCountries(): Promise<any> {
    const response = await this.api.get(API_ENDPOINTS.GEOGRAPHY.COUNTRIES);
    return response.data;
  }

  // Utilitaires
  isAuthenticated(): boolean {
    return !!this.accessToken;
  }

  getAccessToken(): string | null {
    return this.accessToken;
  }

  // Méthodes HTTP génériques
  async get(endpoint: string): Promise<any> {
    const response = await this.api.get(endpoint);
    return response.data;
  }

  async post(endpoint: string, data?: any): Promise<any> {
    const response = await this.api.post(endpoint, data);
    return response.data;
  }

  async put(endpoint: string, data?: any): Promise<any> {
    const response = await this.api.put(endpoint, data);
    return response.data;
  }

  async delete(endpoint: string): Promise<any> {
    const response = await this.api.delete(endpoint);
    return response.data;
  }

  async patch(endpoint: string, data?: any): Promise<any> {
    const response = await this.api.patch(endpoint, data);
    return response.data;
  }
}

// Instance singleton
export const apiService = new ApiService();
export default apiService;
