// Service API centralisé pour TIM CASH
const API_BASE_URL = import.meta.env.VITE_API_URL || "http://localhost:8001";

// Fonction helper pour les requêtes avec authentification
const fetchWithAuth = async (endpoint: string, options: RequestInit = {}) => {
  // Utiliser le même nom de clé que AuthContext.tsx
  const token = localStorage.getItem("tim_access_token");

  const headers = {
    "Content-Type": "application/json",
    ...(token && { Authorization: `Bearer ${token}` }),
    ...options.headers,
  };

  const response = await fetch(`${API_BASE_URL}${endpoint}`, {
    ...options,
    headers,
  });

  if (!response.ok) {
    const error = await response.text();
    throw new Error(error || `HTTP error! status: ${response.status}`);
  }

  return response.json();
};

// === DASHBOARD ===
export const getDashboardStats = async () => {
  return fetchWithAuth("/admin/dashboard/stats");
};

// === USERS ===
export const getUsers = async () => {
  return fetchWithAuth("/admin/users");
};

export const getUserById = async (userId: string) => {
  return fetchWithAuth(`/admin/users/${userId}`);
};

export const updateUser = async (userId: string, data: any) => {
  return fetchWithAuth(`/admin/users/${userId}`, {
    method: "PUT",
    body: JSON.stringify(data),
  });
};

// === WALLETS ===
export const getWallets = async () => {
  return fetchWithAuth("/admin/wallets");
};

export const getWalletById = async (walletId: string) => {
  return fetchWithAuth(`/admin/wallets/${walletId}`);
};

export const getWalletTransactions = async (walletId: string) => {
  return fetchWithAuth(`/admin/wallets/${walletId}/transactions`);
};

// === TRANSACTIONS ===
export const getTransactions = async (params?: {
  type?: string;
  status?: string;
  startDate?: string;
  endDate?: string;
  limit?: number;
}) => {
  const queryParams = new URLSearchParams();
  if (params) {
    Object.entries(params).forEach(([key, value]) => {
      if (value) queryParams.append(key, value.toString());
    });
  }

  const queryString = queryParams.toString();
  return fetchWithAuth(
    `/admin/transactions${queryString ? `?${queryString}` : ""}`
  );
};

export const getTransactionById = async (transactionId: string) => {
  return fetchWithAuth(`/admin/transactions/${transactionId}`);
};

// === RECHARGES ===
export const getRecharges = async (params?: {
  status?: string;
  startDate?: string;
  endDate?: string;
}) => {
  const queryParams = new URLSearchParams();
  if (params) {
    Object.entries(params).forEach(([key, value]) => {
      if (value) queryParams.append(key, value.toString());
    });
  }

  const queryString = queryParams.toString();
  return fetchWithAuth(
    `/admin/recharges${queryString ? `?${queryString}` : ""}`
  );
};

export const getRechargeById = async (rechargeId: string) => {
  return fetchWithAuth(`/admin/recharges/${rechargeId}`);
};

// === LOANS (PRÊTS) ===
export const getLoans = async (params?: { status?: string; type?: string }) => {
  const queryParams = new URLSearchParams();
  if (params) {
    Object.entries(params).forEach(([key, value]) => {
      if (value) queryParams.append(key, value.toString());
    });
  }

  const queryString = queryParams.toString();
  return fetchWithAuth(`/admin/loans${queryString ? `?${queryString}` : ""}`);
};

export const getLoanById = async (loanId: string) => {
  return fetchWithAuth(`/admin/loans/${loanId}`);
};

export const approveLoan = async (loanId: string, data: any) => {
  return fetchWithAuth(`/admin/loans/${loanId}/approve`, {
    method: "POST",
    body: JSON.stringify(data),
  });
};

export const rejectLoan = async (loanId: string, reason: string) => {
  return fetchWithAuth(`/admin/loans/${loanId}/reject`, {
    method: "POST",
    body: JSON.stringify({ reason }),
  });
};

// === LOAN REPAYMENTS ===
export const getLoanRepayments = async () => {
  return fetchWithAuth("/api/loans/all");
};

export const getOverdueLoans = async () => {
  return fetchWithAuth("/api/loans/all");
};

// === ADMIN STOCK ===
export const getAdminStock = async () => {
  return fetchWithAuth("/admin/stock");
};

export const updateAdminStock = async (data: any) => {
  return fetchWithAuth("/admin/stock", {
    method: "PUT",
    body: JSON.stringify(data),
  });
};

// === COMMISSIONS ===
export const getCommissions = async () => {
  return fetchWithAuth("/admin/commissions");
};

export const updateCommission = async (commissionId: string, data: any) => {
  return fetchWithAuth(`/admin/commissions/${commissionId}`, {
    method: "PUT",
    body: JSON.stringify(data),
  });
};

// === NOTIFICATIONS ===
export const getNotifications = async () => {
  return fetchWithAuth("/admin/notifications");
};

export const markNotificationAsRead = async (notificationId: string) => {
  return fetchWithAuth(`/admin/notifications/${notificationId}/read`, {
    method: "POST",
  });
};

// === ANALYTICS ===
export const getAnalytics = async (params?: {
  startDate?: string;
  endDate?: string;
  type?: string;
}) => {
  const queryParams = new URLSearchParams();
  if (params) {
    Object.entries(params).forEach(([key, value]) => {
      if (value) queryParams.append(key, value.toString());
    });
  }

  const queryString = queryParams.toString();
  return fetchWithAuth(
    `/admin/analytics${queryString ? `?${queryString}` : ""}`
  );
};

export const getNFCAnalytics = async () => {
  return fetchWithAuth("/admin/analytics/nfc");
};

// === REPORTS ===
export const generateReport = async (data: {
  type: string;
  startDate: string;
  endDate: string;
  format: string;
}) => {
  return fetchWithAuth("/admin/reports/generate", {
    method: "POST",
    body: JSON.stringify(data),
  });
};

export const getReports = async () => {
  return fetchWithAuth("/admin/reports");
};

export const getCinetPayReports = async () => {
  return fetchWithAuth("/admin/reports/cinetpay");
};

// === WORKFLOWS ===
export const getWorkflows = async () => {
  return fetchWithAuth("/admin/workflows");
};

// === TIM BUSINESS RECHARGES ===
export const getTIMBusinessRecharges = async () => {
  return fetchWithAuth("/admin/tim-business-recharges");
};

// === MONITORING ===
export const getSystemMonitoring = async () => {
  return fetchWithAuth("/admin/monitoring");
};

export default {
  getDashboardStats,
  getUsers,
  getUserById,
  updateUser,
  getWallets,
  getWalletById,
  getWalletTransactions,
  getTransactions,
  getTransactionById,
  getRecharges,
  getRechargeById,
  getLoans,
  getLoanById,
  approveLoan,
  rejectLoan,
  getLoanRepayments,
  getOverdueLoans,
  getAdminStock,
  updateAdminStock,
  getCommissions,
  updateCommission,
  getNotifications,
  markNotificationAsRead,
  getAnalytics,
  getNFCAnalytics,
  generateReport,
  getReports,
  getCinetPayReports,
  getWorkflows,
  getTIMBusinessRecharges,
  getSystemMonitoring,
};
