import React, { useState, useEffect } from 'react';
import {
  GitBranch,
  Play,
  Pause,
  CheckCircle,
  Clock,
  AlertTriangle,
  Users,
  Zap,
  Activity,
  Building2,
  CreditCard,
  ArrowRight,
  Filter,
  Search,
  Calendar,
  MoreVertical,
  Eye,
  Edit,
  Trash2,
  Plus
} from 'lucide-react';
import { getWorkflows } from '../../services/api';

interface WorkflowStep {
  name: string;
  status: string;
}

interface WorkflowData {
  id: string;
  name: string;
  description: string;
  status: string;
  category: string;
  triggers: number;
  executions: number;
  successRate: number;
  lastRun: string | null;
  steps: WorkflowStep[];
}

const Workflows: React.FC = () => {
  const [activeFilter, setActiveFilter] = useState('all');
  const [searchQuery, setSearchQuery] = useState('');
  const [workflows, setWorkflows] = useState<WorkflowData[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Récupérer les workflows depuis l'API
  useEffect(() => {
    const fetchWorkflows = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await getWorkflows();

        // Transformer les données de l'API au format attendu par le frontend
        const workflowsData = response.workflows.map((workflow: any) => ({
          id: workflow.id,
          name: workflow.name,
          description: workflow.description,
          status: workflow.status,
          category: workflow.category,
          triggers: workflow.triggers,
          executions: workflow.executions,
          successRate: workflow.success_rate,
          lastRun: workflow.last_run,
          steps: workflow.steps || []
        }));

        setWorkflows(workflowsData);
      } catch (err: any) {
        console.error('Erreur lors de la récupération des workflows:', err);
        setError(err.message || 'Erreur lors de la récupération des workflows');

        // En cas d'erreur, utiliser les données mockées pour la démo
        setWorkflows([
    {
      id: 'wf-001',
      name: 'Validation Prêt SOS',
      description: 'Processus automatique de validation des demandes de prêts SOS',
      status: 'active',
      category: 'loans',
      triggers: 3,
      executions: 247,
      successRate: 98.4,
      lastRun: '2024-01-20T14:30:00Z',
      steps: [
        { name: 'Vérification KYC', status: 'completed' },
        { name: 'Analyse Crédit', status: 'completed' },
        { name: 'Approbation Auto', status: 'completed' },
        { name: 'Décaissement', status: 'completed' }
      ]
    },
    {
      id: 'wf-002',
      name: 'Recharge CinetPay',
      description: 'Workflow de traitement des recharges via CinetPay',
      status: 'active',
      category: 'payments',
      triggers: 8,
      executions: 1847,
      successRate: 99.2,
      lastRun: '2024-01-20T15:45:00Z',
      steps: [
        { name: 'Validation Paiement', status: 'completed' },
        { name: 'Crédit Compte', status: 'completed' },
        { name: 'Notification SMS', status: 'completed' }
      ]
    },
    {
      id: 'wf-003',
      name: 'Remboursement Automatique',
      description: 'Déduction automatique des prêts lors des recharges',
      status: 'active',
      category: 'loans',
      triggers: 5,
      executions: 156,
      successRate: 100,
      lastRun: '2024-01-20T12:20:00Z',
      steps: [
        { name: 'Détection Prêt Actif', status: 'completed' },
        { name: 'Calcul Montant', status: 'completed' },
        { name: 'Déduction Auto', status: 'completed' },
        { name: 'Mise à jour Solde', status: 'completed' }
      ]
    },
    {
      id: 'wf-004',
      name: 'Alerte Limite Quotidienne',
      description: 'Notification quand un utilisateur approche sa limite',
      status: 'paused',
      category: 'monitoring',
      triggers: 2,
      executions: 89,
      successRate: 95.5,
      lastRun: '2024-01-19T16:30:00Z',
      steps: [
        { name: 'Vérification Solde', status: 'completed' },
        { name: 'Calcul Pourcentage', status: 'completed' },
        { name: 'Envoi Alerte', status: 'pending' }
      ]
    },
    {
      id: 'wf-005',
      name: 'Activation Compte TIM BUSINESS',
      description: 'Processus d\'activation des comptes professionnels',
      status: 'draft',
      category: 'accounts',
      triggers: 0,
      executions: 0,
      successRate: 0,
      lastRun: null,
      steps: [
        { name: 'Vérification Documents', status: 'pending' },
        { name: 'Validation Manuelle', status: 'pending' },
        { name: 'Activation Compte', status: 'pending' }
      ]
    }
        ]);
      } finally {
        setLoading(false);
      }
    };

    fetchWorkflows();
  }, []);

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'active':
        return <Play className="w-4 h-4 text-amber-600" />;
      case 'paused':
        return <Pause className="w-4 h-4 text-yellow-600" />;
      case 'draft':
        return <Clock className="w-4 h-4 text-amber-600" />;
      default:
        return <Clock className="w-4 h-4 text-amber-600" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
        return 'bg-amber-100 text-amber-800';
      case 'paused':
        return 'bg-yellow-100 text-yellow-800';
      case 'draft':
        return 'bg-orange-100 text-orange-800';
      default:
        return 'bg-amber-100 text-amber-800';
    }
  };

  const getCategoryIcon = (category: string) => {
    switch (category) {
      case 'loans':
        return <CreditCard className="w-5 h-5 text-amber-600" />;
      case 'payments':
        return <Activity className="w-5 h-5 text-orange-600" />;
      case 'monitoring':
        return <Eye className="w-5 h-5 text-yellow-600" />;
      case 'accounts':
        return <Building2 className="w-5 h-5 text-amber-700" />;
      default:
        return <GitBranch className="w-5 h-5 text-amber-600" />;
    }
  };

  const getStepStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="w-4 h-4 text-green-600" />;
      case 'pending':
        return <Clock className="w-4 h-4 text-amber-400" />;
      case 'error':
        return <AlertTriangle className="w-4 h-4 text-red-600" />;
      default:
        return <Clock className="w-4 h-4 text-gray-400" />;
    }
  };

  const filteredWorkflows = workflows.filter(workflow => {
    const matchesFilter = activeFilter === 'all' || workflow.status === activeFilter;
    const matchesSearch = workflow.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         workflow.description.toLowerCase().includes(searchQuery.toLowerCase());
    return matchesFilter && matchesSearch;
  });

  const stats = {
    total: workflows.length,
    active: workflows.filter(w => w.status === 'active').length,
    paused: workflows.filter(w => w.status === 'paused').length,
    draft: workflows.filter(w => w.status === 'draft').length,
    totalExecutions: workflows.reduce((sum, w) => sum + w.executions, 0),
    avgSuccessRate: workflows.reduce((sum, w) => sum + w.successRate, 0) / workflows.length
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-blue-100 p-6">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-blue-900 mb-2">Workflows TIM CASH</h1>
          <p className="text-blue-700">Automatisation des processus métier et gestion des flux de travail</p>
        </div>

        {/* Stats */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          {/* Total Workflows */}
          <div className="bg-white p-6 rounded-xl shadow-md border border-blue-100 transform transition-all hover:scale-105">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-blue-600">Total Workflows</p>
                <p className="text-2xl font-bold text-blue-900">{workflows.length}</p>
              </div>
              <div className="p-3 rounded-lg bg-blue-50 text-blue-600">
                <GitBranch className="w-6 h-6" />
              </div>
            </div>
          </div>

          {/* Active Workflows */}
          <div className="bg-white p-6 rounded-xl shadow-md border border-green-100 transform transition-all hover:scale-105">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-green-600">Workflows Actifs</p>
                <p className="text-2xl font-bold text-green-800">
                  {workflows.filter(w => w.status === 'active').length}
                </p>
              </div>
              <div className="p-3 rounded-lg bg-green-50 text-green-600">
                <Play className="w-6 h-6" />
              </div>
            </div>
          </div>

          {/* Total Executions */}
          <div className="bg-white p-6 rounded-xl shadow-md border border-purple-100 transform transition-all hover:scale-105">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-purple-600">Exécutions Totales</p>
                <p className="text-2xl font-bold text-purple-900">
                  {workflows.reduce((sum, w) => sum + w.executions, 0).toLocaleString()}
                </p>
              </div>
              <div className="p-3 rounded-lg bg-purple-50 text-purple-600">
                <Zap className="w-6 h-6" />
              </div>
            </div>
          </div>

          {/* Success Rate */}
          <div className="bg-white p-6 rounded-xl shadow-md border border-yellow-100 transform transition-all hover:scale-105">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-yellow-600">Taux de Réussite</p>
                <p className="text-2xl font-bold text-yellow-800">
                  {(workflows.reduce((sum, w) => sum + w.successRate, 0) / workflows.length).toFixed(1)}%
                </p>
              </div>
              <div className="p-3 rounded-lg bg-yellow-50 text-yellow-600">
                <CheckCircle className="w-6 h-6" />
              </div>
            </div>
          </div>
        </div>

        {/* Filters and Search */}
        <div className="flex flex-col lg:flex-row justify-between items-start lg:items-center gap-4 mb-6">
          <div className="flex flex-wrap gap-2">
            {[
              { id: 'all', label: 'Tous', count: workflows.length, color: 'blue' },
              { id: 'active', label: 'Actifs', count: workflows.filter(w => w.status === 'active').length, color: 'green' },
              { id: 'paused', label: 'En pause', count: workflows.filter(w => w.status === 'paused').length, color: 'yellow' },
              { id: 'draft', label: 'Brouillons', count: workflows.filter(w => w.status === 'draft').length, color: 'gray' },
            ].map((filter) => (
              <button
                key={filter.id}
                onClick={() => setActiveFilter(filter.id)}
                className={`px-4 py-2 text-sm font-medium rounded-lg flex items-center gap-2 transition-colors duration-200 ${
                  activeFilter === filter.id
                    ? `bg-${filter.color}-100 text-${filter.color}-800 border border-${filter.color}-200 shadow-inner`
                    : 'bg-white text-gray-600 border border-gray-200 hover:bg-gray-50 hover:shadow-md'
                }`}
              >
                {filter.label}
                <span className={`bg-${filter.color}-100 text-${filter.color}-800 text-xs font-semibold px-2 py-0.5 rounded-full`}>
                  {filter.count}
                </span>
              </button>
            ))}
          </div>

          <div className="relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-blue-400" />
            <input
              type="text"
              placeholder="Rechercher un workflow..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10 pr-4 py-2 border border-blue-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 w-80"
            />
          </div>
        </div>

        {/* Workflows Grid */}
        <div className="grid gap-6">
          {loading ? (
            <div className="bg-white rounded-xl shadow-md border border-blue-50 p-12 text-center">
              <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
              <p className="mt-4 text-blue-600">Chargement des workflows...</p>
            </div>
          ) : filteredWorkflows.length === 0 ? (
            <div className="bg-white rounded-xl shadow-md border border-blue-50 p-12 text-center">
              <GitBranch className="w-16 h-16 text-blue-300 mx-auto mb-4" />
              <p className="text-blue-600 font-medium">Aucun workflow trouvé</p>
              <p className="text-blue-500 text-sm mt-2">
                {searchQuery || activeFilter !== 'all'
                  ? 'Essayez de modifier vos filtres de recherche'
                  : 'Les workflows créés depuis l\'application mobile apparaîtront ici'}
              </p>
            </div>
          ) : (
            filteredWorkflows.map((workflow) => (
              <div key={workflow.id} className="bg-white rounded-xl shadow-md border border-blue-50 p-6 hover:shadow-lg transition-all duration-200 transform hover:-translate-y-1">
              <div className="flex items-start justify-between">
                <div className="flex items-start space-x-4 flex-1">
                  <div className={`w-12 h-12 rounded-lg flex items-center justify-center ${
                    workflow.status === 'active' ? 'bg-green-50 text-green-600' : 
                    workflow.status === 'paused' ? 'bg-yellow-50 text-yellow-600' :
                    'bg-blue-50 text-blue-600'
                  }`}>
                    {getCategoryIcon(workflow.category)}
                  </div>
                
                  <div className="flex-1">
                    <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between sm:gap-4">
                      <h3 className="text-lg font-bold text-gray-900">{workflow.name}</h3>
                      <div className="flex items-center gap-2 mt-1 sm:mt-0">
                        <span className={`inline-flex items-center px-3 py-1 rounded-full text-xs font-semibold ${
                          workflow.status === 'active' 
                            ? 'bg-green-100 text-green-800 border border-green-200' 
                            : workflow.status === 'paused'
                            ? 'bg-yellow-100 text-yellow-800 border border-yellow-200'
                            : 'bg-gray-100 text-gray-800 border border-gray-200'
                        }`}>
                          {workflow.status === 'active' ? (
                            <>
                              <Play className="w-3 h-3 mr-1" />
                              Actif
                            </>
                          ) : workflow.status === 'paused' ? (
                            <>
                              <Pause className="w-3 h-3 mr-1" />
                              En pause
                            </>
                          ) : (
                            <>
                              <Clock className="w-3 h-3 mr-1" />
                              Brouillon
                            </>
                          )}
                        </span>
                      </div>
                    </div>
                      
                    <p className="text-sm text-gray-600 mt-1">{workflow.description}</p>
                      
                    <div className="mt-3 flex flex-wrap items-center gap-2">

                      {/* Workflow Steps */}
                      <div className="mt-4 flex flex-wrap items-center gap-2">
                        {workflow.steps.map((step, index) => (
                          <React.Fragment key={index}>
                            <div className={`flex items-center space-x-2 px-3 py-1.5 rounded-lg border ${
                              step.status === 'completed' 
                                ? 'bg-green-50 border-green-100' 
                                : step.status === 'error' 
                                ? 'bg-red-50 border-red-100' 
                                : 'bg-yellow-50 border-yellow-100'
                            }`}>
                              {step.status === 'completed' ? (
                                <CheckCircle className="w-4 h-4 text-green-600" />
                              ) : step.status === 'error' ? (
                                <AlertTriangle className="w-4 h-4 text-red-600" />
                              ) : (
                                <Clock className="w-4 h-4 text-yellow-600" />
                              )}
                              <span className={`text-sm font-medium ${
                                step.status === 'completed' 
                                  ? 'text-green-700' 
                                  : step.status === 'error' 
                                  ? 'text-red-700' 
                                  : 'text-yellow-700'
                              }`}>
                                {step.name}
                              </span>
                            </div>
                            {index < workflow.steps.length - 1 && (
                              <ArrowRight className="w-4 h-4 text-gray-300" />
                            )}
                          </React.Fragment>
                        ))}
                      </div>

                      <div className="flex items-center space-x-1">
                        <Calendar className="w-4 h-4 text-gray-400" />
                        <span className="text-sm text-gray-600">
                          Dernière exécution: {workflow.lastRun 
                            ? new Date(workflow.lastRun).toLocaleDateString('fr-FR') + ' à ' + 
                              new Date(workflow.lastRun).toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' })
                            : 'Jamais'
                          }
                        </span>
                      </div>
                    </div>
                  </div>
                </div>

                <div className="flex items-center space-x-1">
                  <button 
                    className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors border border-blue-100 hover:border-blue-200"
                    title="Voir les détails"
                  >
                    <Eye className="w-5 h-5" />
                  </button>
                  <button 
                    className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors border border-blue-100 hover:border-blue-200"
                    title="Modifier"
                  >
                    <Edit className="w-5 h-5" />
                  </button>
                  <button 
                    className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors border border-red-100 hover:border-red-200"
                    title="Supprimer"
                  >
                    <Trash2 className="w-5 h-5" />
                  </button>
                  <button 
                    className="p-2 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors border border-gray-200 hover:border-gray-300"
                    title="Plus d'options"
                  >
                    <MoreVertical className="w-5 h-5" />
                  </button>
                </div>
              </div>
            </div>
            ))
          )}
        </div>
      </div>
    </div>
  );
};

export default Workflows;
