import React, { useState, useEffect } from 'react';
import { Users, Wallet, TrendingUp, TrendingDown, Search, Filter, Eye, EyeOff } from 'lucide-react';
import { getWallets } from '../../services/api';

interface WalletData {
  id: string;
  userId: string;
  userName: string;
  timType: 'TIM_MINI' | 'TIM_MAXI' | 'TIM_BUSINESS';
  balance: number;
  maxBalance: number;
  isActive: boolean;
  lastTransaction?: string;
  createdAt: string;
}

const Wallets: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedType, setSelectedType] = useState<string>('all');
  const [showBalances, setShowBalances] = useState(true);
  const [wallets, setWallets] = useState<WalletData[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Récupérer les portefeuilles depuis l'API
  useEffect(() => {
    const fetchWallets = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await getWallets();

        // Transformer les données de l'API au format attendu par le frontend
        const walletsData = response.wallets.map((wallet: any) => ({
          id: wallet.id,
          userId: wallet.user_id,
          userName: wallet.user_name,
          timType: wallet.tim_type,
          balance: wallet.balance,
          maxBalance: wallet.max_balance || 0,
          isActive: wallet.is_active,
          lastTransaction: wallet.updated_at || wallet.created_at,
          createdAt: wallet.created_at
        }));

        setWallets(walletsData);
      } catch (err: any) {
        console.error('Erreur lors de la récupération des wallets:', err);
        setError(err.message || 'Erreur lors de la récupération des portefeuilles');
      } finally {
        setLoading(false);
      }
    };

    fetchWallets();
  }, []);

  const getTimTypeColor = (type: string) => {
    switch (type) {
      case 'TIM_MINI':
        return 'bg-amber-100 text-amber-800';
      case 'TIM_MAXI':
        return 'bg-orange-100 text-orange-800';
      case 'TIM_BUSINESS':
        return 'bg-yellow-100 text-yellow-800';
      default:
        return 'bg-amber-100 text-amber-800';
    }
  };

  const getTimTypeLabel = (type: string) => {
    switch (type) {
      case 'TIM_MINI':
        return 'TIM MINI';
      case 'TIM_MAXI':
        return 'TIM MAXI';
      case 'TIM_BUSINESS':
        return 'TIM BUSINESS';
      default:
        return type;
    }
  };

  const filteredWallets = wallets.filter(wallet => {
    const matchesSearch = wallet.userName.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesType = selectedType === 'all' || wallet.timType === selectedType;
    return matchesSearch && matchesType;
  });

  const totalBalance = wallets.reduce((sum, wallet) => sum + wallet.balance, 0);
  const activeWallets = wallets.filter(w => w.isActive).length;
  const walletsByType = {
    TIM_MINI: wallets.filter(w => w.timType === 'TIM_MINI').length,
    TIM_MAXI: wallets.filter(w => w.timType === 'TIM_MAXI').length,
    TIM_BUSINESS: wallets.filter(w => w.timType === 'TIM_BUSINESS').length
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-amber-100 rounded-lg">
              <Wallet className="w-6 h-6 text-amber-600" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-amber-900">Portefeuilles Mobiles</h1>
              <p className="text-amber-600">Gestion des comptes TIM MINI, MAXI et BUSINESS</p>
            </div>
          </div>
          <button
            onClick={() => setShowBalances(!showBalances)}
            className="flex items-center space-x-2 bg-amber-100 text-amber-700 px-4 py-2 rounded-lg hover:bg-amber-200 transition-colors"
          >
            {showBalances ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
            <span>{showBalances ? 'Masquer' : 'Afficher'} Soldes</span>
          </button>
        </div>
      </div>

      {/* Statistiques */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-600">Total Portefeuilles</p>
              <p className="text-2xl font-bold text-amber-900">{wallets.length}</p>
            </div>
            <Users className="w-8 h-8 text-amber-600" />
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-600">Portefeuilles Actifs</p>
              <p className="text-2xl font-bold text-amber-700">{activeWallets}</p>
            </div>
            <TrendingUp className="w-8 h-8 text-amber-700" />
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-600">Solde Total</p>
              <p className="text-2xl font-bold text-orange-600">
                {showBalances ? `${totalBalance.toLocaleString()} FCFA` : '••••••••• FCFA'}
              </p>
            </div>
            <Wallet className="w-8 h-8 text-orange-600" />
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-600">TIM BUSINESS</p>
              <p className="text-2xl font-bold text-yellow-600">{walletsByType.TIM_BUSINESS}</p>
            </div>
            <TrendingDown className="w-8 h-8 text-yellow-600" />
          </div>
        </div>
      </div>

      {/* Répartition par type */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-amber-600">TIM MINI</span>
            <span className="text-lg font-bold text-amber-700">{walletsByType.TIM_MINI}</span>
          </div>
          <div className="w-full bg-amber-200 rounded-full h-2">
            <div 
              className="bg-amber-600 h-2 rounded-full" 
              style={{ width: `${(walletsByType.TIM_MINI / wallets.length) * 100}%` }}
            ></div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-amber-600">TIM MAXI</span>
            <span className="text-lg font-bold text-orange-700">{walletsByType.TIM_MAXI}</span>
          </div>
          <div className="w-full bg-orange-200 rounded-full h-2">
            <div 
              className="bg-orange-600 h-2 rounded-full" 
              style={{ width: `${(walletsByType.TIM_MAXI / wallets.length) * 100}%` }}
            ></div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-amber-600">TIM BUSINESS</span>
            <span className="text-lg font-bold text-yellow-700">{walletsByType.TIM_BUSINESS}</span>
          </div>
          <div className="w-full bg-yellow-200 rounded-full h-2">
            <div 
              className="bg-yellow-600 h-2 rounded-full" 
              style={{ width: `${(walletsByType.TIM_BUSINESS / wallets.length) * 100}%` }}
            ></div>
          </div>
        </div>
      </div>

      {/* Filtres */}
      <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
        <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-4 md:space-y-0">
          <div className="flex items-center space-x-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-amber-400 w-4 h-4" />
              <input
                type="text"
                placeholder="Rechercher un utilisateur..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 pr-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
              />
            </div>
            <div className="flex items-center space-x-2">
              <Filter className="w-4 h-4 text-amber-400" />
              <select
                value={selectedType}
                onChange={(e) => setSelectedType(e.target.value)}
                className="px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
              >
                <option value="all">Tous les types</option>
                <option value="TIM_MINI">TIM MINI</option>
                <option value="TIM_MAXI">TIM MAXI</option>
                <option value="TIM_BUSINESS">TIM BUSINESS</option>
              </select>
            </div>
          </div>
        </div>
      </div>

      {/* Liste des portefeuilles */}
      <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30">
        <div className="p-6 border-b border-amber-200/50">
          <h3 className="text-lg font-semibold text-amber-900">
            Portefeuilles ({filteredWallets.length})
          </h3>
        </div>
        <div className="divide-y divide-amber-200/50">
          {loading ? (
            <div className="p-12 text-center">
              <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600"></div>
              <p className="mt-4 text-amber-600">Chargement des portefeuilles...</p>
            </div>
          ) : error ? (
            <div className="p-12 text-center">
              <p className="text-red-600 font-medium">{error}</p>
              <button
                onClick={() => window.location.reload()}
                className="mt-4 px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700"
              >
                Réessayer
              </button>
            </div>
          ) : filteredWallets.length === 0 ? (
            <div className="p-12 text-center">
              <Wallet className="w-16 h-16 text-amber-300 mx-auto mb-4" />
              <p className="text-amber-600 font-medium">Aucun portefeuille trouvé</p>
              <p className="text-amber-500 text-sm mt-2">
                {searchTerm || selectedType !== 'all'
                  ? 'Essayez de modifier vos filtres de recherche'
                  : 'Les portefeuilles créés depuis l\'application mobile apparaîtront ici'}
              </p>
            </div>
          ) : (
            filteredWallets.map((wallet) => (
            <div key={wallet.id} className="p-6 hover:bg-amber-50/50 transition-colors">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <div className="w-12 h-12 bg-gradient-to-r from-amber-500 to-orange-600 rounded-full flex items-center justify-center">
                    <span className="text-white font-bold text-lg">
                      {wallet.userName.charAt(0).toUpperCase()}
                    </span>
                  </div>
                  <div>
                    <div className="flex items-center space-x-2 mb-1">
                      <h4 className="font-medium text-amber-900">{wallet.userName}</h4>
                      <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getTimTypeColor(wallet.timType)}`}>
                        {getTimTypeLabel(wallet.timType)}
                      </span>
                      <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${
                        wallet.isActive ? 'bg-amber-100 text-amber-800' : 'bg-red-100 text-red-800'
                      }`}>
                        {wallet.isActive ? 'Actif' : 'Inactif'}
                      </span>
                    </div>
                    <div className="flex items-center space-x-4 text-sm text-amber-500">
                      <span>ID: {wallet.userId.substring(0, 8)}...</span>
                      <span>Créé: {new Date(wallet.createdAt).toLocaleDateString('fr-FR')}</span>
                      {wallet.lastTransaction && (
                        <span>Dernière activité: {new Date(wallet.lastTransaction).toLocaleDateString('fr-FR')}</span>
                      )}
                    </div>
                  </div>
                </div>
                <div className="text-right">
                  <p className="text-lg font-bold text-amber-900">
                    {showBalances ? `${wallet.balance.toLocaleString()} FCFA` : '••••••••• FCFA'}
                  </p>
                  <p className="text-sm text-amber-600">
                    {wallet.maxBalance > 0 
                      ? `Limite: ${wallet.maxBalance.toLocaleString()} FCFA`
                      : 'Limite: Illimitée'
                    }
                  </p>
                  <div className="w-32 bg-amber-200 rounded-full h-2 mt-2">
                    <div 
                      className={`h-2 rounded-full ${
                        wallet.timType === 'TIM_MINI' ? 'bg-amber-600' :
                        wallet.timType === 'TIM_MAXI' ? 'bg-orange-600' : 'bg-yellow-600'
                      }`}
                      style={{ 
                        width: wallet.maxBalance > 0 
                          ? `${Math.min((wallet.balance / wallet.maxBalance) * 100, 100)}%`
                          : '50%'
                      }}
                    ></div>
                  </div>
                </div>
              </div>
            </div>
            ))
          )}
        </div>
      </div>
    </div>
  );
};

export default Wallets;
