import React, { useState } from 'react';
import { 
  Wallet as WalletIcon, 
  Plus, 
  Minus, 
  ArrowUpRight, 
  ArrowDownLeft, 
  RefreshCw,
  CreditCard,
  Eye,
  EyeOff,
  TrendingUp,
  DollarSign
} from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';

const Wallet: React.FC = () => {
  const { user } = useAuth();
  const [showBalance, setShowBalance] = useState(true);
  const [activeTab, setActiveTab] = useState('overview');

  // Mock wallet data
  const walletData = {
    balance: 12847.50,
    currency: 'USD',
    monthlyIncome: 8920.00,
    monthlyExpenses: 3247.00,
    availableCredit: 5000.00,
    pendingTransactions: 2
  };

  const quickActions = [
    {
      id: 'reload',
      title: 'Recharger',
      description: 'Ajouter des fonds',
      icon: Plus,
      color: 'bg-green-500 hover:bg-green-600',
      action: () => console.log('Reload wallet')
    },
    {
      id: 'withdraw',
      title: 'Retirer',
      description: 'Retirer des fonds',
      icon: Minus,
      color: 'bg-red-500 hover:bg-red-600',
      action: () => console.log('Withdraw funds')
    },
    {
      id: 'transfer',
      title: 'Transférer',
      description: 'Envoyer de l\'argent',
      icon: ArrowUpRight,
      color: 'bg-blue-500 hover:bg-blue-600',
      action: () => console.log('Transfer money')
    },
    {
      id: 'request',
      title: 'Demander',
      description: 'Demander un paiement',
      icon: ArrowDownLeft,
      color: 'bg-purple-500 hover:bg-purple-600',
      action: () => console.log('Request payment')
    }
  ];

  const recentActivity = [
    {
      id: '1',
      type: 'credit',
      amount: 1500,
      description: 'Rechargement par virement bancaire',
      date: '2024-01-15T10:30:00Z',
      status: 'completed'
    },
    {
      id: '2',
      type: 'debit',
      amount: 250,
      description: 'Transfert vers John Smith',
      date: '2024-01-14T15:45:00Z',
      status: 'completed'
    },
    {
      id: '3',
      type: 'debit',
      amount: 89.99,
      description: 'Paiement marchand - Amazon',
      date: '2024-01-14T09:20:00Z',
      status: 'completed'
    }
  ];

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Mon Portefeuille</h1>
          <p className="text-gray-600 mt-2">Gérez vos fonds et transactions</p>
        </div>
        <div className="flex space-x-4">
          <button className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium transition-colors duration-200">
            Nouvelle Transaction
          </button>
          <button className="border border-gray-300 text-gray-700 hover:bg-gray-50 px-4 py-2 rounded-lg font-medium transition-colors duration-200">
            Historique Complet
          </button>
        </div>
      </div>

      {/* Wallet Balance Card */}
      <div className="bg-gradient-to-br from-blue-600 to-blue-800 rounded-2xl shadow-xl p-8 text-white">
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center space-x-3">
            <div className="w-12 h-12 bg-white bg-opacity-20 rounded-xl flex items-center justify-center">
              <WalletIcon className="w-6 h-6 text-white" />
            </div>
            <div>
              <h2 className="text-xl font-semibold">Solde Principal</h2>
              <p className="text-blue-100">Portefeuille {walletData.currency}</p>
            </div>
          </div>
          <button
            onClick={() => setShowBalance(!showBalance)}
            className="p-2 hover:bg-white hover:bg-opacity-10 rounded-lg transition-colors duration-200"
          >
            {showBalance ? <Eye className="w-5 h-5" /> : <EyeOff className="w-5 h-5" />}
          </button>
        </div>
        
        <div className="mb-6">
          <div className="text-4xl font-bold mb-2">
            {showBalance ? `${walletData.balance.toLocaleString()} ${walletData.currency}` : '••••••'}
          </div>
          <div className="flex items-center space-x-4 text-blue-100">
            <div className="flex items-center space-x-1">
              <TrendingUp className="w-4 h-4" />
              <span className="text-sm">+5.2% ce mois</span>
            </div>
            <div className="flex items-center space-x-1">
              <DollarSign className="w-4 h-4" />
              <span className="text-sm">{walletData.pendingTransactions} en attente</span>
            </div>
          </div>
        </div>

        <div className="grid grid-cols-2 gap-4">
          <div className="bg-white bg-opacity-10 rounded-lg p-4">
            <p className="text-blue-100 text-sm">Revenus ce mois</p>
            <p className="text-xl font-semibold">+{walletData.monthlyIncome.toLocaleString()} {walletData.currency}</p>
          </div>
          <div className="bg-white bg-opacity-10 rounded-lg p-4">
            <p className="text-blue-100 text-sm">Dépenses ce mois</p>
            <p className="text-xl font-semibold">-{walletData.monthlyExpenses.toLocaleString()} {walletData.currency}</p>
          </div>
        </div>
      </div>

      {/* Quick Actions */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {quickActions.map((action) => {
          const IconComponent = action.icon;
          return (
            <button
              key={action.id}
              onClick={action.action}
              className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 hover:shadow-md transition-all duration-200 text-left group"
            >
              <div className="flex items-center justify-between mb-4">
                <div className={`w-12 h-12 ${action.color} rounded-lg flex items-center justify-center transition-colors duration-200`}>
                  <IconComponent className="w-6 h-6 text-white" />
                </div>
                <ArrowUpRight className="w-5 h-5 text-gray-400 group-hover:text-gray-600 transition-colors duration-200" />
              </div>
              <h3 className="font-semibold text-gray-900 mb-1">{action.title}</h3>
              <p className="text-sm text-gray-600">{action.description}</p>
            </button>
          );
        })}
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6">
            {[
              { id: 'overview', label: 'Vue d\'ensemble' },
              { id: 'transactions', label: 'Transactions' },
              { id: 'cards', label: 'Cartes' },
              { id: 'settings', label: 'Paramètres' }
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-200 ${
                  activeTab === tab.id
                    ? 'border-blue-500 text-blue-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
              >
                {tab.label}
              </button>
            ))}
          </nav>
        </div>

        <div className="p-6">
          {activeTab === 'overview' && (
            <div className="space-y-6">
              <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <div className="lg:col-span-2">
                  <h3 className="text-lg font-semibold text-gray-900 mb-4">Activité Récente</h3>
                  <div className="space-y-4">
                    {recentActivity.map((activity) => (
                      <div key={activity.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                        <div className="flex items-center space-x-4">
                          <div className={`w-10 h-10 rounded-lg flex items-center justify-center ${
                            activity.type === 'credit' ? 'bg-green-100' : 'bg-red-100'
                          }`}>
                            {activity.type === 'credit' ? (
                              <ArrowDownLeft className="w-5 h-5 text-green-600" />
                            ) : (
                              <ArrowUpRight className="w-5 h-5 text-red-600" />
                            )}
                          </div>
                          <div>
                            <p className="font-medium text-gray-900">{activity.description}</p>
                            <p className="text-sm text-gray-500">
                              {new Date(activity.date).toLocaleDateString('fr-FR')}
                            </p>
                          </div>
                        </div>
                        <div className="text-right">
                          <p className={`font-semibold ${
                            activity.type === 'credit' ? 'text-green-600' : 'text-red-600'
                          }`}>
                            {activity.type === 'credit' ? '+' : '-'}{activity.amount.toLocaleString()} {walletData.currency}
                          </p>
                          <span className="inline-flex px-2 py-1 text-xs font-medium rounded-full bg-green-100 text-green-800">
                            Terminé
                          </span>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>

                <div className="space-y-6">
                  <div className="bg-gray-50 rounded-lg p-6">
                    <h4 className="font-semibold text-gray-900 mb-4">Crédit Disponible</h4>
                    <div className="text-2xl font-bold text-gray-900 mb-2">
                      {walletData.availableCredit.toLocaleString()} {walletData.currency}
                    </div>
                    <p className="text-sm text-gray-600 mb-4">Limite de crédit approuvée</p>
                    <button className="w-full bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium transition-colors duration-200">
                      Demander un Prêt
                    </button>
                  </div>

                  <div className="bg-gray-50 rounded-lg p-6">
                    <h4 className="font-semibold text-gray-900 mb-4">Conseils Financiers</h4>
                    <p className="text-sm text-gray-600 mb-4">
                      Économisez 20% de vos revenus mensuels pour construire une base financière solide.
                    </p>
                    <button className="text-blue-600 hover:text-blue-700 text-sm font-medium">
                      En savoir plus →
                    </button>
                  </div>
                </div>
              </div>
            </div>
          )}

          {activeTab === 'transactions' && (
            <div className="text-center py-12">
              <RefreshCw className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Historique des Transactions</h3>
              <p className="text-gray-600">Fonctionnalité en cours de développement</p>
            </div>
          )}

          {activeTab === 'cards' && (
            <div className="text-center py-12">
              <CreditCard className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Gestion des Cartes</h3>
              <p className="text-gray-600">Fonctionnalité en cours de développement</p>
            </div>
          )}

          {activeTab === 'settings' && (
            <div className="text-center py-12">
              <WalletIcon className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Paramètres du Portefeuille</h3>
              <p className="text-gray-600">Fonctionnalité en cours de développement</p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default Wallet;