import React, { useState, useEffect } from 'react';
import { 
  ArrowUpRight, 
  ArrowDownLeft, 
  Filter, 
  Search,
  CheckCircle,
  Clock,
  XCircle,
  TrendingUp,
  TrendingDown,
  DollarSign
} from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';
import { apiService } from '../../services/apiService';

interface Transfer {
  id: string;
  reference: string;
  type: string;
  amount: number;
  commission_amount: number;
  status: string;
  description?: string;
  merchant_name?: string;
  location?: string;
  created_at: string;
  completed_at?: string | null;
}

interface TransferStats {
  total_transfers: number;
  total_amount_sent: number;
  total_amount_received: number;
  total_fees: number;
  completed: number;
  pending: number;
  failed: number;
}

const Transfers: React.FC = () => {
  const { } = useAuth();
  const [transfers, setTransfers] = useState<Transfer[]>([]);
  const [stats, setStats] = useState<TransferStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState({
    status: '',
    type: '',
    search: ''
  });

  useEffect(() => {
    fetchTransfers();
    fetchStats();
  }, [filter]);

  const fetchTransfers = async () => {
    try {
      setLoading(true);
      
      const params = new URLSearchParams();
      if (filter.status) params.append('status', filter.status);
      if (filter.type) params.append('type', filter.type);
      
      const data = await apiService.getTransactions(50, 0);
      setTransfers(data.transactions || []);
    } catch (error) {
      console.error('Erreur chargement transferts:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchStats = async () => {
    try {
      // Calculer les stats à partir des transactions
      const data = await apiService.getTransactions(1000, 0);
      const transactions = data.transactions || [];
      
      // Calculer les statistiques
      const transferTransactions = transactions.filter((t: Transfer) => 
        t.type === 'transfer_in' || t.type === 'transfer_out' || t.type === 'mobile_transfer'
      );
      
      const totalSent = transferTransactions
        .filter((t: Transfer) => t.type === 'transfer_out')
        .reduce((sum: number, t: Transfer) => sum + Math.abs(t.amount), 0);
      
      const totalReceived = transferTransactions
        .filter((t: Transfer) => t.type === 'transfer_in')
        .reduce((sum: number, t: Transfer) => sum + t.amount, 0);
      
      const totalFees = transferTransactions
        .reduce((sum: number, t: Transfer) => sum + (t.commission_amount || 0), 0);
      
      const completed = transferTransactions.filter((t: Transfer) => t.status === 'completed').length;
      const pending = transferTransactions.filter((t: Transfer) => t.status === 'pending').length;
      const failed = transferTransactions.filter((t: Transfer) => t.status === 'failed').length;
      
      setStats({
        total_transfers: transferTransactions.length,
        total_amount_sent: totalSent,
        total_amount_received: totalReceived,
        total_fees: totalFees,
        completed,
        pending,
        failed
      });
    } catch (error) {
      console.error('Erreur chargement stats:', error);
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="w-5 h-5 text-green-500" />;
      case 'pending':
        return <Clock className="w-5 h-5 text-yellow-500" />;
      case 'failed':
        return <XCircle className="w-5 h-5 text-red-500" />;
      default:
        return <Clock className="w-5 h-5 text-gray-500" />;
    }
  };

  const getStatusText = (status: string) => {
    const statusMap: Record<string, string> = {
      completed: 'Complété',
      pending: 'En cours',
      failed: 'Échoué',
      cancelled: 'Annulé'
    };
    return statusMap[status] || status;
  };

  const getTypeText = (type: string) => {
    const typeMap: Record<string, string> = {
      mobile_money_transfer: 'Mobile Money',
      transfer_out: 'Envoi',
      transfer_in: 'Réception',
      mobile_transfer: 'Transfert Mobile',
      virement_bancaire: 'Virement Bancaire'
    };
    return typeMap[type] || type;
  };

  const formatAmount = (amount: number) => {
    return new Intl.NumberFormat('fr-FR').format(Math.abs(amount));
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return new Intl.DateTimeFormat('fr-FR', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    }).format(date);
  };

  const filteredTransfers = transfers.filter(transfer => {
    if (filter.search) {
      const searchLower = filter.search.toLowerCase();
      return (
        transfer.reference.toLowerCase().includes(searchLower) ||
        transfer.description?.toLowerCase().includes(searchLower) ||
        transfer.merchant_name?.toLowerCase().includes(searchLower)
      );
    }
    return true;
  });

  return (
    <div className="min-h-screen bg-gradient-to-br from-amber-50 via-orange-50 to-red-50 p-6">
      <div className="max-w-7xl mx-auto">
        {/* En-tête */}
        <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6 mb-8">
          <h1 className="text-3xl font-bold text-amber-900 mb-2">🔄 Historique des Transferts</h1>
          <p className="text-amber-700">
            Consultez tous vos transferts effectués depuis votre compte TIM CASH
          </p>
        </div>

        {/* Statistiques */}
        {stats && (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-lg border border-amber-200 p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Total Transferts</p>
                  <p className="text-2xl font-bold text-gray-900">{stats.total_transfers}</p>
                </div>
                <DollarSign className="w-10 h-10 text-blue-500 opacity-20" />
              </div>
            </div>

            <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-xl shadow-lg border border-red-200 p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Montant Envoyé</p>
                  <p className="text-2xl font-bold text-gray-900">{formatAmount(stats.total_amount_sent)} FCFA</p>
                </div>
                <TrendingDown className="w-10 h-10 text-red-500 opacity-20" />
              </div>
            </div>

            <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-lg border border-green-200 p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Montant Reçu</p>
                  <p className="text-2xl font-bold text-gray-900">{formatAmount(stats.total_amount_received)} FCFA</p>
                </div>
                <TrendingUp className="w-10 h-10 text-green-500 opacity-20" />
              </div>
            </div>

            <div className="bg-gradient-to-br from-yellow-50 to-amber-50 rounded-xl shadow-lg border border-yellow-200 p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Frais Totaux</p>
                  <p className="text-2xl font-bold text-gray-900">{formatAmount(stats.total_fees)} FCFA</p>
                </div>
                <DollarSign className="w-10 h-10 text-amber-500 opacity-20" />
              </div>
            </div>
          </div>
        )}

        {/* Filtres */}
        <div className="bg-gradient-to-r from-amber-50/50 to-orange-50/50 rounded-xl shadow-lg border border-amber-200 p-6 mb-6">
          <div className="flex items-center space-x-2 mb-4">
            <Filter className="w-5 h-5 text-amber-600" />
            <h2 className="text-lg font-semibold text-gray-900">Filtres</h2>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            {/* Recherche */}
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
              <input
                type="text"
                placeholder="Rechercher..."
                value={filter.search}
                onChange={(e) => setFilter({ ...filter, search: e.target.value })}
                className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
              />
            </div>

            {/* Filtre par statut */}
            <select
              value={filter.status}
              onChange={(e) => setFilter({ ...filter, status: e.target.value })}
              className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
            >
              <option value="">Tous les statuts</option>
              <option value="completed">Complété</option>
              <option value="pending">En cours</option>
              <option value="failed">Échoué</option>
            </select>

            {/* Filtre par type */}
            <select
              value={filter.type}
              onChange={(e) => setFilter({ ...filter, type: e.target.value })}
              className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
            >
              <option value="">Tous les types</option>
              <option value="mobile_money_transfer">Mobile Money</option>
              <option value="transfer_out">Envoi</option>
              <option value="transfer_in">Réception</option>
              <option value="virement_bancaire">Virement Bancaire</option>
            </select>
          </div>
        </div>

        {/* Liste des transferts */}
        <div className="bg-white rounded-xl shadow-lg border border-amber-200 overflow-hidden">
          <div className="px-6 py-4 border-b border-gray-200">
            <h2 className="text-lg font-semibold text-gray-900">
              Transferts ({filteredTransfers.length})
            </h2>
          </div>

          {loading ? (
            <div className="flex items-center justify-center py-12">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600"></div>
            </div>
          ) : filteredTransfers.length === 0 ? (
            <div className="text-center py-12">
              <p className="text-gray-500">Aucun transfert trouvé</p>
            </div>
          ) : (
            <div className="divide-y divide-gray-200">
              {filteredTransfers.map((transfer) => (
                <div
                  key={transfer.id}
                  className="px-6 py-4 hover:bg-amber-50/30 transition-colors cursor-pointer"
                >
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4 flex-1">
                      {/* Icône */}
                      <div className={`p-3 rounded-full ${
                        transfer.amount < 0 ? 'bg-red-100' : 'bg-green-100'
                      }`}>
                        {transfer.amount < 0 ? (
                          <ArrowUpRight className="w-6 h-6 text-red-600" />
                        ) : (
                          <ArrowDownLeft className="w-6 h-6 text-green-600" />
                        )}
                      </div>

                      {/* Détails */}
                      <div className="flex-1">
                        <div className="flex items-center space-x-2">
                          <p className="font-semibold text-gray-900">
                            {transfer.merchant_name || getTypeText(transfer.type)}
                          </p>
                          {getStatusIcon(transfer.status)}
                        </div>
                        <p className="text-sm text-gray-500">{transfer.reference}</p>
                        {transfer.description && (
                          <p className="text-sm text-gray-600 mt-1">{transfer.description}</p>
                        )}
                        <p className="text-xs text-gray-400 mt-1">
                          {formatDate(transfer.created_at)}
                        </p>
                      </div>

                      {/* Montant */}
                      <div className="text-right">
                        <p className={`text-lg font-bold ${
                          transfer.amount < 0 ? 'text-red-600' : 'text-green-600'
                        }`}>
                          {transfer.amount < 0 ? '-' : '+'} {formatAmount(transfer.amount)} FCFA
                        </p>
                        {transfer.commission_amount > 0 && (
                          <p className="text-xs text-gray-500">
                            Frais: {formatAmount(transfer.commission_amount)} FCFA
                          </p>
                        )}
                        <p className="text-xs text-gray-500 mt-1">
                          {getStatusText(transfer.status)}
                        </p>
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default Transfers;
