import React, { useState, useEffect } from 'react';
import { PlusCircle, TrendingUp, Activity, Clock, CheckCircle, XCircle, Search, Filter } from 'lucide-react';
import { getRecharges } from '../../services/api';

interface RechargeData {
  id: string;
  userId: string;
  userName: string;
  timType: 'TIM_MINI' | 'TIM_MAXI' | 'TIM_BUSINESS';
  amount: number;
  commission: number;
  method: 'cinetpay' | 'tim_business';
  status: 'pending' | 'completed' | 'failed';
  transactionId: string;
  createdAt: string;
  completedAt?: string;
}

const Recharges: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedStatus, setSelectedStatus] = useState<string>('all');
  const [selectedMethod, setSelectedMethod] = useState<string>('all');
  const [recharges, setRecharges] = useState<RechargeData[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Récupérer les recharges depuis l'API
  useEffect(() => {
    const fetchRecharges = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await getRecharges();

        // Transformer les données de l'API au format attendu par le frontend
        const rechargesData = response.recharges.map((recharge: any) => ({
          id: recharge.id,
          userId: recharge.user_id,
          userName: recharge.user_name,
          timType: recharge.tim_type,
          amount: recharge.amount,
          commission: recharge.commission,
          method: recharge.method,
          status: recharge.status,
          transactionId: recharge.transaction_id,
          createdAt: recharge.created_at,
          completedAt: recharge.completed_at
        }));

        setRecharges(rechargesData);
      } catch (err: any) {
        console.error('Erreur lors de la récupération des recharges:', err);
        setError(err.message || 'Erreur lors de la récupération des recharges');
      } finally {
        setLoading(false);
      }
    };

    fetchRecharges();
  }, []);

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="w-4 h-4 text-amber-600" />;
      case 'pending':
        return <Clock className="w-4 h-4 text-yellow-600" />;
      case 'failed':
        return <XCircle className="w-4 h-4 text-red-600" />;
      default:
        return <Activity className="w-4 h-4 text-amber-600" />;
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case 'completed':
        return 'Terminé';
      case 'pending':
        return 'En cours';
      case 'failed':
        return 'Échoué';
      default:
        return status;
    }
  };

  const getMethodLabel = (method: string) => {
    switch (method) {
      case 'cinetpay':
        return 'CinetPay';
      case 'tim_business':
        return 'TIM BUSINESS';
      default:
        return method;
    }
  };

  const getMethodColor = (method: string) => {
    switch (method) {
      case 'cinetpay':
        return 'bg-amber-100 text-amber-800';
      case 'tim_business':
        return 'bg-orange-100 text-orange-800';
      default:
        return 'bg-amber-100 text-amber-800';
    }
  };

  const getTimTypeColor = (type: string) => {
    switch (type) {
      case 'TIM_MINI':
        return 'bg-amber-100 text-amber-800';
      case 'TIM_MAXI':
        return 'bg-orange-100 text-orange-800';
      case 'TIM_BUSINESS':
        return 'bg-yellow-100 text-yellow-800';
      default:
        return 'bg-amber-100 text-amber-800';
    }
  };

  const filteredRecharges = recharges.filter(recharge => {
    const matchesSearch = recharge.userName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         recharge.transactionId.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = selectedStatus === 'all' || recharge.status === selectedStatus;
    const matchesMethod = selectedMethod === 'all' || recharge.method === selectedMethod;
    return matchesSearch && matchesStatus && matchesMethod;
  });

  const totalAmount = recharges.reduce((sum, recharge) => sum + recharge.amount, 0);
  const totalCommission = recharges.reduce((sum, recharge) => sum + recharge.commission, 0);
  const completedRecharges = recharges.filter(r => r.status === 'completed').length;
  const pendingRecharges = recharges.filter(r => r.status === 'pending').length;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-amber-100 rounded-lg">
              <PlusCircle className="w-6 h-6 text-amber-600" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-amber-900">Recharges CinetPay</h1>
              <p className="text-amber-600">Supervision des recharges mobiles TIM CASH</p>
            </div>
          </div>
        </div>
      </div>

      {/* Statistiques */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-600">Total Recharges</p>
              <p className="text-2xl font-bold text-amber-900">{recharges.length}</p>
            </div>
            <Activity className="w-8 h-8 text-amber-600" />
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-600">Montant Total</p>
              <p className="text-2xl font-bold text-amber-700">{totalAmount.toLocaleString()} FCFA</p>
            </div>
            <TrendingUp className="w-8 h-8 text-amber-700" />
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-600">Commissions</p>
              <p className="text-2xl font-bold text-orange-600">{totalCommission.toLocaleString()} FCFA</p>
            </div>
            <PlusCircle className="w-8 h-8 text-orange-600" />
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-600">Taux de Réussite</p>
              <p className="text-2xl font-bold text-yellow-600">
                {Math.round((completedRecharges / recharges.length) * 100)}%
              </p>
            </div>
            <CheckCircle className="w-8 h-8 text-yellow-600" />
          </div>
        </div>
      </div>

      {/* Répartition par statut */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-amber-600">Terminées</span>
            <span className="text-lg font-bold text-amber-700">{completedRecharges}</span>
          </div>
          <div className="w-full bg-amber-200 rounded-full h-2">
            <div 
              className="bg-amber-600 h-2 rounded-full" 
              style={{ width: `${(completedRecharges / recharges.length) * 100}%` }}
            ></div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-amber-600">En cours</span>
            <span className="text-lg font-bold text-yellow-600">{pendingRecharges}</span>
          </div>
          <div className="w-full bg-yellow-200 rounded-full h-2">
            <div 
              className="bg-yellow-600 h-2 rounded-full" 
              style={{ width: `${(pendingRecharges / recharges.length) * 100}%` }}
            ></div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-amber-600">Échouées</span>
            <span className="text-lg font-bold text-red-600">
              {recharges.filter(r => r.status === 'failed').length}
            </span>
          </div>
          <div className="w-full bg-red-200 rounded-full h-2">
            <div 
              className="bg-red-600 h-2 rounded-full" 
              style={{ width: `${(recharges.filter(r => r.status === 'failed').length / recharges.length) * 100}%` }}
            ></div>
          </div>
        </div>
      </div>

      {/* Filtres */}
      <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
        <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-4 md:space-y-0">
          <div className="flex items-center space-x-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
              <input
                type="text"
                placeholder="Rechercher par nom ou ID transaction..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              />
            </div>
            <div className="flex items-center space-x-2">
              <Filter className="w-4 h-4 text-gray-400" />
              <select
                value={selectedStatus}
                onChange={(e) => setSelectedStatus(e.target.value)}
                className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              >
                <option value="all">Tous les statuts</option>
                <option value="completed">Terminé</option>
                <option value="pending">En cours</option>
                <option value="failed">Échoué</option>
              </select>
              <select
                value={selectedMethod}
                onChange={(e) => setSelectedMethod(e.target.value)}
                className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              >
                <option value="all">Toutes les méthodes</option>
                <option value="cinetpay">CinetPay</option>
                <option value="tim_business">TIM BUSINESS</option>
              </select>
            </div>
          </div>
        </div>
      </div>

      {/* Liste des recharges */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">
            Recharges Récentes ({filteredRecharges.length})
          </h3>
        </div>
        <div className="divide-y divide-gray-200">
          {loading ? (
            <div className="p-12 text-center">
              <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600"></div>
              <p className="mt-4 text-amber-600">Chargement des recharges...</p>
            </div>
          ) : error ? (
            <div className="p-12 text-center">
              <p className="text-red-600 font-medium">{error}</p>
              <button
                onClick={() => window.location.reload()}
                className="mt-4 px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700"
              >
                Réessayer
              </button>
            </div>
          ) : filteredRecharges.length === 0 ? (
            <div className="p-12 text-center">
              <PlusCircle className="w-16 h-16 text-amber-300 mx-auto mb-4" />
              <p className="text-amber-600 font-medium">Aucune recharge trouvée</p>
              <p className="text-amber-500 text-sm mt-2">
                {searchTerm || selectedStatus !== 'all' || selectedMethod !== 'all'
                  ? 'Essayez de modifier vos filtres de recherche'
                  : 'Les recharges effectuées depuis l\'application mobile apparaîtront ici'}
              </p>
            </div>
          ) : (
            filteredRecharges.map((recharge) => (
              <div key={recharge.id} className="p-6 hover:bg-gray-50 transition-colors">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <div className="w-12 h-12 bg-gradient-to-r from-green-500 to-blue-600 rounded-full flex items-center justify-center">
                    {getStatusIcon(recharge.status)}
                  </div>
                  <div>
                    <div className="flex items-center space-x-2 mb-1">
                      <h4 className="font-medium text-gray-900">{recharge.userName}</h4>
                      <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getTimTypeColor(recharge.timType)}`}>
                        {recharge.timType}
                      </span>
                      <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getMethodColor(recharge.method)}`}>
                        {getMethodLabel(recharge.method)}
                      </span>
                      <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(recharge.status)}`}>
                        {getStatusLabel(recharge.status)}
                      </span>
                    </div>
                    <div className="flex items-center space-x-4 text-sm text-gray-500">
                      <span>ID: {recharge.transactionId}</span>
                      <span>Créé: {new Date(recharge.createdAt).toLocaleDateString('fr-FR')} à {new Date(recharge.createdAt).toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' })}</span>
                      {recharge.completedAt && (
                        <span>Terminé: {new Date(recharge.completedAt).toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' })}</span>
                      )}
                    </div>
                  </div>
                </div>
                <div className="text-right">
                  <p className="text-lg font-bold text-gray-900">
                    {recharge.amount.toLocaleString()} FCFA
                  </p>
                  <p className="text-sm text-gray-500">
                    Commission: {recharge.commission.toLocaleString()} FCFA
                  </p>
                  <p className="text-xs text-gray-400">
                    Taux: {((recharge.commission / recharge.amount) * 100).toFixed(2)}%
                  </p>
                </div>
              </div>
            </div>
            ))
          )}
        </div>
      </div>
    </div>
  );
};

export default Recharges;
