import React, { useState } from "react";
import {
  Users as UsersIcon,
  Search,
  Filter,
  MoreHorizontal,
  Edit,
  Shield,
  ShieldOff,
  Eye,
  Mail,
  Phone,
  Calendar,
  UserCheck,
  UserX,
  Crown,
  User,
  RefreshCw,
} from "lucide-react";
import { useUsers } from "../../hooks/useUsers";

const Users: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedRole, setSelectedRole] = useState("all");
  const [selectedStatus, setSelectedStatus] = useState("all");
  const [selectedUser, setSelectedUser] = useState<string | null>(null);
  const [pieceId, setPieceId] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [validationResult, setValidationResult] = useState(null);
  const [errorMessage, setErrorMessage] = useState("");
  const userToken = localStorage.getItem("tim_access_token") || "";

  // Utiliser le hook useUsers pour charger les données réelles avec auto-refresh
  const { users, totalCount, loading, error, refetch } = useUsers(true, 60000);

  const getRoleIcon = (role: string) => {
    switch (role) {
      case "admin":
        return <Crown className="w-4 h-4 text-yellow-600" />;
      case "manager":
        return <Shield className="w-4 h-4 text-blue-600" />;
      default:
        return <User className="w-4 h-4 text-gray-600" />;
    }
  };

  const getRoleColor = (role: string) => {
    switch (role) {
      case "admin":
        return "bg-yellow-100 text-yellow-800";
      case "manager":
        return "bg-blue-100 text-blue-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getRoleLabel = (role: string) => {
    switch (role) {
      case "admin":
        return "Administrateur";
      case "manager":
        return "Gestionnaire";
      default:
        return "Utilisateur";
    }
  };

  const validatePiece = async (pieceId: string) => {
    setIsLoading(true);
    setErrorMessage("");
    setValidationResult(null);
    console.log(pieceId);
    const baseUrl = import.meta.env.VITE_API_URL;
    console.log(baseUrl);

    try {
      const response = await fetch(
        `${baseUrl}/api/financials/admin/kyc/validate/${pieceId}`,
        {
          method: "PATCH",
          headers: {
            "Content-Type": "application/json",
            authorization: `Bearer ${userToken}`,
          },
        }
      );

      if (!response.ok) {
        throw new Error("Erreur lors de la validation de la pièce");
      }

      const result = await response.json();
      setValidationResult(result);
    } catch (error) {
      setErrorMessage(error.message);
    } finally {
      setIsLoading(false);
    }
  };
  const activateUser = async (userId: string) => {
    const baseUrl = import.meta.env.VITE_API_URL;
    try {
      const response = await fetch(`${baseUrl}/api/active/${userId}`, {
        method: "PATCH",
        headers: {
          authorization: `Bearer ${userToken}`,
        },
      });

      if (!response.ok) {
        throw new Error("Erreur lors de l'activation du compte");
      }

      await refetch(); // recharge les utilisateurs
      alert("Compte activé avec succès");
    } catch (error) {
      alert(error.message);
    }
  };

  const blockUser = async (userId: string) => {
    const baseUrl = import.meta.env.VITE_API_URL;
    try {
      const response = await fetch(`${baseUrl}/api/block/${userId}`, {
        method: "PATCH",
        headers: {
          authorization: `Bearer ${userToken}`,
        },
      });

      if (!response.ok) {
        throw new Error("Erreur lors de la désactivation du compte");
      }

      await refetch(); // recharge les utilisateurs
      alert("Compte bloqué avec succès");
    } catch (error) {
      alert(error.message);
    }
  };
  const filteredUsers = users.filter((user) => {
    const matchesSearch =
      user.firstName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.lastName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.email.toLowerCase().includes(searchTerm.toLowerCase());

    const matchesRole = selectedRole === "all" || user.role === selectedRole;

    const matchesStatus =
      selectedStatus === "all" ||
      (selectedStatus === "verified" && user.isVerified) ||
      (selectedStatus === "unverified" && !user.isVerified) ||
      (selectedStatus === "blocked" && user.isBlocked) ||
      (selectedStatus === "active" && !user.isBlocked);

    return matchesSearch && matchesRole && matchesStatus;
  });

  const totalUsers = users.length;
  const verifiedUsers = users.filter((u) => u.isVerified).length;
  const blockedUsers = users.filter((u) => u.isBlocked).length;
  const totalWalletBalance = users.reduce((sum, u) => sum + u.walletBalance, 0);

  // Afficher un loader pendant le chargement
  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600 mx-auto"></div>
          <p className="mt-4 text-amber-700 font-medium">
            Chargement des utilisateurs...
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-8">
      {/* Afficher l'erreur si présente */}
      {error && (
        <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
          <div className="flex items-start space-x-3">
            <div className="flex-1">
              <p className="font-semibold">Erreur de chargement</p>
              <p className="text-sm mt-1">{error}</p>
              {error.includes("Non authentifié") && (
                <p className="text-sm mt-2">
                  💡 <strong>Solution:</strong> Déconnectez-vous puis
                  reconnectez-vous pour obtenir un nouveau token.
                </p>
              )}
              {error.includes("401") && (
                <p className="text-sm mt-2">
                  💡 <strong>Solution:</strong> Votre session a expiré.
                  Reconnectez-vous à l'application.
                </p>
              )}
            </div>
          </div>
          <div className="mt-3 flex space-x-2">
            <button
              onClick={fetchUsers}
              className="text-sm bg-red-600 hover:bg-red-700 text-white px-3 py-1 rounded"
            >
              Réessayer
            </button>
            <button
              onClick={() => {
                localStorage.clear();
                window.location.href = "/login";
              }}
              className="text-sm bg-gray-600 hover:bg-gray-700 text-white px-3 py-1 rounded"
            >
              Se reconnecter
            </button>
          </div>
        </div>
      )}

      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-amber-900">
              👥 Gestion des Utilisateurs
            </h1>
            <p className="text-amber-700 mt-2">
              Gérez les comptes utilisateurs TIM CASH et leurs permissions
            </p>
          </div>
          <div className="flex space-x-4">
            <button
              onClick={() =>
                alert("Fonction d'export en cours de développement")
              }
              className="bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg font-medium transition-all duration-200 shadow-md hover:shadow-lg transform hover:scale-105"
            >
              Exporter
            </button>
          </div>
        </div>
      </div>

      {/* Stats Cards avec thème marron doré */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-lg border border-amber-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-amber-700">
                Total Utilisateurs
              </p>
              <p className="text-3xl font-bold text-amber-900">{totalUsers}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg flex items-center justify-center shadow-md">
              <UsersIcon className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-lg border border-green-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-green-700">
                Utilisateurs Vérifiés
              </p>
              <p className="text-3xl font-bold text-green-900">
                {verifiedUsers}
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-green-400 to-emerald-500 rounded-lg flex items-center justify-center shadow-md">
              <UserCheck className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-xl shadow-lg border border-red-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-red-700">
                Utilisateurs Bloqués
              </p>
              <p className="text-3xl font-bold text-red-900">{blockedUsers}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-red-400 to-rose-500 rounded-lg flex items-center justify-center shadow-md">
              <UserX className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-purple-50 to-indigo-50 rounded-xl shadow-lg border border-purple-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-purple-700">Solde Total</p>
              <p className="text-3xl font-bold text-purple-900">
                {totalWalletBalance.toLocaleString()} FCFA
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-purple-400 to-indigo-500 rounded-lg flex items-center justify-center shadow-md">
              <Shield className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Filters avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/50 to-orange-50/50 rounded-xl shadow-lg border border-amber-200 p-6">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
          <div className="flex flex-col sm:flex-row space-y-4 sm:space-y-0 sm:space-x-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
              <input
                type="text"
                placeholder="Rechercher par nom ou email..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 pr-4 py-2 w-full sm:w-80 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              />
            </div>

            <select
              value={selectedRole}
              onChange={(e) => setSelectedRole(e.target.value)}
              className="px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
            >
              <option value="all">Tous les rôles</option>
              <option value="admin">Administrateurs</option>
              <option value="manager">Gestionnaires</option>
              <option value="user">Utilisateurs</option>
            </select>

            <select
              value={selectedStatus}
              onChange={(e) => setSelectedStatus(e.target.value)}
              className="px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
            >
              <option value="all">Tous les statuts</option>
              <option value="active">Actifs</option>
              <option value="verified">Vérifiés</option>
              <option value="unverified">Non vérifiés</option>
              <option value="blocked">Bloqués</option>
            </select>
          </div>

          <div className="flex items-center space-x-4">
            <span className="text-sm text-gray-600">
              {filteredUsers.length} utilisateur(s) trouvé(s)
            </span>
            <button className="flex items-center space-x-2 px-4 py-2 border border-amber-300 rounded-lg hover:bg-amber-50 transition-colors duration-200">
              <Filter className="w-4 h-4" />
              <span>Plus de filtres</span>
            </button>
          </div>
        </div>
      </div>

      {/* Users Table avec thème marron doré */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200 overflow-hidden">
        <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50/30 to-orange-50/30">
          <h3 className="text-lg font-semibold text-amber-900">
            Liste des Utilisateurs
          </h3>
        </div>

        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gradient-to-r from-amber-50 to-orange-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Utilisateur
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Rôle
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Statut
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Solde Portefeuille
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Transactions
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Dernière Connexion
                </th>
                <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {filteredUsers.map((user) => (
                <tr
                  key={user.id}
                  className="hover:bg-amber-50/30 transition-colors duration-150"
                >
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="flex items-center">
                      <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-blue-600 rounded-full flex items-center justify-center">
                        <span className="text-white font-medium text-sm">
                          {user.firstName[0]}
                          {user.lastName[0]}
                        </span>
                      </div>
                      <div className="ml-4">
                        <div className="text-sm font-medium text-gray-900">
                          {user.firstName} {user.lastName}
                        </div>
                        <div className="text-sm text-gray-500 flex items-center">
                          <Mail className="w-3 h-3 mr-1" />
                          {user.email}
                        </div>
                        <div className="text-sm text-gray-500 flex items-center">
                          <Phone className="w-3 h-3 mr-1" />
                          {user.phone}
                        </div>
                      </div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span
                      className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getRoleColor(
                        user.role
                      )}`}
                    >
                      {getRoleIcon(user.role)}
                      <span className="ml-1">{getRoleLabel(user.role)}</span>
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="flex flex-col space-y-1">
                      {user.isVerified ? (
                        <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                          <UserCheck className="w-3 h-3 mr-1" />
                          Vérifié
                        </span>
                      ) : (
                        <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                          <UserX className="w-3 h-3 mr-1" />
                          Non vérifié
                        </span>
                      )}
                      {user.isBlocked && (
                        <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
                          <ShieldOff className="w-3 h-3 mr-1" />
                          Bloqué
                        </span>
                      )}
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    <div className="font-medium">
                      {user.walletBalance.toLocaleString()} F CFA
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    <div className="font-medium">{user.totalTransactions}</div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    <div className="flex items-center">
                      <Calendar className="w-3 h-3 mr-1" />
                      {new Date(user.lastLogin).toLocaleDateString("fr-FR")}
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                    <div className="flex items-center justify-end space-x-2">
                      <button
                        onClick={() =>
                          setSelectedUser(
                            selectedUser === user.id ? null : user.id
                          )
                        }
                        className="text-blue-600 hover:text-blue-900 p-1 rounded hover:bg-blue-50 transition-colors duration-200"
                      >
                        <Eye className="w-4 h-4" />
                      </button>
                      <button
                        onClick={() =>
                          alert(
                            `Modifier l'utilisateur ${user.firstName} ${user.lastName}`
                          )
                        }
                        className="text-gray-600 hover:text-gray-900 p-1 rounded hover:bg-gray-50 transition-colors duration-200"
                      >
                        <Edit className="w-4 h-4" />
                      </button>
                      {!user.isBlocked ? (
                        <button
                          onClick={() => blockUser(user.id)}
                          className="text-red-600 hover:text-red-900 p-1 rounded hover:bg-red-50 transition"
                        >
                          <ShieldOff className="w-4 h-4" />
                        </button>
                      ) : (
                        <button
                          onClick={() => activateUser(user.id)}
                          className="text-green-600 hover:text-green-900 p-1 rounded hover:bg-green-50 transition"
                        >
                          <Shield className="w-4 h-4" />
                        </button>
                      )}
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {filteredUsers.length === 0 && (
          <div className="p-12 text-center">
            <UsersIcon className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">
              Aucun utilisateur trouvé
            </h3>
            <p className="text-gray-600">
              Aucun utilisateur ne correspond à vos critères de recherche.
            </p>
          </div>
        )}
      </div>

      {/* User Details Modal */}
      {selectedUser && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-xl shadow-xl max-w-2xl w-full mx-4 max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-gray-200">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold text-gray-900">
                  Détails de l'Utilisateur
                </h3>
                <button
                  onClick={() => setSelectedUser(null)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  ×
                </button>
              </div>
            </div>
            <div className="p-6">
              {(() => {
                const userDetails = users.find((u) => u.id === selectedUser);
                if (!userDetails) return null;

                return (
                  <div className="space-y-6">
                    <div className="flex items-center space-x-4">
                      <div className="w-16 h-16 bg-gradient-to-br from-blue-500 to-blue-600 rounded-full flex items-center justify-center">
                        <span className="text-white font-medium text-xl">
                          {userDetails.firstName[0]}
                          {userDetails.lastName[0]}
                        </span>
                      </div>
                      <div>
                        <h4 className="text-xl font-semibold text-gray-900">
                          {userDetails.firstName} {userDetails.lastName}
                        </h4>
                        <p className="text-gray-600">{userDetails.email}</p>
                      </div>
                    </div>

                    <div className="grid grid-cols-2 gap-6">
                      <div>
                        <h5 className="font-medium text-gray-900 mb-3">
                          Informations Personnelles
                        </h5>
                        <div className="space-y-2 text-sm">
                          <div className="flex justify-between">
                            <span className="text-gray-600">Téléphone:</span>
                            <span>{userDetails.phone}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Rôle:</span>
                            <span>{getRoleLabel(userDetails.role)}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Vérifié:</span>
                            <span>
                              {userDetails.isVerified ? "Oui" : "Non"}
                            </span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Bloqué:</span>
                            <span>{userDetails.isBlocked ? "Oui" : "Non"}</span>
                          </div>
                        </div>
                      </div>

                      <div>
                        <h5 className="font-medium text-gray-900 mb-3">
                          Activité Financière
                        </h5>
                        <div className="space-y-2 text-sm">
                          <div className="flex justify-between">
                            <span className="text-gray-600">Solde:</span>
                            <span className="font-medium">
                              {userDetails.walletBalance.toLocaleString()} F CFA
                            </span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Transactions:</span>
                            <span>{userDetails.totalTransactions}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Inscription:</span>
                            <span>
                              {new Date(
                                userDetails.createdAt
                              ).toLocaleDateString("fr-FR")}
                            </span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">
                              Dernière connexion:
                            </span>
                            <span>
                              {new Date(
                                userDetails.lastLogin
                              ).toLocaleDateString("fr-FR")}
                            </span>
                          </div>
                        </div>
                      </div>
                    </div>

                    {/* SECTION KYC */}
                    {userDetails.kyc && (
                      <div className="bg-gradient-to-br from-amber-50 to-orange-50 border border-amber-200 rounded-xl p-6 shadow-sm space-y-4">
                        <h5 className="text-lg font-semibold text-amber-900 flex items-center">
                          <Shield className="w-5 h-5 mr-2 text-amber-700" />
                          Vérification d'identité (KYC)
                        </h5>

                        {/* Statut */}
                        <div className="flex items-center space-x-3">
                          <span className="text-sm text-gray-700 font-medium">
                            Statut :
                          </span>
                          {userDetails.kyc.status === "approved" && (
                            <span className="px-3 py-1 bg-green-100 text-green-800 rounded-full text-xs font-semibold">
                              ✔ Approuvé
                            </span>
                          )}
                          {userDetails.kyc.status === "pending" && (
                            <span className="px-3 py-1 bg-yellow-100 text-yellow-800 rounded-full text-xs font-semibold">
                              ⏳ En attente
                            </span>
                          )}
                          {userDetails.kyc.status === "rejected" && (
                            <span className="px-3 py-1 bg-red-100 text-red-800 rounded-full text-xs font-semibold">
                              ✖ Rejeté
                            </span>
                          )}
                        </div>

                        {/* Dates KYC */}
                        <div className="grid grid-cols-2 gap-4 text-sm">
                          <div className="flex justify-between">
                            <span className="text-gray-600">Soumis le :</span>
                            <span>
                              {new Date(
                                userDetails.kyc.submitted_at
                              ).toLocaleDateString("fr-FR")}
                            </span>
                          </div>
                          {userDetails.kyc.reviewed_at && (
                            <div className="flex justify-between">
                              <span className="text-gray-600">
                                Vérifié le :
                              </span>
                              <span>
                                {new Date(
                                  userDetails.kyc.reviewed_at
                                ).toLocaleDateString("fr-FR")}
                              </span>
                            </div>
                          )}
                        </div>

                        {/* IMAGES KYC */}
                        <div>
                          <h6 className="font-medium text-gray-800 mb-3">
                            Documents soumis :
                          </h6>

                          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                            {/* Recto */}
                            <div>
                              <p className="text-sm text-gray-600 mb-1">
                                Recto :
                              </p>
                              <img
                                src={userDetails.kyc.document_front_image}
                                alt="Document recto"
                                className="rounded-lg border border-amber-200 shadow-sm max-h-52 w-full object-cover"
                              />
                            </div>

                            {/* Verso */}
                            {userDetails.kyc.document_back_image && (
                              <div>
                                <p className="text-sm text-gray-600 mb-1">
                                  Verso :
                                </p>
                                <img
                                  src={userDetails.kyc.document_back_image}
                                  alt="Document verso"
                                  className="rounded-lg border border-amber-200 shadow-sm max-h-52 w-full object-cover"
                                />
                              </div>
                            )}
                          </div>
                        </div>
                      </div>
                    )}

                    <div className="flex space-x-4 pt-4 border-t border-gray-200">
                      <button
                        onClick={() => validatePiece(userDetails.kyc.kyc_id)}
                        className="flex-1 bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium transition-colors duration-200"
                      >
                        Valider Kyc
                      </button>
                      {userDetails.isBlocked ? (
                        <button
                          onClick={() => validatePiece(userDetails.kyc.kyc_id)}
                          disabled={isLoading}
                          className="flex-1 bg-green-600 hover:bg-green-700 text-white py-2 px-4 rounded-lg font-medium transition-colors duration-200"
                        >
                          Débloquer
                        </button>
                      ) : (
                        <button
                          onClick={() => {
                            if (
                              confirm(
                                `Bloquer ${userDetails.firstName} ${userDetails.lastName}?`
                              )
                            ) {
                              alert(
                                "Fonction de blocage en cours d'implémentation"
                              );
                              setSelectedUser(null);
                            }
                          }}
                          className="flex-1 bg-red-600 hover:bg-red-700 text-white py-2 px-4 rounded-lg font-medium transition-colors duration-200"
                        >
                          Bloquer
                        </button>
                      )}
                    </div>
                  </div>
                );
              })()}
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Users;
