import React, { useState } from 'react';
import { 
  ArrowUpRight, 
  ArrowDownLeft, 
  RefreshCw, 
  CreditCard, 
  DollarSign,
  Search,
  Filter,
  Download,
  Eye,
  MoreHorizontal,
  CheckCircle,
  Clock,
  XCircle,
  AlertCircle
} from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';
import { useTransactions } from '../../hooks/useTransactions';
import { useAdminTransactions } from '../../hooks/useAdminTransactions';

// Interface unifiée pour les transactions
interface UnifiedTransaction {
  id: string;
  reference: string;
  transaction_type: string;
  amount: number;
  commission_amount: number;
  status: string;
  description?: string;
  merchant_name?: string;
  location?: string;
  created_at: string;
  completed_at?: string;
  user_email?: string;
  user_name?: string;
  payment_method?: string;
}

const Transactions: React.FC = () => {
  const { user } = useAuth();
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedFilter, setSelectedFilter] = useState('all');
  const [dateRange, setDateRange] = useState('30');
  const [selectedTransaction, setSelectedTransaction] = useState<string | null>(null);

  const isAdmin = user?.role === 'admin' || user?.role === 'manager' || (user as any)?.is_admin;

  // Charger les transactions selon le rôle
  const userTransactionsResult = useTransactions(100, 0, !isAdmin, 30000);
  const adminTransactionsResult = useAdminTransactions(100, 0, selectedFilter !== 'all' ? selectedFilter : undefined, undefined, isAdmin, 30000);
  
  // Utiliser les bonnes données selon le rôle
  const transactionsData = isAdmin ? adminTransactionsResult : userTransactionsResult;
  const { transactions: rawTransactions, total, loading, refetch } = transactionsData;

  // Normaliser les transactions pour une interface unifiée
  const transactions: UnifiedTransaction[] = Array.isArray(rawTransactions)
    ? rawTransactions.map((t: any) => ({
        id: t.id,
        reference: t.reference,
        transaction_type: t.transaction_type || t.type,
        amount: t.amount,
        commission_amount: t.commission_amount || 0,
        status: t.status,
        description: t.description || `Transaction ${t.transaction_type || t.type}`,
        merchant_name: t.merchant_name,
        location: t.location,
        created_at: t.created_at || t.createdAt,
        completed_at: t.completed_at,
        user_email: t.user_email,
        user_name: t.user_name,
        payment_method: t.payment_method
      }))
    : [];

  const getTransactionIcon = (type: string) => {
    switch (type) {
      case 'credit':
      case 'recharge_cinetpay':
      case 'nfc_deposit':
        return <ArrowDownLeft className="w-5 h-5 text-green-600" />;
      case 'debit':
      case 'nfc_withdrawal':
        return <ArrowUpRight className="w-5 h-5 text-red-600" />;
      case 'transfer':
      case 'mobile_transfer':
      case 'mobile_money_transfer':
        return <RefreshCw className="w-5 h-5 text-blue-600" />;
      case 'loan':
      case 'pret_mobile':
        return <CreditCard className="w-5 h-5 text-purple-600" />;
      case 'repayment':
      case 'remboursement_auto':
        return <DollarSign className="w-5 h-5 text-indigo-600" />;
      default:
        return <DollarSign className="w-5 h-5 text-gray-600" />;
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="w-4 h-4 text-green-600" />;
      case 'pending':
        return <Clock className="w-4 h-4 text-yellow-600" />;
      case 'failed':
        return <XCircle className="w-4 h-4 text-red-600" />;
      case 'cancelled':
        return <AlertCircle className="w-4 h-4 text-gray-600" />;
      default:
        return <Clock className="w-4 h-4 text-gray-600" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      case 'cancelled':
        return 'bg-gray-100 text-gray-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getTransactionTypeLabel = (type: string) => {
    switch (type) {
      case 'credit':
      case 'recharge_cinetpay':
        return 'Recharge';
      case 'debit':
      case 'nfc_withdrawal':
        return 'Retrait';
      case 'transfer':
      case 'mobile_transfer':
        return 'Transfert';
      case 'mobile_money_transfer':
        return 'Mobile Money';
      case 'nfc_deposit':
        return 'Dépôt NFC';
      case 'loan':
      case 'pret_mobile':
        return 'Prêt';
      case 'repayment':
      case 'remboursement_auto':
        return 'Remboursement';
      default:
        return type.replace('_', ' ').toUpperCase();
    }
  };

  const formatAmount = (amount: number, type: string) => {
    const isPositive = amount > 0 || type.includes('recharge') || type.includes('deposit') || type.includes('loan');
    const sign = isPositive ? '+' : '';
    return `${sign} ${Math.abs(amount).toLocaleString('fr-FR')} FCFA`;
  };

  const filteredTransactions = transactions.filter(transaction => {
    const matchesSearch = (transaction.description || '').toLowerCase().includes(searchTerm.toLowerCase()) ||
                         transaction.reference.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         (transaction.user_email || '').toLowerCase().includes(searchTerm.toLowerCase());
    const matchesFilter = selectedFilter === 'all' || transaction.transaction_type === selectedFilter || transaction.status === selectedFilter;
    return matchesSearch && matchesFilter;
  });

  const totalAmount = filteredTransactions.reduce((sum, t) => {
    if (t.transaction_type.includes('recharge') || t.transaction_type.includes('deposit')) return sum + Math.abs(t.amount);
    if (t.transaction_type.includes('withdrawal') || t.transaction_type.includes('transfer')) return sum - Math.abs(t.amount);
    return sum;
  }, 0);

  return (
    <div className="space-y-8">
      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-amber-900">💸 Flux Financiers</h1>
            <p className="text-amber-700 mt-2">
              {isAdmin 
                ? `Surveillez toutes les transactions de la plateforme TIM CASH (${total} transactions)` 
                : 'Suivez l\'historique de vos transactions'
              }
            </p>
          </div>
          <div className="flex space-x-4">
            <button 
              onClick={refetch}
              disabled={loading}
              className="bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg font-medium transition-all duration-200 shadow-md hover:shadow-lg transform hover:scale-105 flex items-center space-x-2 disabled:opacity-50"
            >
              <Download className="w-4 h-4" />
              <span>{loading ? 'Chargement...' : 'Actualiser'}</span>
            </button>
          </div>
        </div>
      </div>

      {/* Stats Cards avec thème marron doré */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-lg border border-amber-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-amber-700">Total Transactions</p>
              <p className="text-3xl font-bold text-amber-900">{filteredTransactions.length}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg flex items-center justify-center shadow-md">
              <DollarSign className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-lg border border-green-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-green-700">Montant Net</p>
              <p className={`text-3xl font-bold ${totalAmount >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                {totalAmount >= 0 ? '+' : ''}{totalAmount.toLocaleString()} FCFA
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-green-400 to-emerald-500 rounded-lg flex items-center justify-center shadow-md">
              <ArrowUpRight className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-yellow-50 to-amber-50 rounded-xl shadow-lg border border-yellow-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-yellow-700">En Attente</p>
              <p className="text-3xl font-bold text-yellow-900">
                {filteredTransactions.filter(t => t.status === 'pending').length}
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-yellow-400 to-amber-500 rounded-lg flex items-center justify-center shadow-md">
              <Clock className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-xl shadow-lg border border-red-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-red-700">Échecs</p>
              <p className="text-3xl font-bold text-red-900">
                {filteredTransactions.filter(t => t.status === 'failed').length}
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-red-400 to-rose-500 rounded-lg flex items-center justify-center shadow-md">
              <XCircle className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Filters and Search avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/50 to-orange-50/50 rounded-xl shadow-lg border border-amber-200 p-6">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
          <div className="flex flex-col sm:flex-row space-y-4 sm:space-y-0 sm:space-x-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
              <input
                type="text"
                placeholder="Rechercher par description ou référence..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 pr-4 py-2 w-full sm:w-80 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent bg-white"
              />
            </div>
            
            <select
              value={selectedFilter}
              onChange={(e) => setSelectedFilter(e.target.value)}
              className="px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent bg-white"
            >
              <option value="all">Tous les types</option>
              <option value="credit">Crédits</option>
              <option value="debit">Débits</option>
              <option value="transfer">Transferts</option>
              <option value="loan">Prêts</option>
              <option value="repayment">Remboursements</option>
              <option value="completed">Terminées</option>
              <option value="pending">En attente</option>
              <option value="failed">Échouées</option>
            </select>

            <select
              value={dateRange}
              onChange={(e) => setDateRange(e.target.value)}
              className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="7">7 derniers jours</option>
              <option value="30">30 derniers jours</option>
              <option value="90">90 derniers jours</option>
              <option value="365">Cette année</option>
              <option value="all">Toutes les dates</option>
            </select>
          </div>

          <div className="flex items-center space-x-4">
            <span className="text-sm text-gray-600">
              {filteredTransactions.length} transaction(s) trouvée(s)
            </span>
            <button className="flex items-center space-x-2 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors duration-200">
              <Filter className="w-4 h-4" />
              <span>Plus de filtres</span>
            </button>
          </div>
        </div>
      </div>

      {/* Transactions List */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">Historique des Transactions</h3>
        </div>
        
        <div className="divide-y divide-gray-200">
          {filteredTransactions.map((transaction) => (
            <div key={transaction.id} className="p-6 hover:bg-gray-50 transition-colors duration-150">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <div className="w-12 h-12 bg-gray-100 rounded-lg flex items-center justify-center">
                    {getTransactionIcon(transaction.transaction_type)}
                  </div>
                  <div className="flex-1">
                    <div className="flex items-center space-x-3 mb-1">
                      <h4 className="font-medium text-gray-900">
                        {getTransactionTypeLabel(transaction.transaction_type)}
                      </h4>
                      <span className={`inline-flex items-center px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(transaction.status)}`}>
                        {getStatusIcon(transaction.status)}
                        <span className="ml-1">
                          {transaction.status === 'completed' ? 'Terminé' :
                           transaction.status === 'pending' ? 'En attente' :
                           transaction.status === 'failed' ? 'Échoué' : transaction.status}
                        </span>
                      </span>
                    </div>
                    <p className="text-sm text-gray-600 mb-1">{transaction.description}</p>
                    <div className="flex items-center space-x-4 text-xs text-gray-500">
                      <span>Réf: {transaction.reference}</span>
                      <span>•</span>
                      <span>{new Date(transaction.created_at).toLocaleDateString('fr-FR')} à {new Date(transaction.created_at).toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' })}</span>
                      {transaction.commission_amount > 0 && (
                        <>
                          <span>•</span>
                          <span>Frais: {transaction.commission_amount} FCFA</span>
                        </>
                      )}
                    </div>
                    {isAdmin && (transaction.user_email || transaction.user_name) && (
                      <div className="flex items-center space-x-2 mt-2 text-xs text-gray-500">
                        <span>Utilisateur: <span className="font-medium">{transaction.user_name || transaction.user_email}</span></span>
                        {transaction.user_email && transaction.user_name && (
                          <span>({transaction.user_email})</span>
                        )}
                      </div>
                    )}
                  </div>
                </div>
                
                <div className="flex items-center space-x-4">
                  <div className="text-right">
                    <p className={`text-lg font-semibold ${
                      transaction.transaction_type.includes('recharge') || transaction.transaction_type.includes('deposit') || transaction.amount > 0
                        ? 'text-green-600' 
                        : 'text-red-600'
                    }`}>
                      {formatAmount(transaction.amount, transaction.transaction_type)}
                    </p>
                    {transaction.commission_amount > 0 && (
                      <p className="text-sm text-gray-500">
                        Frais: -{transaction.commission_amount} FCFA
                      </p>
                    )}
                  </div>
                  
                  <div className="flex items-center space-x-2">
                    <button 
                      onClick={() => setSelectedTransaction(selectedTransaction === transaction.id ? null : transaction.id)}
                      className="p-2 text-gray-400 hover:text-gray-600 hover:bg-gray-100 rounded-lg transition-colors duration-200"
                    >
                      <Eye className="w-4 h-4" />
                    </button>
                    <button className="p-2 text-gray-400 hover:text-gray-600 hover:bg-gray-100 rounded-lg transition-colors duration-200">
                      <MoreHorizontal className="w-4 h-4" />
                    </button>
                  </div>
                </div>
              </div>

              {/* Transaction Details */}
              {selectedTransaction === transaction.id && (
                <div className="mt-4 pt-4 border-t border-gray-200">
                  <div className="bg-gray-50 rounded-lg p-4">
                    <h5 className="font-medium text-gray-900 mb-3">Détails de la Transaction</h5>
                    <div className="grid grid-cols-2 md:grid-cols-3 gap-4 text-sm">
                      <div>
                        <p className="text-gray-600">ID Transaction</p>
                        <p className="font-medium">{transaction.id}</p>
                      </div>
                      <div>
                        <p className="text-gray-600">Type</p>
                        <p className="font-medium">{getTransactionTypeLabel(transaction.transaction_type)}</p>
                      </div>
                      <div>
                        <p className="text-gray-600">Statut</p>
                        <p className="font-medium">
                          {transaction.status === 'completed' ? 'Terminé' :
                           transaction.status === 'pending' ? 'En attente' :
                           transaction.status === 'failed' ? 'Échoué' : transaction.status}
                        </p>
                      </div>
                      <div>
                        <p className="text-gray-600">Montant</p>
                        <p className="font-medium">{transaction.amount.toLocaleString()} FCFA</p>
                      </div>
                      <div>
                        <p className="text-gray-600">Frais</p>
                        <p className="font-medium">{transaction.commission_amount} FCFA</p>
                      </div>
                      <div>
                        <p className="text-gray-600">Date</p>
                        <p className="font-medium">{new Date(transaction.created_at).toLocaleString('fr-FR')}</p>
                      </div>
                    </div>
                    {transaction.status === 'pending' && (
                      <div className="mt-4 flex space-x-2">
                        <button className="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors duration-200">
                          Approuver
                        </button>
                        <button className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors duration-200">
                          Annuler
                        </button>
                      </div>
                    )}
                  </div>
                </div>
              )}
            </div>
          ))}
        </div>

        {filteredTransactions.length === 0 && (
          <div className="p-12 text-center">
            <DollarSign className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">Aucune transaction trouvée</h3>
            <p className="text-gray-600">
              Aucune transaction ne correspond à vos critères de recherche.
            </p>
          </div>
        )}
      </div>
    </div>
  );
};

export default Transactions;