import React, { useState, useEffect } from 'react';
import {
  DollarSign,
  TrendingUp,
  Users,
  ArrowUpRight,
  Calendar,
  Download,
  RefreshCw,
  Phone,
  CheckCircle,
  Clock,
  XCircle,
  Search
} from 'lucide-react';
import { getTIMBusinessRecharges } from '../../services/api';

interface Recharge {
  id: string;
  reference: string;
  amount: number;
  total_commission: number;
  business_commission: number;
  timcash_commission: number;
  total_amount: number;
  status: string;
  beneficiary_phone: string;
  beneficiary_account_type: string;
  business_user_id: string;
  business_user_name?: string;
  created_at: string;
  completed_at: string | null;
  description?: string;
}

interface Stats {
  total_recharges: number;
  total_amount: number;
  total_commissions_business: number;
  total_commissions_timcash: number;
  today_recharges: number;
  today_amount: number;
}

const TIMBusinessRecharges: React.FC = () => {
  const [recharges, setRecharges] = useState<Recharge[]>([]);
  const [stats, setStats] = useState<Stats>({
    total_recharges: 0,
    total_amount: 0,
    total_commissions_business: 0,
    total_commissions_timcash: 0,
    today_recharges: 0,
    today_amount: 0
  });
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterStatus, setFilterStatus] = useState('all');
  const [dateFilter, setDateFilter] = useState('all');

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    setLoading(true);
    setError(null);
    try {
      const response = await getTIMBusinessRecharges();

      // Transformer les données de l'API au format attendu
      const rechargesData = response.recharges.map((recharge: any) => ({
        id: recharge.id,
        reference: recharge.reference,
        amount: recharge.amount,
        total_commission: recharge.total_commission,
        business_commission: recharge.business_commission,
        timcash_commission: recharge.timcash_commission,
        total_amount: recharge.total_amount,
        status: recharge.status,
        beneficiary_phone: recharge.beneficiary_phone,
        beneficiary_account_type: recharge.beneficiary_account_type,
        business_user_id: recharge.business_user_id,
        business_user_name: recharge.business_user_name,
        created_at: recharge.created_at,
        completed_at: recharge.completed_at,
        description: recharge.description
      }));

      setRecharges(rechargesData);
      setStats(response.stats);
    } catch (err: any) {
      console.error('Erreur lors du chargement des données:', err);
      setError(err.message || 'Erreur lors du chargement des recharges TIM BUSINESS');
    } finally {
      setLoading(false);
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="w-5 h-5 text-green-600" />;
      case 'pending':
        return <Clock className="w-5 h-5 text-yellow-600" />;
      case 'failed':
        return <XCircle className="w-5 h-5 text-red-600" />;
      default:
        return <Clock className="w-5 h-5 text-gray-600" />;
    }
  };

  const getStatusBadge = (status: string) => {
    const styles = {
      completed: 'bg-green-100 text-green-800',
      pending: 'bg-yellow-100 text-yellow-800',
      failed: 'bg-red-100 text-red-800',
      processing: 'bg-blue-100 text-blue-800'
    };

    const labels = {
      completed: 'Complétée',
      pending: 'En attente',
      failed: 'Échouée',
      processing: 'En cours'
    };

    return (
      <span className={`inline-flex px-3 py-1 text-xs font-medium rounded-full ${styles[status as keyof typeof styles] || 'bg-gray-100 text-gray-800'}`}>
        {labels[status as keyof typeof labels] || status}
      </span>
    );
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleString('fr-FR', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const formatAmount = (amount: number) => {
    return amount.toLocaleString('fr-FR') + ' FCFA';
  };

  const filteredRecharges = recharges.filter(recharge => {
    const matchesSearch = 
      recharge.reference.toLowerCase().includes(searchTerm.toLowerCase()) ||
      recharge.beneficiary_phone.includes(searchTerm);
    
    const matchesStatus = filterStatus === 'all' || recharge.status === filterStatus;
    
    return matchesSearch && matchesStatus;
  });

  return (
    <div className="space-y-8">
      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-amber-900">💼 Recharges TIM BUSINESS</h1>
            <p className="text-amber-700 mt-2">
              Historique et statistiques des recharges avec commissions
            </p>
          </div>
          <div className="flex space-x-4">
            <button
              onClick={loadData}
              className="border border-amber-300 text-amber-700 hover:bg-amber-50 px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2"
            >
              <RefreshCw className="w-4 h-4" />
              <span>Actualiser</span>
            </button>
            <button className="bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md hover:shadow-lg transform hover:scale-105 flex items-center space-x-2">
              <Download className="w-4 h-4" />
              <span>Exporter</span>
            </button>
          </div>
        </div>
      </div>

      {/* Statistiques */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {/* Total Recharges */}
        <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Total Recharges</p>
              <p className="text-2xl font-bold text-amber-900 mt-2">{stats.total_recharges}</p>
              <p className="text-sm text-gray-500 mt-1">{formatAmount(stats.total_amount)}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-lg flex items-center justify-center">
              <Users className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        {/* Commission TIM BUSINESS */}
        <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Commission TIM BUSINESS (4%)</p>
              <p className="text-2xl font-bold text-green-600 mt-2">{formatAmount(stats.total_commissions_business)}</p>
              <p className="text-sm text-gray-500 mt-1">Créditée sur wallets</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-green-500 to-emerald-600 rounded-lg flex items-center justify-center">
              <ArrowUpRight className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        {/* Commission TIM CASH */}
        <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Commission TIM CASH (6%)</p>
              <p className="text-2xl font-bold text-orange-600 mt-2">{formatAmount(stats.total_commissions_timcash)}</p>
              <p className="text-sm text-gray-500 mt-1">Reversée sur CinetPay</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-orange-500 to-red-600 rounded-lg flex items-center justify-center">
              <DollarSign className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        {/* Aujourd'hui */}
        <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Aujourd'hui</p>
              <p className="text-2xl font-bold text-amber-900 mt-2">{stats.today_recharges}</p>
              <p className="text-sm text-gray-500 mt-1">{formatAmount(stats.today_amount)}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-amber-500 to-orange-600 rounded-lg flex items-center justify-center">
              <TrendingUp className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Filtres et recherche */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          {/* Recherche */}
          <div className="relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              type="text"
              placeholder="Rechercher par référence ou téléphone..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="w-full pl-10 pr-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
            />
          </div>

          {/* Filtre statut */}
          <select
            value={filterStatus}
            onChange={(e) => setFilterStatus(e.target.value)}
            className="px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
          >
            <option value="all">Tous les statuts</option>
            <option value="completed">Complétées</option>
            <option value="pending">En attente</option>
            <option value="failed">Échouées</option>
          </select>

          {/* Filtre date */}
          <select
            value={dateFilter}
            onChange={(e) => setDateFilter(e.target.value)}
            className="px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
          >
            <option value="all">Toutes les dates</option>
            <option value="today">Aujourd'hui</option>
            <option value="week">Cette semaine</option>
            <option value="month">Ce mois</option>
          </select>
        </div>
      </div>

      {/* Liste des recharges */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200">
        <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50 to-orange-50">
          <h3 className="text-lg font-semibold text-amber-900">Historique des recharges</h3>
          <p className="text-sm text-amber-700 mt-1">{filteredRecharges.length} transaction(s)</p>
        </div>

        {loading ? (
          <div className="p-12 text-center">
            <RefreshCw className="w-8 h-8 text-amber-600 animate-spin mx-auto" />
            <p className="text-gray-600 mt-4">Chargement des recharges TIM BUSINESS...</p>
          </div>
        ) : error ? (
          <div className="p-12 text-center">
            <p className="text-red-600 font-medium mb-4">{error}</p>
            <button
              onClick={loadData}
              className="px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700 transition-colors"
            >
              Réessayer
            </button>
          </div>
        ) : filteredRecharges.length === 0 ? (
          <div className="p-12 text-center">
            <div className="w-16 h-16 bg-amber-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <DollarSign className="w-8 h-8 text-amber-600" />
            </div>
            <p className="text-gray-600">
              {searchTerm || filterStatus !== 'all' || dateFilter !== 'all'
                ? 'Aucune recharge ne correspond à vos critères de recherche'
                : 'Aucune recharge TIM BUSINESS trouvée'}
            </p>
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50 border-b border-gray-200">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Référence
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Bénéficiaire
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Montant
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Commissions
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Statut
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Date
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {filteredRecharges.map((recharge) => (
                  <tr key={recharge.id} className="hover:bg-amber-50/30 transition-colors">
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center">
                        {getStatusIcon(recharge.status)}
                        <div className="ml-3">
                          <p className="text-sm font-medium text-gray-900">{recharge.reference}</p>
                          {recharge.description && (
                            <p className="text-xs text-gray-500">{recharge.description}</p>
                          )}
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center">
                        <Phone className="w-4 h-4 text-gray-400 mr-2" />
                        <div>
                          <p className="text-sm text-gray-900">{recharge.beneficiary_phone}</p>
                          <p className="text-xs text-gray-500">{recharge.beneficiary_account_type}</p>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div>
                        <p className="text-sm font-semibold text-gray-900">{formatAmount(recharge.amount)}</p>
                        <p className="text-xs text-gray-500">Total: {formatAmount(recharge.total_amount)}</p>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="space-y-1">
                        <div className="flex items-center text-xs">
                          <span className="text-green-600 font-medium">TIM BUSINESS (4%):</span>
                          <span className="ml-2 text-gray-900">{formatAmount(recharge.business_commission)}</span>
                        </div>
                        <div className="flex items-center text-xs">
                          <span className="text-orange-600 font-medium">TIM CASH (6%):</span>
                          <span className="ml-2 text-gray-900">{formatAmount(recharge.timcash_commission)}</span>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      {getStatusBadge(recharge.status)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center text-sm text-gray-500">
                        <Calendar className="w-4 h-4 mr-2" />
                        {formatDate(recharge.created_at)}
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>

      {/* Informations sur les commissions */}
      <div className="bg-gradient-to-r from-amber-50 to-orange-50 rounded-xl shadow-lg border border-amber-200 p-6">
        <h3 className="text-lg font-semibold text-amber-900 mb-4">ℹ️ Répartition des commissions</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="bg-white rounded-lg p-4 border border-amber-200">
            <div className="flex items-center space-x-3 mb-3">
              <div className="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center">
                <ArrowUpRight className="w-5 h-5 text-green-600" />
              </div>
              <div>
                <p className="font-semibold text-gray-900">Commission TIM BUSINESS</p>
                <p className="text-sm text-gray-600">4% du montant rechargé</p>
              </div>
            </div>
            <p className="text-sm text-gray-700">
              Cette commission est automatiquement créditée sur le wallet du compte TIM BUSINESS qui effectue la recharge.
            </p>
          </div>

          <div className="bg-white rounded-lg p-4 border border-amber-200">
            <div className="flex items-center space-x-3 mb-3">
              <div className="w-10 h-10 bg-orange-100 rounded-lg flex items-center justify-center">
                <DollarSign className="w-5 h-5 text-orange-600" />
              </div>
              <div>
                <p className="font-semibold text-gray-900">Commission TIM CASH</p>
                <p className="text-sm text-gray-600">6% du montant rechargé</p>
              </div>
            </div>
            <p className="text-sm text-gray-700">
              Cette commission est reversée sur le compte de l'entreprise TIM CASH et transférée vers CinetPay.
            </p>
          </div>
        </div>

        <div className="mt-6 bg-white rounded-lg p-4 border border-amber-200">
          <p className="text-sm font-medium text-gray-900 mb-2">Exemple de transaction :</p>
          <div className="space-y-1 text-sm text-gray-700">
            <p>• Recharge de <strong>10 000 FCFA</strong></p>
            <p>• Commission totale : <strong>1 000 FCFA</strong> (10%)</p>
            <p className="ml-4">├─ TIM BUSINESS reçoit : <strong className="text-green-600">400 FCFA</strong> (4%)</p>
            <p className="ml-4">└─ TIM CASH reçoit : <strong className="text-orange-600">600 FCFA</strong> (6%)</p>
            <p>• Total débité du compte TIM BUSINESS : <strong>11 000 FCFA</strong></p>
            <p>• Bénéficiaire reçoit : <strong>10 000 FCFA</strong></p>
          </div>
        </div>
      </div>
    </div>
  );
};

export default TIMBusinessRecharges;
