import React, { useState, useEffect } from "react";
import {
  Wallet,
  Plus,
  Minus,
  TrendingUp,
  TrendingDown,
  Users,
  CreditCard,
  AlertTriangle,
  CheckCircle,
  Clock,
  DollarSign,
  ArrowUpRight,
  ArrowDownLeft,
  History,
  Settings,
  Banknote,
  RefreshCw,
} from "lucide-react";
import { apiService } from "../../services/apiService";

interface StockTransaction {
  id: string;
  type:
    | "credit"
    | "debit"
    | "loan_disbursement"
    | "loan_repayment"
    | "commission";
  amount: number;
  description: string;
  userId?: string;
  userName?: string;
  userType?: "tim_mini" | "tim_maxi" | "tim_business";
  status: "completed" | "pending" | "failed";
  createdAt: string;
  reference: string;
}

interface LoanStats {
  totalLoansActive: number;
  totalAmountLent: number;
  totalAmountRepaid: number;
  overdueLoans: number;
  commissionsEarned: number;
}

const StockAccount: React.FC = () => {
  const [showAddFundsModal, setShowAddFundsModal] = useState(false);
  const [showWithdrawModal, setShowWithdrawModal] = useState(false);
  const [showResetModal, setShowResetModal] = useState(false);
  const [showSetBalanceModal, setShowSetBalanceModal] = useState(false);
  const [selectedPeriod, setSelectedPeriod] = useState("today");
  const [amount, setAmount] = useState("");
  const [description, setDescription] = useState("");
  const [resetBalance, setResetBalance] = useState("0");
  const [newBalance, setNewBalance] = useState("");

  // État du compte stock
  const [stockBalance, setStockBalance] = useState(0);
  const [reserveBalance, setReserveBalance] = useState(0);
  const [totalCredits, setTotalCredits] = useState(0);
  const [totalDebits, setTotalDebits] = useState(0);
  const [totalCommissions, setTotalCommissions] = useState(0);
  const [loading, setLoading] = useState(true);

  // Statistiques des prêts
  const [loanStats, setLoanStats] = useState<LoanStats>({
    totalLoansActive: 0,
    totalAmountLent: 0,
    totalAmountRepaid: 0,
    overdueLoans: 0,
    commissionsEarned: 0,
  });

  // Historique des transactions du compte stock
  const [stockTransactions, setStockTransactions] = useState<
    StockTransaction[]
  >([]);

  useEffect(() => {
    fetchStockData();
    fetchTransactions();
  }, []);

  const fetchStockData = async () => {
    try {
      setLoading(true);

      // Vérifier si l'utilisateur est authentifié
      if (!apiService.isAuthenticated()) {
        console.error("Non authentifié - redirection vers login");
        window.location.href = "/login";
        return;
      }

      // Ajouter un timestamp pour éviter le cache
      const data = await apiService.get(`/admin/stock?_t=${Date.now()}`);
      setStockBalance(data.balance || 0);
      setTotalCredits(data.total_credits || 0);
      setTotalDebits(data.total_debits || 0);
      setTotalCommissions(data.total_commissions || 0);

      console.log("✅ Stock rechargé:", {
        balance: data.balance,
        total_credits: data.total_credits,
        total_debits: data.total_debits,
        total_commissions: data.total_commissions,
      });
    } catch (error: any) {
      console.error("Erreur chargement stock:", error);

      // Si erreur 403, l'utilisateur n'est pas admin
      if (
        error.message?.includes("Forbidden") ||
        error.message?.includes("Admin access required")
      ) {
        alert(
          "Accès refusé. Vous devez être administrateur pour accéder à cette page."
        );
        window.location.href = "/dashboard";
      } else if (
        error.message?.includes("Not authenticated") ||
        error.message?.includes("Unauthorized")
      ) {
        alert("Session expirée. Veuillez vous reconnecter.");
        window.location.href = "/login";
      }
    } finally {
      setLoading(false);
    }
  };

  const fetchTransactions = async () => {
    try {
      const data = await apiService.getTransactions();
      const stockTxns = (data.transactions || [])
        .filter(
          (t: any) =>
            t.description?.includes("STOCK") || t.reference?.includes("STOCK")
        )
        .map((t: any) => ({
          id: t.id,
          type: (t.amount > 0 ? "credit" : "debit") as "credit" | "debit",
          amount: Math.abs(t.amount),
          description: t.description,
          status: t.status as "completed" | "pending" | "failed",
          createdAt: t.created_at,
          reference: t.reference,
        }));
      setStockTransactions(stockTxns);

      console.log(
        "✅ Transactions rechargées:",
        stockTxns.length,
        "transactions trouvées"
      );
    } catch (error) {
      console.error("Erreur chargement transactions:", error);
    }
  };

  const handleAddFunds = async () => {
    if (!amount || !description) return;

    try {
      setLoading(true);
      const response = await apiService.post("/admin/stock/update", {
        amount: parseFloat(amount),
        type: "credit",
        description: description,
      });

      console.log("Ajout de fonds réussi:", response);

      // Rafraîchir les données
      await fetchStockData();
      await fetchTransactions();

      setShowAddFundsModal(false);
      setAmount("");
      setDescription("");
      alert("Fonds ajoutés avec succès!");
    } catch (error: any) {
      console.error("Erreur ajout de fonds:", error);
      alert(
        "Erreur lors de l'ajout de fonds: " +
          (error.message || "Erreur inconnue")
      );
    } finally {
      setLoading(false);
    }
  };

  const handleWithdrawFunds = async () => {
    if (!amount || !description) return;

    try {
      setLoading(true);
      const response = await apiService.post("/admin/stock/update", {
        amount: parseFloat(amount),
        type: "debit",
        description: description,
      });

      console.log("Retrait réussi:", response);

      // Rafraîchir les données
      await fetchStockData();
      await fetchTransactions();

      setShowWithdrawModal(false);
      setAmount("");
      setDescription("");
      alert("Retrait effectué avec succès!");
    } catch (error: any) {
      console.error("Erreur retrait:", error);
      alert("Erreur lors du retrait: " + (error.message || "Erreur inconnue"));
    } finally {
      setLoading(false);
    }
  };

  const handleResetStock = async () => {
    const confirmation = window.confirm(
      "⚠️ ATTENTION: Cette action va réinitialiser TOUS les compteurs du stock admin!\n\n" +
        "Cela va mettre à zéro:\n" +
        "- Total crédits\n" +
        "- Total débits\n" +
        "- Total commissions\n\n" +
        "Le solde sera défini à: " +
        parseFloat(resetBalance).toLocaleString() +
        " FCFA\n\n" +
        "Êtes-vous ABSOLUMENT SÛR de vouloir continuer?"
    );

    if (!confirmation) return;

    try {
      setLoading(true);
      const response = await apiService.post("/admin/stock/reset", {
        balance: parseFloat(resetBalance),
        confirm: true,
      });

      console.log("Stock réinitialisé:", response);

      // Rafraîchir les données
      await fetchStockData();
      await fetchTransactions();

      setShowResetModal(false);
      setResetBalance("0");
      alert("✅ Stock réinitialisé avec succès!");
    } catch (error: any) {
      console.error("Erreur réinitialisation:", error);
      alert(
        "Erreur lors de la réinitialisation: " +
          (error.message || "Erreur inconnue")
      );
    } finally {
      setLoading(false);
    }
  };

  const handleSetBalance = async () => {
    if (!newBalance || !description) return;

    try {
      setLoading(true);
      const response = await apiService.post("/admin/stock/set-balance", {
        balance: parseFloat(newBalance),
        description: description,
      });

      console.log("Solde défini:", response);

      // Rafraîchir les données
      await fetchStockData();
      await fetchTransactions();

      setShowSetBalanceModal(false);
      setNewBalance("");
      setDescription("");
      alert(
        `✅ Solde défini avec succès!\nAncien: ${response.old_balance?.toLocaleString()} FCFA\nNouveau: ${response.new_balance?.toLocaleString()} FCFA`
      );
    } catch (error: any) {
      console.error("Erreur définition solde:", error);
      alert(
        "Erreur lors de la définition du solde: " +
          (error.message || "Erreur inconnue")
      );
    } finally {
      setLoading(false);
    }
  };

  const getTransactionIcon = (type: string) => {
    switch (type) {
      case "credit":
        return <ArrowUpRight className="w-4 h-4 text-green-600" />;
      case "debit":
        return <ArrowDownLeft className="w-4 h-4 text-red-600" />;
      case "loan_disbursement":
        return <CreditCard className="w-4 h-4 text-blue-600" />;
      case "loan_repayment":
        return <CheckCircle className="w-4 h-4 text-green-600" />;
      case "commission":
        return <DollarSign className="w-4 h-4 text-orange-600" />;
      default:
        return <History className="w-4 h-4 text-gray-600" />;
    }
  };

  const getTransactionColor = (type: string) => {
    switch (type) {
      case "credit":
        return "text-green-600";
      case "debit":
        return "text-red-600";
      case "loan_disbursement":
        return "text-blue-600";
      case "loan_repayment":
        return "text-green-600";
      case "commission":
        return "text-orange-600";
      default:
        return "text-gray-600";
    }
  };

  const getUserTypeLabel = (userType?: string) => {
    switch (userType) {
      case "tim_mini":
        return "TIM MINI";
      case "tim_maxi":
        return "TIM MAXI";
      case "tim_business":
        return "TIM BUSINESS";
      default:
        return "";
    }
  };

  return (
    <div className="p-6 max-w-7xl mx-auto">
      {/* Header avec thème marron doré */}
      <div className="mb-8 bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-2xl font-bold text-amber-900 mb-2">
              💼 Compte Stock
            </h1>
            <p className="text-amber-700">
              Gestion du fonds de prêts automatiques TIM CASH
            </p>
          </div>
          <div className="flex space-x-3">
            <button
              onClick={async () => {
                await fetchStockData();
                await fetchTransactions();
              }}
              disabled={loading}
              className="flex items-center space-x-2 px-4 py-2 bg-gradient-to-r from-blue-600 to-indigo-600 text-white rounded-lg hover:from-blue-700 hover:to-indigo-700 transition-all shadow-md disabled:opacity-50"
              title="Rafraîchir les données"
            >
              <RefreshCw
                className={`w-4 h-4 ${loading ? "animate-spin" : ""}`}
              />
              <span>Rafraîchir</span>
            </button>
            <button
              onClick={() => setShowSetBalanceModal(true)}
              className="flex items-center space-x-2 px-4 py-2 bg-gradient-to-r from-purple-600 to-indigo-600 text-white rounded-lg hover:from-purple-700 hover:to-indigo-700 transition-all shadow-md"
            >
              <Settings className="w-4 h-4" />
              <span>Définir le solde</span>
            </button>
            <button
              onClick={() => setShowAddFundsModal(true)}
              className="flex items-center space-x-2 px-4 py-2 bg-gradient-to-r from-green-600 to-emerald-600 text-white rounded-lg hover:from-green-700 hover:to-emerald-700 transition-all shadow-md"
            >
              <Plus className="w-4 h-4" />
              <span>Alimenter</span>
            </button>
            <button
              onClick={() => setShowWithdrawModal(true)}
              className="flex items-center space-x-2 px-4 py-2 bg-gradient-to-r from-red-600 to-rose-600 text-white rounded-lg hover:from-red-700 hover:to-rose-700 transition-all shadow-md"
            >
              <Minus className="w-4 h-4" />
              <span>Retirer</span>
            </button>
            <button
              onClick={() => setShowResetModal(true)}
              className="flex items-center space-x-2 px-4 py-2 bg-gradient-to-r from-gray-700 to-gray-900 text-white rounded-lg hover:from-gray-800 hover:to-black transition-all shadow-md border border-red-500"
              title="Réinitialiser le stock (DANGER)"
            >
              <AlertTriangle className="w-4 h-4" />
              <span>Réinitialiser</span>
            </button>
          </div>
        </div>
      </div>

      {/* Soldes */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
        <div className="bg-gradient-to-r from-amber-500 to-orange-600 rounded-xl p-6 text-white shadow-lg">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-amber-100 text-sm">Compte Stock Principal</p>
              <p className="text-3xl font-bold">
                {stockBalance.toLocaleString()} FCFA
              </p>
              <p className="text-amber-100 text-sm mt-1">
                Disponible pour prêts
              </p>
            </div>
            <div className="bg-amber-400 bg-opacity-30 p-3 rounded-lg">
              <Wallet className="w-8 h-8" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-r from-purple-500 to-pink-600 rounded-xl p-6 text-white shadow-lg">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-purple-100 text-sm">Compte Réserve CinetPay</p>
              <p className="text-3xl font-bold">
                {reserveBalance.toLocaleString()} FCFA
              </p>
              <p className="text-purple-100 text-sm mt-1">
                Frais d'inscription sécurisés
              </p>
            </div>
            <div className="bg-purple-400 bg-opacity-30 p-3 rounded-lg">
              <Banknote className="w-8 h-8" />
            </div>
          </div>
        </div>
      </div>

      {/* Statistiques des prêts */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4 mb-8">
        <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-lg p-4 border border-blue-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-blue-400 to-indigo-500 p-2 rounded-lg">
              <CreditCard className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-blue-700 font-medium">Prêts Actifs</p>
              <p className="text-xl font-semibold text-blue-900">
                {loanStats.totalLoansActive}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-lg p-4 border border-green-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-green-400 to-emerald-500 p-2 rounded-lg">
              <TrendingUp className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-green-700 font-medium">
                Montant Prêté
              </p>
              <p className="text-xl font-semibold text-green-900">
                {loanStats.totalAmountLent.toLocaleString()}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-purple-50 to-pink-50 rounded-lg p-4 border border-purple-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-purple-400 to-pink-500 p-2 rounded-lg">
              <CheckCircle className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-purple-700 font-medium">Remboursé</p>
              <p className="text-xl font-semibold text-purple-900">
                {loanStats.totalAmountRepaid.toLocaleString()}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-lg p-4 border border-red-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-red-400 to-rose-500 p-2 rounded-lg">
              <AlertTriangle className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-red-700 font-medium">En Retard</p>
              <p className="text-xl font-semibold text-red-900">
                {loanStats.overdueLoans}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-lg p-4 border border-amber-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-amber-400 to-orange-500 p-2 rounded-lg">
              <DollarSign className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-amber-700 font-medium">Commissions</p>
              <p className="text-xl font-semibold text-amber-900">
                {loanStats.commissionsEarned.toLocaleString()}
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* Historique des transactions avec thème marron doré */}
      <div className="bg-white rounded-lg border border-amber-200 shadow-lg">
        <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50/30 to-orange-50/30">
          <div className="flex items-center justify-between">
            <h2 className="text-lg font-semibold text-amber-900">
              Historique des Transactions
            </h2>
            <div className="flex items-center space-x-3">
              <select
                value={selectedPeriod}
                onChange={(e) => setSelectedPeriod(e.target.value)}
                className="border border-amber-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              >
                <option value="today">Aujourd'hui</option>
                <option value="week">Cette semaine</option>
                <option value="month">Ce mois</option>
                <option value="all">Tout</option>
              </select>
            </div>
          </div>
        </div>

        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gradient-to-r from-amber-50 to-orange-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Transaction
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Utilisateur
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Montant
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Statut
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Date
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Référence
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {stockTransactions.length === 0 ? (
                <tr>
                  <td colSpan={6} className="px-6 py-12 text-center">
                    <div className="inline-flex items-center justify-center w-16 h-16 bg-amber-100 rounded-full mb-4">
                      <History className="w-8 h-8 text-amber-600" />
                    </div>
                    <h3 className="text-lg font-semibold text-gray-900 mb-2">
                      Aucune transaction
                    </h3>
                    <p className="text-gray-600">
                      L'historique des transactions du compte stock apparaîtra
                      ici
                    </p>
                  </td>
                </tr>
              ) : (
                stockTransactions.map((transaction) => (
                  <tr key={transaction.id} className="hover:bg-gray-50">
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center space-x-3">
                        {getTransactionIcon(transaction.type)}
                        <div>
                          <p className="text-sm font-medium text-gray-900">
                            {transaction.description}
                          </p>
                          <p className="text-xs text-gray-500">
                            {transaction.type === "credit"
                              ? "Crédit"
                              : transaction.type === "debit"
                              ? "Débit"
                              : transaction.type === "loan_disbursement"
                              ? "Prêt accordé"
                              : transaction.type === "loan_repayment"
                              ? "Remboursement"
                              : "Commission"}
                          </p>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      {transaction.userName ? (
                        <div>
                          <p className="text-sm font-medium text-gray-900">
                            {transaction.userName}
                          </p>
                          <p className="text-xs text-gray-500">
                            {getUserTypeLabel(transaction.userType)}
                          </p>
                        </div>
                      ) : (
                        <span className="text-sm text-gray-500">Admin</span>
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span
                        className={`text-sm font-medium ${getTransactionColor(
                          transaction.type
                        )}`}
                      >
                        {transaction.type === "credit" ||
                        transaction.type === "loan_repayment" ||
                        transaction.type === "commission"
                          ? "+"
                          : "-"}
                        {transaction.amount.toLocaleString()} FCFA
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span
                        className={`inline-flex items-center px-2 py-1 text-xs font-medium rounded-full ${
                          transaction.status === "completed"
                            ? "bg-green-100 text-green-800"
                            : transaction.status === "pending"
                            ? "bg-yellow-100 text-yellow-800"
                            : "bg-red-100 text-red-800"
                        }`}
                      >
                        {transaction.status === "completed"
                          ? "Terminé"
                          : transaction.status === "pending"
                          ? "En cours"
                          : "Échoué"}
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                      {new Date(transaction.createdAt).toLocaleDateString(
                        "fr-FR",
                        {
                          day: "2-digit",
                          month: "2-digit",
                          year: "numeric",
                          hour: "2-digit",
                          minute: "2-digit",
                        }
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                      {transaction.reference}
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Modal Alimenter */}
      {showAddFundsModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-md">
            <div className="p-6 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">
                Alimenter le Compte Stock
              </h3>
              <p className="text-gray-600">
                Ajouter des fonds pour les prêts automatiques
              </p>
            </div>
            <div className="p-6 space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Montant (FCFA)
                </label>
                <input
                  type="number"
                  value={amount}
                  onChange={(e) => setAmount(e.target.value)}
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-green-500 focus:border-transparent"
                  placeholder="1000000"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Description
                </label>
                <textarea
                  value={description}
                  onChange={(e) => setDescription(e.target.value)}
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-green-500 focus:border-transparent"
                  rows={3}
                  placeholder="Alimentation mensuelle du compte stock"
                />
              </div>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => setShowAddFundsModal(false)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors duration-200"
              >
                Annuler
              </button>
              <button
                onClick={handleAddFunds}
                className="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors duration-200"
              >
                Alimenter
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Modal Retirer */}
      {showWithdrawModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-md">
            <div className="p-6 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">
                Retirer du Compte Stock
              </h3>
              <p className="text-gray-600">Retirer des fonds du compte stock</p>
            </div>
            <div className="p-6 space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Montant (FCFA)
                </label>
                <input
                  type="number"
                  value={amount}
                  onChange={(e) => setAmount(e.target.value)}
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-red-500 focus:border-transparent"
                  placeholder="500000"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Description
                </label>
                <textarea
                  value={description}
                  onChange={(e) => setDescription(e.target.value)}
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-red-500 focus:border-transparent"
                  rows={3}
                  placeholder="Retrait pour maintenance système"
                />
              </div>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => setShowWithdrawModal(false)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors duration-200"
              >
                Annuler
              </button>
              <button
                onClick={handleWithdrawFunds}
                className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors duration-200"
              >
                Retirer
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Modal Définir le solde */}
      {showSetBalanceModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-md">
            <div className="p-6 border-b border-gray-200 bg-gradient-to-r from-purple-50 to-indigo-50">
              <h3 className="text-lg font-semibold text-gray-900 flex items-center space-x-2">
                <Settings className="w-6 h-6 text-purple-600" />
                <span>Définir le solde du compte stock</span>
              </h3>
              <p className="text-gray-600 text-sm mt-1">
                Définir directement le montant exact du solde
              </p>
            </div>
            <div className="p-6 space-y-4">
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-3">
                <p className="text-sm text-blue-800">
                  💡 <strong>Info:</strong> Cette action définit le solde à un
                  montant exact (ne s'ajoute pas au solde actuel).
                </p>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Nouveau solde (FCFA)
                </label>
                <input
                  type="number"
                  value={newBalance}
                  onChange={(e) => setNewBalance(e.target.value)}
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                  placeholder="10000000"
                />
                <p className="text-xs text-gray-500 mt-1">
                  Solde actuel: {stockBalance.toLocaleString()} FCFA
                </p>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Description
                </label>
                <textarea
                  value={description}
                  onChange={(e) => setDescription(e.target.value)}
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                  rows={3}
                  placeholder="Ex: Alimentation initiale du stock pour les prêts"
                />
              </div>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => {
                  setShowSetBalanceModal(false);
                  setNewBalance("");
                  setDescription("");
                }}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors duration-200"
              >
                Annuler
              </button>
              <button
                onClick={handleSetBalance}
                className="px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors duration-200"
              >
                Définir le solde
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Modal Réinitialiser */}
      {showResetModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-md border-4 border-red-500">
            <div className="p-6 border-b border-gray-200 bg-red-50">
              <div className="flex items-center space-x-3">
                <AlertTriangle className="w-8 h-8 text-red-600" />
                <div>
                  <h3 className="text-lg font-semibold text-red-900">
                    ⚠️ DANGER - Réinitialiser le Stock
                  </h3>
                  <p className="text-red-700 text-sm">
                    Cette action est IRRÉVERSIBLE!
                  </p>
                </div>
              </div>
            </div>
            <div className="p-6 space-y-4">
              <div className="bg-yellow-50 border border-yellow-300 rounded-lg p-4">
                <p className="text-sm text-yellow-800 font-medium mb-2">
                  Cette action va:
                </p>
                <ul className="text-sm text-yellow-700 space-y-1 list-disc list-inside">
                  <li>
                    Mettre à zéro tous les compteurs (crédits, débits,
                    commissions)
                  </li>
                  <li>Définir un nouveau solde de départ</li>
                  <li>Supprimer l'historique des compteurs</li>
                </ul>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Nouveau solde de départ (FCFA)
                </label>
                <input
                  type="number"
                  value={resetBalance}
                  onChange={(e) => setResetBalance(e.target.value)}
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-red-500 focus:border-transparent"
                  placeholder="0"
                />
              </div>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3 bg-gray-50">
              <button
                onClick={() => {
                  setShowResetModal(false);
                  setResetBalance("0");
                }}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-100 transition-colors duration-200"
              >
                Annuler
              </button>
              <button
                onClick={handleResetStock}
                className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors duration-200 font-semibold"
              >
                ⚠️ Réinitialiser
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default StockAccount;
