import React, { useState } from 'react';
import { 
  Settings as SettingsIcon, 
  User, 
  Shield, 
  Bell, 
  CreditCard,
  Globe,
  Smartphone,
  Eye,
  EyeOff,
  Save,
  Trash2,
  Download,
  Camera,
  Zap,
  Activity,
  Building2
} from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';

const Settings: React.FC = () => {
  const { user } = useAuth();
  const [activeTab, setActiveTab] = useState('profile');
  const [showCurrentPassword, setShowCurrentPassword] = useState(false);
  const [showNewPassword, setShowNewPassword] = useState(false);

  const [profileData, setProfileData] = useState({
    firstName: user?.firstName || 'Marie',
    lastName: user?.lastName || 'Kouassi',
    email: user?.email || 'marie.kouassi@timcash.ci',
    phone: user?.phone || '+225 07 12 34 56 78',
    dateOfBirth: '1990-05-15',
    address: 'Cocody Riviera 3, Rue des Jardins',
    city: 'Abidjan',
    country: 'Côte d\'Ivoire',
    language: 'fr',
    timAccountType: 'TIM_MAXI',
    accountLimit: 12000000,
    currentBalance: 2847000,
    nfcEnabled: true,
    kycStatus: 'verified'
  });

  const [securityData, setSecurityData] = useState({
    currentPassword: '',
    newPassword: '',
    confirmPassword: '',
    twoFactorEnabled: true,
    loginNotifications: true,
    sessionTimeout: '30',
    pinCode: '****',
    biometricEnabled: true,
    nfcSecurityLevel: 'high',
    transactionLimits: {
      dailyNFC: 500000,
      dailyTransfer: 1000000,
      monthlyLoan: 10000
    }
  });

  const [notificationSettings, setNotificationSettings] = useState({
    emailNotifications: true,
    pushNotifications: true,
    smsNotifications: true,
    nfcPaymentAlerts: true,
    loanSOSReminders: true,
    rechargeConfirmations: true,
    autoRepaymentNotifications: true,
    commissionEarnings: true,
    accountLimitWarnings: true,
    weeklyTIMReports: true,
    marketingOffers: false
  });

  const [paymentSettings, setPaymentSettings] = useState({
    defaultCurrency: 'FCFA',
    autoReload: true,
    reloadThreshold: 50000,
    reloadAmount: 100000,
    preferredRechargeMethod: 'cinetpay',
    autoLoanRepayment: true,
    nfcDailyLimit: 500000,
    businessCommissionRate: 2.0,
    cinetpayCommissionRate: 0.5,
    preferredBank: 'SGCI'
  });

  const isAdmin = user?.role === 'admin' || user?.role === 'manager';

  const tabs = [
    { id: 'profile', label: 'Profil', icon: User },
    { id: 'security', label: 'Sécurité', icon: Shield },
    { id: 'notifications', label: 'Notifications', icon: Bell },
    { id: 'payment', label: 'Paiements', icon: CreditCard },
    { id: 'preferences', label: 'Préférences', icon: SettingsIcon },
    ...(isAdmin ? [{ id: 'admin', label: 'Administration', icon: Globe }] : [])
  ];

  const handleSave = (section: string) => {
    console.log(`Saving ${section} settings`);
    // Implement save functionality
    alert(`Paramètres ${section} sauvegardés avec succès!`);
  };

  const handleProfileImageUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      console.log('Uploading profile image:', file.name);
      // Implement image upload
    }
  };

  return (
    <div className="space-y-8">
      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-amber-900">⚙️ Paramètres</h1>
            <p className="text-amber-700 mt-2">
              Gérez vos préférences et paramètres de compte
            </p>
          </div>
          <div className="flex space-x-4">
            <button className="border border-amber-300 text-amber-700 hover:bg-amber-50 px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2">
              <Download className="w-4 h-4" />
              <span>Exporter Données</span>
            </button>
            <button className="bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md hover:shadow-lg transform hover:scale-105 flex items-center space-x-2">
              <Save className="w-4 h-4" />
              <span>Sauvegarder</span>
            </button>
          </div>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
        {/* Sidebar */}
        <div className="lg:col-span-1">
          <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
            <nav className="space-y-2">
              {tabs.map((tab) => {
                const IconComponent = tab.icon;
                return (
                  <button
                    key={tab.id}
                    onClick={() => setActiveTab(tab.id)}
                    className={`w-full flex items-center space-x-3 px-4 py-3 rounded-lg text-left transition-colors duration-200 ${
                      activeTab === tab.id
                        ? 'bg-gradient-to-r from-amber-50 to-orange-50 text-amber-700 border border-amber-300 font-semibold'
                        : 'text-gray-700 hover:bg-amber-50/30'
                    }`}
                  >
                    <IconComponent className="w-5 h-5" />
                    <span className="font-medium">{tab.label}</span>
                  </button>
                );
              })}
            </nav>
          </div>
        </div>

        {/* Content */}
        <div className="lg:col-span-3">
          <div className="bg-white rounded-xl shadow-lg border border-amber-200">
            {/* Profile Tab */}
            {activeTab === 'profile' && (
              <div className="p-6">
                <div className="flex items-center justify-between mb-6">
                  <h3 className="text-lg font-semibold text-amber-900">Informations du Profil</h3>
                  <button
                    onClick={() => handleSave('profil')}
                    className="bg-gradient-to-r from-amber-600 to-orange-600 hover:from-amber-700 hover:to-orange-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md"
                  >
                    Sauvegarder
                  </button>
                </div>

                <div className="space-y-6">
                  {/* Profile Picture */}
                  <div className="flex items-center space-x-6">
                    <div className="relative">
                      <div className="w-24 h-24 bg-gradient-to-br from-blue-500 to-blue-600 rounded-full flex items-center justify-center">
                        <span className="text-white font-bold text-2xl">
                          {profileData.firstName[0]}{profileData.lastName[0]}
                        </span>
                      </div>
                      <label className="absolute bottom-0 right-0 w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center cursor-pointer hover:bg-blue-700 transition-colors duration-200">
                        <Camera className="w-4 h-4 text-white" />
                        <input
                          type="file"
                          accept="image/*"
                          onChange={handleProfileImageUpload}
                          className="hidden"
                        />
                      </label>
                    </div>
                    <div>
                      <h4 className="font-medium text-gray-900">Photo de Profil</h4>
                      <p className="text-sm text-gray-600">JPG, PNG ou GIF. Max 2MB.</p>
                    </div>
                  </div>

                  {/* Personal Information */}
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Prénom
                      </label>
                      <input
                        type="text"
                        value={profileData.firstName}
                        onChange={(e) => setProfileData({...profileData, firstName: e.target.value})}
                        className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Nom
                      </label>
                      <input
                        type="text"
                        value={profileData.lastName}
                        onChange={(e) => setProfileData({...profileData, lastName: e.target.value})}
                        className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Email
                      </label>
                      <input
                        type="email"
                        value={profileData.email}
                        onChange={(e) => setProfileData({...profileData, email: e.target.value})}
                        className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Téléphone
                      </label>
                      <input
                        type="tel"
                        value={profileData.phone}
                        onChange={(e) => setProfileData({...profileData, phone: e.target.value})}
                        className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Date de Naissance
                      </label>
                      <input
                        type="date"
                        value={profileData.dateOfBirth}
                        onChange={(e) => setProfileData({...profileData, dateOfBirth: e.target.value})}
                        className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Pays
                      </label>
                      <select
                        value={profileData.country}
                        onChange={(e) => setProfileData({...profileData, country: e.target.value})}
                        className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      >
                        <option value="Côte d'Ivoire">Côte d'Ivoire</option>
                        <option value="Sénégal">Sénégal</option>
                        <option value="Mali">Mali</option>
                        <option value="Burkina Faso">Burkina Faso</option>
                        <option value="Ghana">Ghana</option>
                      </select>
                    </div>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Adresse
                    </label>
                    <textarea
                      value={profileData.address}
                      onChange={(e) => setProfileData({...profileData, address: e.target.value})}
                      rows={3}
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      placeholder="Votre adresse complète"
                    />
                  </div>

                  {/* Section TIM CASH */}
                  <div className="border-t border-gray-200 pt-6">
                    <h4 className="font-medium text-gray-900 mb-4 flex items-center space-x-2">
                      <Smartphone className="w-5 h-5 text-blue-600" />
                      <span>Informations TIM CASH</span>
                    </h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Type de Compte
                        </label>
                        <div className="flex items-center space-x-2">
                          <span className={`inline-flex px-3 py-1 text-sm font-medium rounded-full ${
                            profileData.timAccountType === 'TIM_MINI' ? 'bg-blue-100 text-blue-800' :
                            profileData.timAccountType === 'TIM_MAXI' ? 'bg-purple-100 text-purple-800' :
                            'bg-orange-100 text-orange-800'
                          }`}>
                            {profileData.timAccountType}
                          </span>
                          <span className={`inline-flex px-2 py-1 text-xs rounded-full ${
                            profileData.kycStatus === 'verified' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'
                          }`}>
                            {profileData.kycStatus === 'verified' ? 'KYC Vérifié' : 'KYC En attente'}
                          </span>
                        </div>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Solde Actuel
                        </label>
                        <div className="text-lg font-bold text-green-600">
                          {profileData.currentBalance.toLocaleString()} FCFA
                        </div>
                        <div className="text-sm text-gray-500">
                          Limite: {profileData.accountLimit > 0 ? `${profileData.accountLimit.toLocaleString()} FCFA` : 'Illimitée'}
                        </div>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          NFC Activé
                        </label>
                        <div className="flex items-center space-x-2">
                          <button className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                            profileData.nfcEnabled ? 'bg-blue-600' : 'bg-gray-200'
                          }`}>
                            <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                              profileData.nfcEnabled ? 'translate-x-6' : 'translate-x-1'
                            }`} />
                          </button>
                          <Zap className={`w-4 h-4 ${profileData.nfcEnabled ? 'text-blue-600' : 'text-gray-400'}`} />
                        </div>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Progression du Compte
                        </label>
                        <div className="w-full bg-gray-200 rounded-full h-2">
                          <div 
                            className={`h-2 rounded-full ${
                              profileData.timAccountType === 'TIM_MINI' ? 'bg-blue-600' :
                              profileData.timAccountType === 'TIM_MAXI' ? 'bg-purple-600' : 'bg-orange-600'
                            }`}
                            style={{ 
                              width: profileData.accountLimit > 0 
                                ? `${Math.min((profileData.currentBalance / profileData.accountLimit) * 100, 100)}%`
                                : '75%'
                            }}
                          />
                        </div>
                        <div className="text-xs text-gray-500 mt-1">
                          {profileData.accountLimit > 0 
                            ? `${Math.round((profileData.currentBalance / profileData.accountLimit) * 100)}% utilisé`
                            : 'Compte professionnel'
                          }
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Security Tab */}
            {activeTab === 'security' && (
              <div className="p-6">
                <div className="flex items-center justify-between mb-6">
                  <h3 className="text-lg font-semibold text-amber-900">Paramètres de Sécurité</h3>
                  <button
                    onClick={() => handleSave('sécurité')}
                    className="bg-gradient-to-r from-amber-600 to-orange-600 hover:from-amber-700 hover:to-orange-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md"
                  >
                    Sauvegarder
                  </button>
                </div>

                <div className="space-y-8">
                  {/* Change Password */}
                  <div>
                    <h4 className="font-medium text-gray-900 mb-4">Changer le Mot de Passe</h4>
                    <div className="space-y-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Mot de passe actuel
                        </label>
                        <div className="relative">
                          <input
                            type={showCurrentPassword ? 'text' : 'password'}
                            value={securityData.currentPassword}
                            onChange={(e) => setSecurityData({...securityData, currentPassword: e.target.value})}
                            className="w-full px-4 py-2 pr-12 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          />
                          <button
                            type="button"
                            onClick={() => setShowCurrentPassword(!showCurrentPassword)}
                            className="absolute right-4 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
                          >
                            {showCurrentPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
                          </button>
                        </div>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Nouveau mot de passe
                        </label>
                        <div className="relative">
                          <input
                            type={showNewPassword ? 'text' : 'password'}
                            value={securityData.newPassword}
                            onChange={(e) => setSecurityData({...securityData, newPassword: e.target.value})}
                            className="w-full px-4 py-2 pr-12 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          />
                          <button
                            type="button"
                            onClick={() => setShowNewPassword(!showNewPassword)}
                            className="absolute right-4 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
                          >
                            {showNewPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
                          </button>
                        </div>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Confirmer le nouveau mot de passe
                        </label>
                        <input
                          type="password"
                          value={securityData.confirmPassword}
                          onChange={(e) => setSecurityData({...securityData, confirmPassword: e.target.value})}
                          className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                        />
                      </div>
                    </div>
                  </div>

                  {/* Two-Factor Authentication */}
                  <div className="border-t border-gray-200 pt-6">
                    <h4 className="font-medium text-gray-900 mb-4">Authentification à Deux Facteurs</h4>
                    <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                      <div>
                        <p className="font-medium text-gray-900">Activer 2FA</p>
                        <p className="text-sm text-gray-600">Sécurisez votre compte avec une authentification à deux facteurs</p>
                      </div>
                      <button className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                        securityData.twoFactorEnabled ? 'bg-blue-600' : 'bg-gray-200'
                      }`}>
                        <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                          securityData.twoFactorEnabled ? 'translate-x-6' : 'translate-x-1'
                        }`} />
                      </button>
                    </div>
                  </div>

                  {/* Session Settings */}
                  <div className="border-t border-gray-200 pt-6">
                    <h4 className="font-medium text-gray-900 mb-4">Paramètres de Session</h4>
                    <div className="space-y-4">
                      <div className="flex items-center justify-between">
                        <div>
                          <p className="font-medium text-gray-900">Notifications de connexion</p>
                          <p className="text-sm text-gray-600">Recevoir des alertes lors de nouvelles connexions</p>
                        </div>
                        <button className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                          securityData.loginNotifications ? 'bg-blue-600' : 'bg-gray-200'
                        }`}>
                          <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                            securityData.loginNotifications ? 'translate-x-6' : 'translate-x-1'
                          }`} />
                        </button>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Expiration de session (minutes)
                        </label>
                        <select
                          value={securityData.sessionTimeout}
                          onChange={(e) => setSecurityData({...securityData, sessionTimeout: e.target.value})}
                          className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                        >
                          <option value="15">15 minutes</option>
                          <option value="30">30 minutes</option>
                          <option value="60">1 heure</option>
                          <option value="120">2 heures</option>
                        </select>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Notifications Tab */}
            {activeTab === 'notifications' && (
              <div className="p-6">
                <div className="flex items-center justify-between mb-6">
                  <h3 className="text-lg font-semibold text-amber-900">Préférences de Notification</h3>
                  <button
                    onClick={() => handleSave('notifications')}
                    className="bg-gradient-to-r from-amber-600 to-orange-600 hover:from-amber-700 hover:to-orange-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md"
                  >
                    Sauvegarder
                  </button>
                </div>

                <div className="space-y-6">
                  {[
                    { key: 'emailNotifications', label: 'Notifications Email', description: 'Recevoir des notifications par email', icon: '\uD83D\uDCE7' },
                    { key: 'pushNotifications', label: 'Notifications Push', description: 'Notifications push dans le navigateur', icon: '\uD83D\uDD14' },
                    { key: 'smsNotifications', label: 'Notifications SMS', description: 'Alertes par SMS sur votre téléphone', icon: '\uD83D\uDCF1' },
                    { key: 'nfcPaymentAlerts', label: 'Alertes Paiements NFC', description: 'Notifications pour tous les paiements sans contact', icon: '\u26A1' },
                    { key: 'loanSOSReminders', label: 'Rappels Prêts SOS', description: 'Rappels d\'\u00e9chéance pour vos prêts d\'urgence', icon: '\uD83C\uDD98' },
                    { key: 'rechargeConfirmations', label: 'Confirmations Recharge', description: 'Notifications de recharge CinetPay et TIM BUSINESS', icon: '\uD83D\uDD04' },
                    { key: 'autoRepaymentNotifications', label: 'Remboursements Auto', description: 'Notifications de remboursement automatique de prêts', icon: '\uD83D\uDCB0' },
                    { key: 'commissionEarnings', label: 'Gains Commissions', description: 'Notifications des commissions TIM BUSINESS gagnées', icon: '\uD83D\uDCB9' },
                    { key: 'accountLimitWarnings', label: 'Alertes Limites', description: 'Alertes quand vous approchez des limites de compte', icon: '\u26A0\uFE0F' },
                    { key: 'weeklyTIMReports', label: 'Rapports TIM CASH', description: 'Résumé hebdomadaire de votre activité TIM CASH', icon: '\uD83D\uDCC8' },
                    { key: 'marketingOffers', label: 'Offres Promotionnelles', description: 'Offres spéciales et promotions TIM CASH', icon: '\uD83C\uDF81' }
                  ].map((setting) => (
                    <div key={setting.key} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                      <div className="flex items-center space-x-3">
                        <span className="text-2xl">{setting.icon}</span>
                        <div>
                          <p className="font-medium text-gray-900">{setting.label}</p>
                          <p className="text-sm text-gray-600">{setting.description}</p>
                        </div>
                      </div>
                      <button 
                        onClick={() => setNotificationSettings({
                          ...notificationSettings,
                          [setting.key]: !notificationSettings[setting.key as keyof typeof notificationSettings]
                        })}
                        className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                          notificationSettings[setting.key as keyof typeof notificationSettings] ? 'bg-blue-600' : 'bg-gray-200'
                        }`}
                      >
                        <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                          notificationSettings[setting.key as keyof typeof notificationSettings] ? 'translate-x-6' : 'translate-x-1'
                        }`} />
                      </button>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Payment Tab */}
            {activeTab === 'payment' && (
              <div className="p-6">
                <div className="flex items-center justify-between mb-6">
                  <h3 className="text-lg font-semibold text-amber-900">Paramètres de Paiement</h3>
                  <button
                    onClick={() => handleSave('paiement')}
                    className="bg-gradient-to-r from-amber-600 to-orange-600 hover:from-amber-700 hover:to-orange-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md"
                  >
                    Sauvegarder
                  </button>
                </div>

                <div className="space-y-6">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Devise par Défaut
                      </label>
                      <select
                        value={paymentSettings.defaultCurrency}
                        onChange={(e) => setPaymentSettings({...paymentSettings, defaultCurrency: e.target.value})}
                        className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      >
                        <option value="FCFA">FCFA - Franc CFA</option>
                        <option value="USD">USD - Dollar Américain</option>
                        <option value="EUR">EUR - Euro</option>
                      </select>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Méthode de Paiement Préférée
                      </label>
                      <select
                        value={paymentSettings.preferredRechargeMethod}
                        onChange={(e) => setPaymentSettings({...paymentSettings, preferredRechargeMethod: e.target.value})}
                        className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      >
                        <option value="cinetpay">CinetPay (Mobile Money)</option>
                        <option value="tim_business">Recharge TIM BUSINESS</option>
                        <option value="bank_transfer">Virement Bancaire</option>
                        <option value="orange_money">Orange Money</option>
                        <option value="mtn_money">MTN Money</option>
                        <option value="moov_money">Moov Money</option>
                      </select>
                    </div>
                  </div>

                  {/* Rechargement Automatique TIM CASH */}
                  <div className="border-t border-gray-200 pt-6">
                    <h4 className="font-medium text-gray-900 mb-4 flex items-center space-x-2">
                      <Activity className="w-5 h-5 text-green-600" />
                      <span>Rechargement Automatique TIM CASH</span>
                    </h4>
                    <div className="space-y-4">
                      <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                        <div>
                          <p className="font-medium text-gray-900">Recharge automatique CinetPay</p>
                          <p className="text-sm text-gray-600">Recharger automatiquement via CinetPay quand le solde est bas</p>
                        </div>
                        <button className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                          paymentSettings.autoReload ? 'bg-blue-600' : 'bg-gray-200'
                        }`}>
                          <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                            paymentSettings.autoReload ? 'translate-x-6' : 'translate-x-1'
                          }`} />
                        </button>
                      </div>
                      
                      <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                        <div>
                          <p className="font-medium text-gray-900">Remboursement automatique des prêts</p>
                          <p className="text-sm text-gray-600">Déduire automatiquement les prêts lors des recharges</p>
                        </div>
                        <button className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                          paymentSettings.autoLoanRepayment ? 'bg-blue-600' : 'bg-gray-200'
                        }`}>
                          <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                            paymentSettings.autoLoanRepayment ? 'translate-x-6' : 'translate-x-1'
                          }`} />
                        </button>
                      </div>
                      
                      {paymentSettings.autoReload && (
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Seuil de Rechargement (FCFA)
                            </label>
                            <input
                              type="number"
                              value={paymentSettings.reloadThreshold}
                              onChange={(e) => setPaymentSettings({...paymentSettings, reloadThreshold: Number(e.target.value)})}
                              className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                              placeholder="50,000"
                            />
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Montant de Rechargement (FCFA)
                            </label>
                            <input
                              type="number"
                              value={paymentSettings.reloadAmount}
                              onChange={(e) => setPaymentSettings({...paymentSettings, reloadAmount: Number(e.target.value)})}
                              className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                              placeholder="100,000"
                            />
                          </div>
                        </div>
                      )}
                    </div>
                  </div>

                  {/* Limites de Transaction TIM CASH */}
                  <div className="border-t border-gray-200 pt-6">
                    <h4 className="font-medium text-gray-900 mb-4 flex items-center space-x-2">
                      <Zap className="w-5 h-5 text-blue-600" />
                      <span>Limites de Transaction NFC</span>
                    </h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Limite quotidienne NFC (FCFA)
                        </label>
                        <input
                          type="number"
                          value={paymentSettings.nfcDailyLimit}
                          onChange={(e) => setPaymentSettings({...paymentSettings, nfcDailyLimit: Number(e.target.value)})}
                          className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                        />
                        <p className="text-xs text-gray-500 mt-1">Montant maximum pour les paiements NFC par jour</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Banque préférée pour virements
                        </label>
                        <select
                          value={paymentSettings.preferredBank}
                          onChange={(e) => setPaymentSettings({...paymentSettings, preferredBank: e.target.value})}
                          className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                        >
                          <option value="SGCI">SGCI - Société Générale</option>
                          <option value="BICICI">BICICI - BNP Paribas</option>
                          <option value="BACI">BACI - Banque Atlantique</option>
                          <option value="UBA">UBA - United Bank for Africa</option>
                          <option value="ECOBANK">Ecobank Côte d'Ivoire</option>
                        </select>
                      </div>
                    </div>
                  </div>

                  {/* Commissions TIM BUSINESS */}
                  {profileData.timAccountType === 'TIM_BUSINESS' && (
                    <div className="border-t border-gray-200 pt-6">
                      <h4 className="font-medium text-gray-900 mb-4 flex items-center space-x-2">
                        <Building2 className="w-5 h-5 text-orange-600" />
                        <span>Paramètres TIM BUSINESS</span>
                      </h4>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Taux de commission sur recharges (%)
                          </label>
                          <input
                            type="number"
                            step="0.1"
                            value={paymentSettings.businessCommissionRate}
                            onChange={(e) => setPaymentSettings({...paymentSettings, businessCommissionRate: Number(e.target.value)})}
                            className="w-full px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                          />
                          <p className="text-xs text-gray-500 mt-1">Commission gagnée sur les recharges effectuées</p>
                        </div>
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Commission CinetPay (%)
                          </label>
                          <input
                            type="number"
                            step="0.1"
                            value={paymentSettings.cinetpayCommissionRate}
                            readOnly
                            className="w-full px-4 py-2 border border-gray-300 rounded-lg bg-gray-50 text-gray-500"
                          />
                          <p className="text-xs text-gray-500 mt-1">Commission payée sur vos propres recharges</p>
                        </div>
                      </div>
                    </div>
                  )}
                </div>
              </div>
            )}

            {/* Preferences Tab */}
            {activeTab === 'preferences' && (
              <div className="p-6">
                <div className="flex items-center justify-between mb-6">
                  <h3 className="text-lg font-semibold text-gray-900">Préférences Générales</h3>
                  <button
                    onClick={() => handleSave('préférences')}
                    className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium transition-colors duration-200"
                  >
                    Sauvegarder
                  </button>
                </div>

                <div className="space-y-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Langue
                    </label>
                    <select
                      value={profileData.language}
                      onChange={(e) => setProfileData({...profileData, language: e.target.value})}
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    >
                      <option value="fr">Français</option>
                      <option value="en">English</option>
                      <option value="es">Español</option>
                      <option value="de">Deutsch</option>
                    </select>
                  </div>

                  <div className="border-t border-gray-200 pt-6">
                    <h4 className="font-medium text-gray-900 mb-4 text-red-600">Zone de Danger</h4>
                    <div className="space-y-4">
                      <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
                        <h5 className="font-medium text-red-900 mb-2">Supprimer le Compte</h5>
                        <p className="text-sm text-red-700 mb-4">
                          Cette action est irréversible. Toutes vos données seront définitivement supprimées.
                        </p>
                        <button className="bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2">
                          <Trash2 className="w-4 h-4" />
                          <span>Supprimer le Compte</span>
                        </button>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Admin Tab */}
            {activeTab === 'admin' && isAdmin && (
              <div className="p-6">
                <div className="flex items-center justify-between mb-6">
                  <h3 className="text-lg font-semibold text-gray-900">Paramètres d'Administration</h3>
                  <button
                    onClick={() => handleSave('administration')}
                    className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium transition-colors duration-200"
                  >
                    Sauvegarder
                  </button>
                </div>

                <div className="space-y-6">
                  <div className="text-center py-12">
                    <Globe className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                    <h3 className="text-lg font-medium text-gray-900 mb-2">Paramètres d'Administration</h3>
                    <p className="text-gray-600">
                      Configuration avancée de la plateforme, gestion des utilisateurs et paramètres système.
                    </p>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default Settings;