import React, { useState } from 'react';
import {
  FileText,
  Download,
  Calendar,
  Filter,
  BarChart3,
  PieChart,
  TrendingUp,
  DollarSign,
  Users,
  CreditCard,
  Eye,
  RefreshCw
} from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';

const Reports: React.FC = () => {
  const { user } = useAuth();
  const [selectedReport, setSelectedReport] = useState('financial');
  const [dateRange, setDateRange] = useState('30');
  const [reportFormat, setReportFormat] = useState('pdf');
  const [viewingReport, setViewingReport] = useState<any>(null);

  const isAdmin = user?.role === 'admin' || user?.role === 'manager';

  const reportTypes = [
    {
      id: 'financial',
      title: 'Rapport Financier',
      description: 'Vue d\'ensemble des revenus, dépenses et bénéfices',
      icon: DollarSign,
      color: 'bg-green-50 text-green-600',
      available: true
    },
    {
      id: 'transactions',
      title: 'Rapport des Transactions',
      description: 'Analyse détaillée de toutes les transactions',
      icon: BarChart3,
      color: 'bg-blue-50 text-blue-600',
      available: true
    },
    {
      id: 'users',
      title: 'Rapport Utilisateurs',
      description: 'Statistiques d\'activité et d\'engagement des utilisateurs',
      icon: Users,
      color: 'bg-purple-50 text-purple-600',
      available: isAdmin
    },
    {
      id: 'loans',
      title: 'Rapport des Prêts',
      description: 'Performance des prêts et taux de remboursement',
      icon: CreditCard,
      color: 'bg-orange-50 text-orange-600',
      available: true
    },
    {
      id: 'performance',
      title: 'Rapport de Performance',
      description: 'KPIs et métriques de performance de la plateforme',
      icon: TrendingUp,
      color: 'bg-indigo-50 text-indigo-600',
      available: isAdmin
    },
    {
      id: 'compliance',
      title: 'Rapport de Conformité',
      description: 'Audit et conformité réglementaire',
      icon: FileText,
      color: 'bg-gray-50 text-gray-600',
      available: isAdmin
    }
  ];

  const recentReports: any[] = [];

  const quickStats = {
    totalReports: 0,
    thisMonth: 0,
    avgGenerationTime: '0 min',
    totalSize: '0 MB'
  };

  const getReportIcon = (type: string) => {
    const reportType = reportTypes.find(r => r.id === type);
    if (!reportType) return FileText;
    return reportType.icon;
  };

  const generateReport = () => {
    console.log('Generating report:', {
      type: selectedReport,
      dateRange,
      format: reportFormat
    });
    // Simulate report generation
    alert(`Génération du rapport en cours...\nType: ${selectedReport}\nPériode: ${dateRange} jours\nFormat: ${reportFormat.toUpperCase()}`);
  };

  const handleViewReport = (report: any) => {
    console.log('Viewing report:', report);
    setViewingReport(report);
    // Ouvrir le rapport dans un nouvel onglet ou modal
    window.open(`/reports/view/${report.id}`, '_blank');
  };

  const handleDownloadReport = (report: any) => {
    console.log('Downloading report:', report);
    // Simuler le téléchargement
    alert(`Téléchargement du rapport "${report.title}" en cours...`);
  };

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Rapports Financiers</h1>
          <p className="text-gray-600 mt-2">
            Générez et consultez des rapports détaillés sur l'activité de la plateforme
          </p>
        </div>
        <div className="flex space-x-4">
          <button className="border border-gray-300 text-gray-700 hover:bg-gray-50 px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2">
            <RefreshCw className="w-4 h-4" />
            <span>Actualiser</span>
          </button>
          <button 
            onClick={generateReport}
            className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2"
          >
            <FileText className="w-4 h-4" />
            <span>Générer Rapport</span>
          </button>
        </div>
      </div>

      {/* Quick Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Total Rapports</p>
              <p className="text-3xl font-bold text-gray-900">{quickStats.totalReports}</p>
            </div>
            <div className="w-12 h-12 bg-blue-50 rounded-lg flex items-center justify-center">
              <FileText className="w-6 h-6 text-blue-600" />
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Ce Mois</p>
              <p className="text-3xl font-bold text-gray-900">{quickStats.thisMonth}</p>
            </div>
            <div className="w-12 h-12 bg-green-50 rounded-lg flex items-center justify-center">
              <Calendar className="w-6 h-6 text-green-600" />
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Temps Moyen</p>
              <p className="text-3xl font-bold text-gray-900">{quickStats.avgGenerationTime}</p>
            </div>
            <div className="w-12 h-12 bg-purple-50 rounded-lg flex items-center justify-center">
              <TrendingUp className="w-6 h-6 text-purple-600" />
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Taille Totale</p>
              <p className="text-3xl font-bold text-gray-900">{quickStats.totalSize}</p>
            </div>
            <div className="w-12 h-12 bg-orange-50 rounded-lg flex items-center justify-center">
              <Download className="w-6 h-6 text-orange-600" />
            </div>
          </div>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        {/* Report Generator */}
        <div className="lg:col-span-2 space-y-6">
          {/* Report Types */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-6">Types de Rapports</h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {reportTypes.filter(report => report.available).map((report) => {
                const IconComponent = report.icon;
                return (
                  <button
                    key={report.id}
                    onClick={() => setSelectedReport(report.id)}
                    className={`p-4 rounded-lg border-2 transition-all duration-200 text-left ${
                      selectedReport === report.id
                        ? 'border-blue-500 bg-blue-50'
                        : 'border-gray-200 hover:border-gray-300 hover:bg-gray-50'
                    }`}
                  >
                    <div className="flex items-center space-x-3 mb-3">
                      <div className={`w-10 h-10 rounded-lg flex items-center justify-center ${report.color}`}>
                        <IconComponent className="w-5 h-5" />
                      </div>
                      <h4 className="font-medium text-gray-900">{report.title}</h4>
                    </div>
                    <p className="text-sm text-gray-600">{report.description}</p>
                  </button>
                );
              })}
            </div>
          </div>

          {/* Report Configuration */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-6">Configuration du Rapport</h3>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Période
                </label>
                <select
                  value={dateRange}
                  onChange={(e) => setDateRange(e.target.value)}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="7">7 derniers jours</option>
                  <option value="30">30 derniers jours</option>
                  <option value="90">90 derniers jours</option>
                  <option value="365">Cette année</option>
                  <option value="custom">Période personnalisée</option>
                </select>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Format
                </label>
                <select
                  value={reportFormat}
                  onChange={(e) => setReportFormat(e.target.value)}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="pdf">PDF</option>
                  <option value="excel">Excel</option>
                  <option value="csv">CSV</option>
                  <option value="json">JSON</option>
                </select>
              </div>

              <div className="flex items-end">
                <button
                  onClick={generateReport}
                  className="w-full bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium transition-colors duration-200 flex items-center justify-center space-x-2"
                >
                  <Download className="w-4 h-4" />
                  <span>Générer</span>
                </button>
              </div>
            </div>
          </div>
        </div>

        {/* Recent Reports */}
        <div className="space-y-6">
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-6">Rapports Récents</h3>
            <div className="space-y-4">
              {recentReports.map((report) => {
                const IconComponent = getReportIcon(report.type);
                return (
                  <div key={report.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <div className="flex items-center space-x-3">
                      <div className="w-8 h-8 bg-white rounded-lg flex items-center justify-center shadow-sm">
                        <IconComponent className="w-4 h-4 text-gray-600" />
                      </div>
                      <div>
                        <p className="font-medium text-gray-900 text-sm">{report.title}</p>
                        <p className="text-xs text-gray-500">
                          {new Date(report.generatedAt).toLocaleDateString('fr-FR')} • {report.size}
                        </p>
                      </div>
                    </div>
                    <div className="flex items-center space-x-2">
                      <button
                        onClick={() => handleViewReport(report)}
                        className="p-1 text-gray-400 hover:text-blue-600 rounded transition-colors"
                        title="Voir le rapport"
                      >
                        <Eye className="w-4 h-4" />
                      </button>
                      <button
                        onClick={() => handleDownloadReport(report)}
                        className="p-1 text-gray-400 hover:text-green-600 rounded transition-colors"
                        title="Télécharger le rapport"
                      >
                        <Download className="w-4 h-4" />
                      </button>
                    </div>
                  </div>
                );
              })}
            </div>
          </div>

          {/* Quick Actions */}
          <div className="bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl shadow-sm p-6 text-white">
            <h3 className="text-lg font-semibold mb-4">Actions Rapides</h3>
            <div className="space-y-3">
              <button className="w-full text-left p-3 bg-white bg-opacity-10 rounded-lg hover:bg-opacity-20 transition-colors duration-200">
                <div className="font-medium">Rapport Mensuel</div>
                <div className="text-sm text-blue-100">Générer le rapport du mois en cours</div>
              </button>
              <button className="w-full text-left p-3 bg-white bg-opacity-10 rounded-lg hover:bg-opacity-20 transition-colors duration-200">
                <div className="font-medium">Analyse Comparative</div>
                <div className="text-sm text-blue-100">Comparer avec la période précédente</div>
              </button>
              <button className="w-full text-left p-3 bg-white bg-opacity-10 rounded-lg hover:bg-opacity-20 transition-colors duration-200">
                <div className="font-medium">Export Personnalisé</div>
                <div className="text-sm text-blue-100">Créer un rapport sur mesure</div>
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Reports;