import React, { useState, useEffect } from 'react';
import {
  FileText,
  Download,
  TrendingUp,
  DollarSign,
  Users,
  Filter,
  Search,
  Eye,
  RefreshCw,
  BarChart3,
  PieChart,
  AlertCircle
} from 'lucide-react';
import { getCinetPayReports } from '../../services/api';

interface CinetPayTransaction {
  id: string;
  userId: string;
  userName: string;
  timType: 'TIM_MINI' | 'TIM_MAXI' | 'TIM_BUSINESS';
  amount: number;
  commission: number;
  commissionRate: number;
  status: 'success' | 'pending' | 'failed';
  transactionId: string;
  cinetPayRef: string;
  paymentMethod: 'mobile_money' | 'card' | 'bank_transfer';
  createdAt: string;
  completedAt?: string;
  autoRepayment?: number; // Remboursement automatique de prêt
}

const CinetPayReports: React.FC = () => {
  const [dateRange, setDateRange] = useState<string>('7d');
  const [selectedStatus, setSelectedStatus] = useState<string>('all');
  const [selectedType, setSelectedType] = useState<string>('all');
  const [searchTerm, setSearchTerm] = useState('');
  const [viewMode, setViewMode] = useState<'table' | 'chart'>('table');
  const [transactions, setTransactions] = useState<CinetPayTransaction[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Récupérer les transactions CinetPay depuis l'API
  useEffect(() => {
    const fetchCinetPayReports = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await getCinetPayReports();

        // Transformer les données
        const transactionsData = response.transactions.map((tx: any) => ({
          id: tx.id,
          userId: tx.user_id,
          userName: tx.user_name,
          timType: tx.tim_type,
          amount: tx.amount,
          commission: tx.commission,
          commissionRate: tx.commission_rate,
          status: tx.status,
          transactionId: tx.transaction_id,
          cinetPayRef: tx.cinetpay_ref,
          paymentMethod: tx.payment_method,
          createdAt: tx.created_at,
          completedAt: tx.completed_at,
          autoRepayment: tx.auto_repayment
        }));

        setTransactions(transactionsData);
      } catch (err: any) {
        console.error('Erreur lors de la récupération des rapports CinetPay:', err);
        setError(err.message || 'Erreur lors de la récupération des rapports CinetPay');
      } finally {
        setLoading(false);
      }
    };

    fetchCinetPayReports();
  }, []);

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'success':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case 'success':
        return 'Réussi';
      case 'pending':
        return 'En cours';
      case 'failed':
        return 'Échoué';
      default:
        return status;
    }
  };

  const getPaymentMethodLabel = (method: string) => {
    switch (method) {
      case 'mobile_money':
        return 'Mobile Money';
      case 'card':
        return 'Carte Bancaire';
      case 'bank_transfer':
        return 'Virement Bancaire';
      default:
        return method;
    }
  };

  const getPaymentMethodColor = (method: string) => {
    switch (method) {
      case 'mobile_money':
        return 'bg-blue-100 text-blue-800';
      case 'card':
        return 'bg-purple-100 text-purple-800';
      case 'bank_transfer':
        return 'bg-green-100 text-green-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const filteredTransactions = transactions.filter(tx => {
    const matchesSearch = tx.userName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         tx.transactionId.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         tx.cinetPayRef.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = selectedStatus === 'all' || tx.status === selectedStatus;
    const matchesType = selectedType === 'all' || tx.timType === selectedType;
    return matchesSearch && matchesStatus && matchesType;
  });

  // Calculs des métriques
  const totalAmount = transactions.reduce((sum, tx) => sum + tx.amount, 0);
  const totalCommissions = transactions.reduce((sum, tx) => sum + tx.commission, 0);
  const successfulTxs = transactions.filter(tx => tx.status === 'success').length;
  const successRate = transactions.length > 0 ? (successfulTxs / transactions.length) * 100 : 0;
  const totalAutoRepayments = transactions.reduce((sum, tx) => sum + (tx.autoRepayment || 0), 0);

  // Données pour graphiques
  const statusData = [
    { name: 'Réussi', value: transactions.filter(tx => tx.status === 'success').length, color: '#10B981' },
    { name: 'En cours', value: transactions.filter(tx => tx.status === 'pending').length, color: '#F59E0B' },
    { name: 'Échoué', value: transactions.filter(tx => tx.status === 'failed').length, color: '#EF4444' }
  ];

  const typeData = [
    { name: 'TIM MINI', value: transactions.filter(tx => tx.timType === 'TIM_MINI').length, color: '#3B82F6' },
    { name: 'TIM MAXI', value: transactions.filter(tx => tx.timType === 'TIM_MAXI').length, color: '#8B5CF6' },
    { name: 'TIM BUSINESS', value: transactions.filter(tx => tx.timType === 'TIM_BUSINESS').length, color: '#F97316' }
  ];

  const handleExport = (format: 'csv' | 'pdf' | 'excel') => {
    console.log(`Exporting ${format}...`);
    // Ici vous ajouteriez la logique d'export
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div className="p-3 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg shadow-md">
              <FileText className="w-6 h-6 text-white" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-amber-900">Rapports CinetPay</h1>
              <p className="text-amber-700">Analyse complète des transactions de recharge</p>
            </div>
          </div>
          <div className="flex items-center space-x-3">
            <button
              onClick={() => setViewMode(viewMode === 'table' ? 'chart' : 'table')}
              className="flex items-center space-x-2 bg-gradient-to-r from-amber-600 to-orange-600 text-white px-4 py-2 rounded-lg hover:from-amber-700 hover:to-orange-700 transition-all shadow-md"
            >
              {viewMode === 'table' ? <BarChart3 className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
              <span>{viewMode === 'table' ? 'Vue Graphique' : 'Vue Tableau'}</span>
            </button>
            <div className="relative">
              <select
                className="appearance-none bg-white border border-amber-300 rounded-lg px-4 py-2 pr-8 focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                onChange={(e) => handleExport(e.target.value as 'csv' | 'pdf' | 'excel')}
                defaultValue=""
              >
                <option value="" disabled>Exporter</option>
                <option value="csv">CSV</option>
                <option value="pdf">PDF</option>
                <option value="excel">Excel</option>
              </select>
              <Download className="absolute right-2 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400 pointer-events-none" />
            </div>
          </div>
        </div>
      </div>

      {/* Métriques principales */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-700">Volume Total</p>
              <p className="text-2xl font-bold text-amber-800">{totalAmount.toLocaleString()} FCFA</p>
              <div className="flex items-center space-x-1 mt-1">
                <TrendingUp className="w-3 h-3 text-amber-600" />
                <span className="text-xs text-amber-600">0%</span>
              </div>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg flex items-center justify-center shadow-md">
              <DollarSign className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-yellow-50 to-amber-50 rounded-xl shadow-sm border border-yellow-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-yellow-700">Commissions</p>
              <p className="text-2xl font-bold text-yellow-800">{totalCommissions.toLocaleString()} FCFA</p>
              <div className="flex items-center space-x-1 mt-1">
                <TrendingUp className="w-3 h-3 text-yellow-600" />
                <span className="text-xs text-yellow-600">0%</span>
              </div>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-yellow-400 to-amber-500 rounded-lg flex items-center justify-center shadow-md">
              <RefreshCw className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-orange-50 to-red-50 rounded-xl shadow-sm border border-orange-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-orange-700">Taux de Réussite</p>
              <p className="text-2xl font-bold text-orange-800">{successRate.toFixed(1)}%</p>
              <div className="flex items-center space-x-1 mt-1">
                <TrendingUp className="w-3 h-3 text-orange-600" />
                <span className="text-xs text-orange-600">0%</span>
              </div>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-orange-400 to-red-500 rounded-lg flex items-center justify-center shadow-md">
              <BarChart3 className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-stone-50 to-amber-50 rounded-xl shadow-sm border border-stone-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-stone-700">Remb. Auto</p>
              <p className="text-2xl font-bold text-stone-800">{totalAutoRepayments.toLocaleString()} FCFA</p>
              <div className="flex items-center space-x-1 mt-1">
                <TrendingUp className="w-3 h-3 text-stone-600" />
                <span className="text-xs text-stone-600">0%</span>
              </div>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-stone-400 to-amber-500 rounded-lg flex items-center justify-center shadow-md">
              <Users className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Filtres */}
      <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-4">
        <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-4 md:space-y-0">
          <div className="flex items-center space-x-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
              <input
                type="text"
                placeholder="Rechercher transaction..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 pr-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
              />
            </div>
            <div className="flex items-center space-x-2">
              <Filter className="w-4 h-4 text-gray-400" />
              <select
                value={dateRange}
                onChange={(e) => setDateRange(e.target.value)}
                className="px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
              >
                <option value="24h">24 heures</option>
                <option value="7d">7 jours</option>
                <option value="30d">30 jours</option>
                <option value="90d">90 jours</option>
              </select>
              <select
                value={selectedStatus}
                onChange={(e) => setSelectedStatus(e.target.value)}
                className="px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
              >
                <option value="all">Tous les statuts</option>
                <option value="success">Réussi</option>
                <option value="pending">En cours</option>
                <option value="failed">Échoué</option>
              </select>
              <select
                value={selectedType}
                onChange={(e) => setSelectedType(e.target.value)}
                className="px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
              >
                <option value="all">Tous les types</option>
                <option value="TIM_MINI">TIM MINI</option>
                <option value="TIM_MAXI">TIM MAXI</option>
                <option value="TIM_BUSINESS">TIM BUSINESS</option>
              </select>
            </div>
          </div>
        </div>
      </div>

      {loading ? (
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-12 text-center">
          <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600 mb-4"></div>
          <p className="text-amber-600">Chargement des rapports CinetPay...</p>
        </div>
      ) : error ? (
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-12 text-center">
          <p className="text-red-600 font-medium mb-4">{error}</p>
          <button
            onClick={() => window.location.reload()}
            className="px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700 transition-colors"
          >
            Réessayer
          </button>
        </div>
      ) : transactions.length === 0 ? (
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-12 text-center">
          <div className="w-16 h-16 bg-gradient-to-br from-amber-200 to-orange-300 rounded-full flex items-center justify-center mx-auto mb-4">
            <AlertCircle className="w-8 h-8 text-amber-700" />
          </div>
          <h3 className="text-xl font-semibold text-amber-900 mb-2">Aucune transaction CinetPay</h3>
          <p className="text-amber-700">
            Les transactions de recharge CinetPay apparaîtront ici lorsque les utilisateurs effectueront des recharges.
          </p>
        </div>
      ) : viewMode === 'chart' ? (
        /* Vue Graphique */
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-6">
            <div className="flex items-center space-x-2 mb-4">
              <PieChart className="w-5 h-5 text-amber-700" />
              <h3 className="text-lg font-semibold text-amber-900">Répartition par Statut</h3>
            </div>
            <div className="space-y-3">
              {statusData.map((item, index) => (
                <div key={index} className="flex items-center justify-between">
                  <div className="flex items-center space-x-2">
                    <div className="w-4 h-4 rounded-full" style={{ backgroundColor: item.color }}></div>
                    <span className="text-sm font-medium text-amber-900">{item.name}</span>
                  </div>
                  <span className="text-sm text-amber-700">{item.value}</span>
                </div>
              ))}
            </div>
          </div>
          <div className="bg-gradient-to-br from-yellow-50 to-amber-50 rounded-xl shadow-sm border border-yellow-200 p-6">
            <div className="flex items-center space-x-2 mb-4">
              <BarChart3 className="w-5 h-5 text-yellow-700" />
              <h3 className="text-lg font-semibold text-yellow-900">Répartition par Type</h3>
            </div>
            <div className="space-y-3">
              {typeData.map((item, index) => (
                <div key={index} className="flex items-center justify-between">
                  <div className="flex items-center space-x-2">
                    <div className="w-4 h-4 rounded-full" style={{ backgroundColor: item.color }}></div>
                    <span className="text-sm font-medium text-yellow-900">{item.name}</span>
                  </div>
                  <span className="text-sm text-yellow-700">{item.value}</span>
                </div>
              ))}
            </div>
          </div>
        </div>
      ) : (
        /* Vue Tableau */
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200">
          <div className="p-6 border-b border-amber-200">
            <h3 className="text-lg font-semibold text-amber-900">
              Transactions CinetPay ({filteredTransactions.length})
            </h3>
          </div>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gradient-to-r from-amber-100 to-orange-100">
                <tr>
                  <th className="text-left py-3 px-4 font-medium text-amber-900">Utilisateur</th>
                  <th className="text-left py-3 px-4 font-medium text-amber-900">Montant</th>
                  <th className="text-left py-3 px-4 font-medium text-amber-900">Commission</th>
                  <th className="text-left py-3 px-4 font-medium text-amber-900">Méthode</th>
                  <th className="text-left py-3 px-4 font-medium text-amber-900">Statut</th>
                  <th className="text-left py-3 px-4 font-medium text-amber-900">Date</th>
                  <th className="text-left py-3 px-4 font-medium text-amber-900">Référence</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-amber-200">
                {filteredTransactions.map((tx) => (
                  <tr key={tx.id} className="hover:bg-amber-50/50">
                    <td className="py-3 px-4">
                      <div>
                        <div className="font-medium text-amber-900">{tx.userName}</div>
                        <div className="text-sm text-amber-600">{tx.timType}</div>
                      </div>
                    </td>
                    <td className="py-3 px-4">
                      <div className="font-medium text-amber-900">{tx.amount.toLocaleString()} FCFA</div>
                      {tx.autoRepayment && tx.autoRepayment > 0 && (
                        <div className="text-sm text-orange-600">Remb: -{tx.autoRepayment.toLocaleString()} FCFA</div>
                      )}
                    </td>
                    <td className="py-3 px-4">
                      <div className="font-medium text-yellow-700">{tx.commission.toLocaleString()} FCFA</div>
                      <div className="text-sm text-amber-600">{tx.commissionRate}%</div>
                    </td>
                    <td className="py-3 px-4">
                      <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getPaymentMethodColor(tx.paymentMethod)}`}>
                        {getPaymentMethodLabel(tx.paymentMethod)}
                      </span>
                    </td>
                    <td className="py-3 px-4">
                      <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(tx.status)}`}>
                        {getStatusLabel(tx.status)}
                      </span>
                    </td>
                    <td className="py-3 px-4">
                      <div className="text-sm text-amber-900">
                        {new Date(tx.createdAt).toLocaleDateString('fr-FR')}
                      </div>
                      <div className="text-sm text-amber-600">
                        {new Date(tx.createdAt).toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' })}
                      </div>
                    </td>
                    <td className="py-3 px-4">
                      <div className="text-sm font-mono text-amber-900">{tx.cinetPayRef}</div>
                      <div className="text-sm text-amber-600">{tx.transactionId}</div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}
    </div>
  );
};

export default CinetPayReports;
