import React, { useState } from 'react';
import { 
  Bell, 
  BellRing, 
  Check, 
  Search,
  MoreHorizontal,
  AlertCircle,
  Info,
  CheckCircle,
  XCircle,
  Clock,
  Settings
} from 'lucide-react';

const Notifications: React.FC = () => {
  const [selectedFilter, setSelectedFilter] = useState('all');
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedNotifications, setSelectedNotifications] = useState<string[]>([]);

  // Liste vide des notifications
  const notifications: any[] = [];

  const getNotificationIcon = (type: string) => {
    switch (type) {
      case 'nfc_payment':
        return <CheckCircle className="w-5 h-5 text-green-600" />;
      case 'nfc_failed':
        return <XCircle className="w-5 h-5 text-red-600" />;
      case 'loan_due':
      case 'loan_penalty':
        return <Clock className="w-5 h-5 text-yellow-600" />;
      case 'recharge_cinetpay':
      case 'business_recharge':
        return <CheckCircle className="w-5 h-5 text-blue-600" />;
      case 'auto_repayment':
        return <CheckCircle className="w-5 h-5 text-purple-600" />;
      case 'account_limit':
        return <AlertCircle className="w-5 h-5 text-orange-600" />;
      case 'commission_earned':
        return <CheckCircle className="w-5 h-5 text-green-600" />;
      case 'bank_transfer':
        return <Info className="w-5 h-5 text-blue-600" />;
      default:
        return <Bell className="w-5 h-5 text-gray-600" />;
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high':
        return 'border-l-red-500 bg-red-50';
      case 'medium':
        return 'border-l-yellow-500 bg-yellow-50';
      case 'low':
        return 'border-l-blue-500 bg-blue-50';
      default:
        return 'border-l-gray-500 bg-gray-50';
    }
  };

  const [notificationList, setNotificationList] = useState(notifications);

  const markAsRead = (ids: string[]) => {
    setNotificationList(prev => 
      prev.map(notif => 
        ids.includes(notif.id) ? { ...notif, read: true } : notif
      )
    );
    setSelectedNotifications([]);
  };

  const deleteNotifications = (ids: string[]) => {
    setNotificationList(prev => 
      prev.filter(notif => !ids.includes(notif.id))
    );
    setSelectedNotifications([]);
  };

  const handleNotificationClick = (notification: any) => {
    if (!notification.read) {
      markAsRead([notification.id]);
    }
    if (notification.actionUrl) {
      // Ici vous pourriez naviguer vers l'URL
      console.log('Navigating to:', notification.actionUrl);
    }
  };

  const filteredNotifications = notificationList.filter(notification => {
    const matchesSearch = 
      notification.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
      notification.message.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesFilter = 
      selectedFilter === 'all' ||
      (selectedFilter === 'unread' && !notification.read) ||
      (selectedFilter === 'read' && notification.read) ||
      notification.type === selectedFilter ||
      notification.priority === selectedFilter;
    
    return matchesSearch && matchesFilter;
  });

  const unreadCount = notificationList.filter(n => !n.read).length;
  const highPriorityCount = notificationList.filter(n => n.priority === 'high' && !n.read).length;

  const toggleNotificationSelection = (id: string) => {
    setSelectedNotifications(prev => 
      prev.includes(id) 
        ? prev.filter(nId => nId !== id)
        : [...prev, id]
    );
  };

  return (
    <div className="space-y-8">
      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-amber-900">🔔 Notifications</h1>
            <p className="text-amber-700 mt-2">
              Gérez vos notifications et alertes importantes
            </p>
          </div>
          <div className="flex space-x-4">
            <button className="border border-amber-300 text-amber-700 hover:bg-amber-50 px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2">
              <Settings className="w-4 h-4" />
              <span>Paramètres</span>
            </button>
            <button 
              onClick={() => markAsRead(notificationList.filter(n => !n.read).map(n => n.id))}
              className="bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md hover:shadow-lg transform hover:scale-105 flex items-center space-x-2"
            >
              <Check className="w-4 h-4" />
              <span>Tout marquer lu</span>
            </button>
          </div>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl shadow-lg border border-blue-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-blue-700">Total</p>
              <p className="text-3xl font-bold text-blue-900">{notificationList.length}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-blue-400 to-indigo-500 rounded-lg flex items-center justify-center shadow-md">
              <Bell className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-yellow-50 to-amber-50 rounded-xl shadow-lg border border-yellow-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-yellow-700">Non Lues</p>
              <p className="text-3xl font-bold text-yellow-900">{unreadCount}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-yellow-400 to-amber-500 rounded-lg flex items-center justify-center shadow-md">
              <BellRing className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-xl shadow-lg border border-red-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-red-700">Priorité Haute</p>
              <p className="text-3xl font-bold text-red-900">{highPriorityCount}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-red-400 to-rose-500 rounded-lg flex items-center justify-center shadow-md">
              <AlertCircle className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-lg border border-green-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-green-700">Aujourd'hui</p>
              <p className="text-3xl font-bold text-green-900">
                {notificationList.filter(n => 
                  new Date(n.timestamp).toDateString() === new Date().toDateString()
                ).length}
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-green-400 to-emerald-500 rounded-lg flex items-center justify-center shadow-md">
              <Clock className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Filters and Search avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/50 to-orange-50/50 rounded-xl shadow-lg border border-amber-200 p-6">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
          <div className="flex flex-col sm:flex-row space-y-4 sm:space-y-0 sm:space-x-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
              <input
                type="text"
                placeholder="Rechercher dans les notifications..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 pr-4 py-2 w-full sm:w-80 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              />
            </div>
            
            <select
              value={selectedFilter}
              onChange={(e) => setSelectedFilter(e.target.value)}
              className="px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
            >
              <option value="all">Toutes</option>
              <option value="unread">Non lues</option>
              <option value="read">Lues</option>
              <option value="high">Priorité haute</option>
              <option value="nfc_payment">Paiements NFC</option>
              <option value="loan_due">Prêts SOS</option>
              <option value="recharge_cinetpay">Recharges CinetPay</option>
              <option value="business_recharge">TIM BUSINESS</option>
              <option value="commission_earned">Commissions</option>
            </select>
          </div>

          <div className="flex items-center space-x-4">
            {selectedNotifications.length > 0 && (
              <div className="flex items-center space-x-2">
                <span className="text-sm text-gray-600">
                  {selectedNotifications.length} sélectionnée(s)
                </span>
                <button
                  onClick={() => markAsRead(selectedNotifications)}
                  className="text-blue-600 hover:text-blue-700 text-sm font-medium"
                >
                  Marquer lues
                </button>
                <button
                  onClick={() => deleteNotifications(selectedNotifications)}
                  className="text-red-600 hover:text-red-700 text-sm font-medium"
                >
                  Supprimer
                </button>
              </div>
            )}
            <span className="text-sm text-gray-600">
              {filteredNotifications.length} notification(s)
            </span>
          </div>
        </div>
      </div>

      {/* Notifications List avec thème marron doré */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200">
        <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50/30 to-orange-50/30">
          <h3 className="text-lg font-semibold text-amber-900">Toutes les Notifications</h3>
        </div>
        
        <div className="divide-y divide-gray-200">
          {filteredNotifications.map((notification) => (
            <div 
              key={notification.id} 
              className={`p-6 border-l-4 transition-colors duration-150 cursor-pointer ${
                notification.read ? 'bg-white' : getPriorityColor(notification.priority)
              } hover:bg-gray-50`}
              onClick={() => handleNotificationClick(notification)}
            >
              <div className="flex items-start justify-between">
                <div className="flex items-start space-x-4 flex-1">
                  <input
                    type="checkbox"
                    checked={selectedNotifications.includes(notification.id)}
                    onChange={() => toggleNotificationSelection(notification.id)}
                    className="mt-1 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                  />
                  
                  <div className="w-10 h-10 bg-white rounded-lg flex items-center justify-center shadow-sm">
                    {getNotificationIcon(notification.type)}
                  </div>
                  
                  <div className="flex-1">
                    <div className="flex items-center space-x-3 mb-1">
                      <h4 className={`font-medium ${notification.read ? 'text-gray-700' : 'text-gray-900'}`}>
                        {notification.title}
                      </h4>
                      {!notification.read && (
                        <div className="w-2 h-2 bg-blue-600 rounded-full"></div>
                      )}
                      <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                        notification.priority === 'high' ? 'bg-red-100 text-red-800' :
                        notification.priority === 'medium' ? 'bg-yellow-100 text-yellow-800' :
                        'bg-blue-100 text-blue-800'
                      }`}>
                        {notification.priority === 'high' ? 'Haute' :
                         notification.priority === 'medium' ? 'Moyenne' : 'Basse'}
                      </span>
                    </div>
                    <p className={`text-sm mb-2 ${notification.read ? 'text-gray-500' : 'text-gray-600'}`}>
                      {notification.message}
                    </p>
                    <div className="flex items-center space-x-4 text-xs text-gray-500">
                      <span>
                        {new Date(notification.timestamp).toLocaleDateString('fr-FR')} à{' '}
                        {new Date(notification.timestamp).toLocaleTimeString('fr-FR', { 
                          hour: '2-digit', 
                          minute: '2-digit' 
                        })}
                      </span>
                      <span className="capitalize">{notification.category}</span>
                    </div>
                  </div>
                </div>
                
                <div className="flex items-center space-x-2 ml-4">
                  {notification.actionUrl && (
                    <button className="text-blue-600 hover:text-blue-700 text-sm font-medium">
                      Voir
                    </button>
                  )}
                  {!notification.read && (
                    <button
                      onClick={() => markAsRead([notification.id])}
                      className="p-1 text-gray-400 hover:text-gray-600 rounded"
                    >
                      <Check className="w-4 h-4" />
                    </button>
                  )}
                  <button className="p-1 text-gray-400 hover:text-gray-600 rounded">
                    <MoreHorizontal className="w-4 h-4" />
                  </button>
                </div>
              </div>
            </div>
          ))}
        </div>

        {filteredNotifications.length === 0 && (
          <div className="p-12 text-center">
            <div className="inline-flex items-center justify-center w-16 h-16 bg-amber-100 rounded-full mb-4">
              <Bell className="w-8 h-8 text-amber-600" />
            </div>
            <h3 className="text-lg font-medium text-gray-900 mb-2">Aucune notification</h3>
            <p className="text-gray-600">
              {searchTerm || selectedFilter !== 'all' 
                ? 'Aucune notification ne correspond à vos critères.'
                : 'Vous n\'avez aucune notification pour le moment.'
              }
            </p>
          </div>
        )}
      </div>

      {/* Notification Settings avec thème marron doré */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
        <h3 className="text-lg font-semibold text-amber-900 mb-6">Paramètres de Notification</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="space-y-4">
            <h4 className="font-medium text-gray-900">Notifications Push</h4>
            <div className="space-y-3">
              {[
                { id: 'nfc_payments', label: 'Paiements NFC', enabled: true },
                { id: 'loan_alerts', label: 'Alertes Prêts SOS', enabled: true },
                { id: 'recharge_notifications', label: 'Recharges CinetPay', enabled: true },
                { id: 'business_commissions', label: 'Commissions TIM BUSINESS', enabled: true },
                { id: 'account_limits', label: 'Limites de Compte', enabled: false }
              ].map((setting) => (
                <div key={setting.id} className="flex items-center justify-between">
                  <span className="text-sm text-gray-700">{setting.label}</span>
                  <button className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                    setting.enabled ? 'bg-blue-600' : 'bg-gray-200'
                  }`}>
                    <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                      setting.enabled ? 'translate-x-6' : 'translate-x-1'
                    }`} />
                  </button>
                </div>
              ))}
            </div>
          </div>
          
          <div className="space-y-4">
            <h4 className="font-medium text-gray-900">Notifications Email</h4>
            <div className="space-y-3">
              {[
                { id: 'daily_summary', label: 'Résumé Quotidien TIM CASH', enabled: true },
                { id: 'weekly_report', label: 'Rapport Hebdomadaire', enabled: true },
                { id: 'loan_reminders', label: 'Rappels d\'Échéance Prêts', enabled: true },
                { id: 'high_priority_only', label: 'Alertes Importantes Uniquement', enabled: false }
              ].map((setting) => (
                <div key={setting.id} className="flex items-center justify-between">
                  <span className="text-sm text-gray-700">{setting.label}</span>
                  <button className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors ${
                    setting.enabled ? 'bg-blue-600' : 'bg-gray-200'
                  }`}>
                    <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition-transform ${
                      setting.enabled ? 'translate-x-6' : 'translate-x-1'
                    }`} />
                  </button>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Notifications;