import React, { useState, useEffect } from 'react';
import { 
  Activity, 
  Server, 
  Database, 
  Cpu, 
  HardDrive,
  TrendingUp,
  TrendingDown,
  AlertCircle,
  CheckCircle,
  RefreshCw,
  Clock,
  Zap,
  Users,
  CreditCard,
  DollarSign,
  AlertTriangle
} from 'lucide-react';
import { getSystemMonitoring } from '../../services/api';

interface SystemMetrics {
  cpu_usage: number;
  memory_usage: number;
  memory_available: number;
  memory_total: number;
  disk_usage: number;
  disk_free: number;
  disk_total: number;
}

interface HealthStatus {
  status: string;
  issues: string[];
  uptime: string;
  last_check: string;
}

interface MonitoringData {
  health: HealthStatus;
  system: SystemMetrics;
  database: {
    total_users: number;
    total_transactions: number;
    total_wallets: number;
    total_loans: number;
    connection_status: string;
  };
  transactions: {
    total: number;
    pending: number;
    completed: number;
    failed: number;
    success_rate: number;
    recent_24h: number;
  };
  loans: {
    total: number;
    active: number;
    overdue: number;
    completed: number;
  };
  performance: {
    api_response_time: number;
    total_volume: number;
    admin_stock_balance: number;
    total_commissions: number;
  };
  activity: {
    recent_transactions_24h: number;
    recent_users_24h: number;
  };
}

const Monitoring: React.FC = () => {
  const [data, setData] = useState<MonitoringData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [autoRefresh, setAutoRefresh] = useState(true);

  const fetchMonitoringData = async () => {
    try {
      setError(null);
      const response = await getSystemMonitoring();
      setData(response);
    } catch (err: any) {
      console.error('Erreur lors de la récupération des données de monitoring:', err);
      setError(err.message || 'Erreur lors de la récupération des données de monitoring');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchMonitoringData();
  }, []);

  // Auto-refresh toutes les 10 secondes
  useEffect(() => {
    if (!autoRefresh) return;

    const interval = setInterval(() => {
      fetchMonitoringData();
    }, 10000);

    return () => clearInterval(interval);
  }, [autoRefresh]);

  const handleRefresh = () => {
    setLoading(true);
    fetchMonitoringData();
  };

  const getHealthColor = (status: string) => {
    switch (status) {
      case 'healthy':
        return 'text-green-600 bg-green-50 border-green-200';
      case 'warning':
        return 'text-yellow-600 bg-yellow-50 border-yellow-200';
      case 'critical':
        return 'text-red-600 bg-red-50 border-red-200';
      default:
        return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  const getHealthIcon = (status: string) => {
    switch (status) {
      case 'healthy':
        return <CheckCircle className="w-6 h-6 text-green-600" />;
      case 'warning':
        return <AlertTriangle className="w-6 h-6 text-yellow-600" />;
      case 'critical':
        return <AlertCircle className="w-6 h-6 text-red-600" />;
      default:
        return <Activity className="w-6 h-6 text-gray-600" />;
    }
  };

  const getUsageColor = (usage: number) => {
    if (usage < 50) return 'bg-green-500';
    if (usage < 75) return 'bg-yellow-500';
    if (usage < 90) return 'bg-orange-500';
    return 'bg-red-500';
  };

  if (loading && !data) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="text-center">
          <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mb-4"></div>
          <p className="text-gray-600">Chargement des données de monitoring...</p>
        </div>
      </div>
    );
  }

  if (error && !data) {
    return (
      <div className="bg-red-50 border border-red-200 rounded-xl p-12 text-center">
        <AlertCircle className="w-12 h-12 text-red-600 mx-auto mb-4" />
        <p className="text-red-600 font-medium mb-4">{error}</p>
        <button 
          onClick={handleRefresh} 
          className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors"
        >
          Réessayer
        </button>
      </div>
    );
  }

  if (!data) return null;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-50 via-indigo-50 to-purple-50 rounded-xl shadow-lg border border-blue-200 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-blue-900 flex items-center space-x-3">
              <Activity className="w-8 h-8" />
              <span>Monitoring Système</span>
            </h1>
            <p className="text-blue-700 mt-2">
              Surveillance en temps réel de la santé et des performances du système
            </p>
          </div>
          <div className="flex items-center space-x-4">
            <div className="flex items-center space-x-2">
              <input
                type="checkbox"
                id="autoRefresh"
                checked={autoRefresh}
                onChange={(e) => setAutoRefresh(e.target.checked)}
                className="w-4 h-4 text-blue-600 rounded focus:ring-blue-500"
              />
              <label htmlFor="autoRefresh" className="text-sm text-blue-700">
                Auto-refresh (10s)
              </label>
            </div>
            <button
              onClick={handleRefresh}
              disabled={loading}
              className="bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md hover:shadow-lg flex items-center space-x-2 disabled:opacity-50"
            >
              <RefreshCw className={`w-4 h-4 ${loading ? 'animate-spin' : ''}`} />
              <span>Actualiser</span>
            </button>
          </div>
        </div>
      </div>

      {/* Health Status */}
      <div className={`rounded-xl shadow-lg border p-6 ${getHealthColor(data.health.status)}`}>
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            {getHealthIcon(data.health.status)}
            <div>
              <h2 className="text-xl font-bold capitalize">{data.health.status === 'healthy' ? 'Système Opérationnel' : data.health.status === 'warning' ? 'Avertissement' : 'Critique'}</h2>
              <p className="text-sm mt-1">
                Dernière vérification: {new Date(data.health.last_check).toLocaleString('fr-FR')}
              </p>
            </div>
          </div>
          <div className="text-right">
            <p className="text-sm font-medium">Disponibilité</p>
            <p className="text-2xl font-bold">{data.health.uptime}</p>
          </div>
        </div>
        {data.health.issues.length > 0 && (
          <div className="mt-4 pt-4 border-t border-current/20">
            <p className="font-medium mb-2">Problèmes détectés:</p>
            <ul className="list-disc list-inside space-y-1">
              {data.health.issues.map((issue, index) => (
                <li key={index} className="text-sm">{issue}</li>
              ))}
            </ul>
          </div>
        )}
      </div>

      {/* System Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        {/* CPU */}
        <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-gradient-to-br from-blue-100 to-indigo-100 rounded-lg flex items-center justify-center">
                <Cpu className="w-6 h-6 text-blue-600" />
              </div>
              <div>
                <p className="text-sm font-medium text-gray-600">CPU Usage</p>
                <p className="text-2xl font-bold text-gray-900">{data.system.cpu_usage}%</p>
              </div>
            </div>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-3">
            <div
              className={`h-3 rounded-full transition-all ${getUsageColor(data.system.cpu_usage)}`}
              style={{ width: `${data.system.cpu_usage}%` }}
            ></div>
          </div>
        </div>

        {/* Memory */}
        <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-gradient-to-br from-purple-100 to-pink-100 rounded-lg flex items-center justify-center">
                <Server className="w-6 h-6 text-purple-600" />
              </div>
              <div>
                <p className="text-sm font-medium text-gray-600">Memory</p>
                <p className="text-2xl font-bold text-gray-900">{data.system.memory_usage}%</p>
              </div>
            </div>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-3 mb-2">
            <div
              className={`h-3 rounded-full transition-all ${getUsageColor(data.system.memory_usage)}`}
              style={{ width: `${data.system.memory_usage}%` }}
            ></div>
          </div>
          <p className="text-xs text-gray-500">
            {data.system.memory_available.toFixed(1)} GB / {data.system.memory_total.toFixed(1)} GB disponible
          </p>
        </div>

        {/* Disk */}
        <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-gradient-to-br from-green-100 to-emerald-100 rounded-lg flex items-center justify-center">
                <HardDrive className="w-6 h-6 text-green-600" />
              </div>
              <div>
                <p className="text-sm font-medium text-gray-600">Disk</p>
                <p className="text-2xl font-bold text-gray-900">{data.system.disk_usage}%</p>
              </div>
            </div>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-3 mb-2">
            <div
              className={`h-3 rounded-full transition-all ${getUsageColor(data.system.disk_usage)}`}
              style={{ width: `${data.system.disk_usage}%` }}
            ></div>
          </div>
          <p className="text-xs text-gray-500">
            {data.system.disk_free.toFixed(1)} GB / {data.system.disk_total.toFixed(1)} GB libre
          </p>
        </div>
      </div>

      {/* Database & Performance */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Database Stats */}
        <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-6">
          <div className="flex items-center space-x-3 mb-6">
            <div className="w-12 h-12 bg-gradient-to-br from-amber-100 to-orange-100 rounded-lg flex items-center justify-center">
              <Database className="w-6 h-6 text-amber-600" />
            </div>
            <div>
              <h3 className="text-lg font-semibold text-gray-900">Base de Données</h3>
              <p className="text-sm text-gray-600">Statut: <span className="text-green-600 font-medium">{data.database.connection_status}</span></p>
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div className="bg-blue-50 rounded-lg p-4">
              <div className="flex items-center space-x-2 mb-2">
                <Users className="w-4 h-4 text-blue-600" />
                <p className="text-sm font-medium text-blue-700">Utilisateurs</p>
              </div>
              <p className="text-2xl font-bold text-blue-900">{data.database.total_users.toLocaleString()}</p>
            </div>
            <div className="bg-green-50 rounded-lg p-4">
              <div className="flex items-center space-x-2 mb-2">
                <Activity className="w-4 h-4 text-green-600" />
                <p className="text-sm font-medium text-green-700">Transactions</p>
              </div>
              <p className="text-2xl font-bold text-green-900">{data.database.total_transactions.toLocaleString()}</p>
            </div>
            <div className="bg-purple-50 rounded-lg p-4">
              <div className="flex items-center space-x-2 mb-2">
                <DollarSign className="w-4 h-4 text-purple-600" />
                <p className="text-sm font-medium text-purple-700">Wallets</p>
              </div>
              <p className="text-2xl font-bold text-purple-900">{data.database.total_wallets.toLocaleString()}</p>
            </div>
            <div className="bg-orange-50 rounded-lg p-4">
              <div className="flex items-center space-x-2 mb-2">
                <CreditCard className="w-4 h-4 text-orange-600" />
                <p className="text-sm font-medium text-orange-700">Prêts</p>
              </div>
              <p className="text-2xl font-bold text-orange-900">{data.database.total_loans.toLocaleString()}</p>
            </div>
          </div>
        </div>

        {/* Performance Metrics */}
        <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-6">
          <div className="flex items-center space-x-3 mb-6">
            <div className="w-12 h-12 bg-gradient-to-br from-indigo-100 to-blue-100 rounded-lg flex items-center justify-center">
              <Zap className="w-6 h-6 text-indigo-600" />
            </div>
            <div>
              <h3 className="text-lg font-semibold text-gray-900">Performance</h3>
              <p className="text-sm text-gray-600">Métriques en temps réel</p>
            </div>
          </div>
          <div className="space-y-4">
            <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-2">
                <Clock className="w-4 h-4 text-gray-600" />
                <span className="text-sm font-medium text-gray-700">Temps de réponse API</span>
              </div>
              <span className="text-lg font-bold text-gray-900">{data.performance.api_response_time}ms</span>
            </div>
            <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-2">
                <DollarSign className="w-4 h-4 text-gray-600" />
                <span className="text-sm font-medium text-gray-700">Volume Total</span>
              </div>
              <span className="text-lg font-bold text-gray-900">{data.performance.total_volume.toLocaleString()} FCFA</span>
            </div>
            <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-2">
                <Server className="w-4 h-4 text-gray-600" />
                <span className="text-sm font-medium text-gray-700">Stock Admin</span>
              </div>
              <span className="text-lg font-bold text-gray-900">{data.performance.admin_stock_balance.toLocaleString()} FCFA</span>
            </div>
            <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-2">
                <TrendingUp className="w-4 h-4 text-gray-600" />
                <span className="text-sm font-medium text-gray-700">Commissions Totales</span>
              </div>
              <span className="text-lg font-bold text-gray-900">{data.performance.total_commissions.toLocaleString()} FCFA</span>
            </div>
          </div>
        </div>
      </div>

      {/* Transactions & Loans */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Transactions */}
        <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-6">Transactions</h3>
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-600">Total</span>
              <span className="text-lg font-bold text-gray-900">{data.transactions.total.toLocaleString()}</span>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-600">En attente</span>
              <span className="text-lg font-bold text-yellow-600">{data.transactions.pending.toLocaleString()}</span>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-600">Complétées</span>
              <span className="text-lg font-bold text-green-600">{data.transactions.completed.toLocaleString()}</span>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-600">Échouées</span>
              <span className="text-lg font-bold text-red-600">{data.transactions.failed.toLocaleString()}</span>
            </div>
            <div className="pt-4 border-t border-gray-200">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-medium text-gray-600">Taux de réussite</span>
                <span className="text-lg font-bold text-blue-600">{data.transactions.success_rate}%</span>
              </div>
              <div className="w-full bg-gray-200 rounded-full h-2">
                <div
                  className="bg-blue-600 h-2 rounded-full transition-all"
                  style={{ width: `${data.transactions.success_rate}%` }}
                ></div>
              </div>
            </div>
            <div className="pt-4 border-t border-gray-200">
              <div className="flex items-center justify-between">
                <span className="text-sm font-medium text-gray-600">Dernières 24h</span>
                <span className="text-lg font-bold text-indigo-600">{data.transactions.recent_24h.toLocaleString()}</span>
              </div>
            </div>
          </div>
        </div>

        {/* Loans */}
        <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-6">Prêts</h3>
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-600">Total</span>
              <span className="text-lg font-bold text-gray-900">{data.loans.total.toLocaleString()}</span>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-600">Actifs</span>
              <span className="text-lg font-bold text-blue-600">{data.loans.active.toLocaleString()}</span>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-600">En retard</span>
              <span className="text-lg font-bold text-red-600">{data.loans.overdue.toLocaleString()}</span>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-600">Complétés</span>
              <span className="text-lg font-bold text-green-600">{data.loans.completed.toLocaleString()}</span>
            </div>
            <div className="pt-4 border-t border-gray-200">
              <div className="grid grid-cols-2 gap-4">
                <div className="bg-blue-50 rounded-lg p-3 text-center">
                  <p className="text-xs font-medium text-blue-700 mb-1">Taux d'activité</p>
                  <p className="text-xl font-bold text-blue-900">
                    {data.loans.total > 0 ? ((data.loans.active / data.loans.total) * 100).toFixed(1) : 0}%
                  </p>
                </div>
                <div className="bg-green-50 rounded-lg p-3 text-center">
                  <p className="text-xs font-medium text-green-700 mb-1">Taux de complétion</p>
                  <p className="text-xl font-bold text-green-900">
                    {data.loans.total > 0 ? ((data.loans.completed / data.loans.total) * 100).toFixed(1) : 0}%
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Activity */}
      <div className="bg-gradient-to-r from-indigo-50 to-purple-50 rounded-xl shadow-lg border border-indigo-200 p-6">
        <h3 className="text-lg font-semibold text-indigo-900 mb-4">Activité Récente (24h)</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="flex items-center space-x-4">
            <div className="w-16 h-16 bg-gradient-to-br from-blue-500 to-indigo-500 rounded-xl flex items-center justify-center">
              <Activity className="w-8 h-8 text-white" />
            </div>
            <div>
              <p className="text-sm font-medium text-indigo-700">Transactions</p>
              <p className="text-3xl font-bold text-indigo-900">{data.activity.recent_transactions_24h.toLocaleString()}</p>
            </div>
          </div>
          <div className="flex items-center space-x-4">
            <div className="w-16 h-16 bg-gradient-to-br from-purple-500 to-pink-500 rounded-xl flex items-center justify-center">
              <Users className="w-8 h-8 text-white" />
            </div>
            <div>
              <p className="text-sm font-medium text-purple-700">Nouveaux Utilisateurs</p>
              <p className="text-3xl font-bold text-purple-900">{data.activity.recent_users_24h.toLocaleString()}</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Monitoring;

