import React, { useState, useEffect } from 'react';
import {
  CreditCard,
  Users,
  AlertTriangle,
  CheckCircle,
  Clock,
  DollarSign,
  Search,
  Filter,
  Eye,
  Ban,
  RefreshCw
} from 'lucide-react';
import { getLoans } from '../../services/api';

interface Loan {
  id: string;
  userId: string;
  userName: string;
  userType: 'tim_maxi' | 'tim_business' | 'tim_mini';
  amount: number;
  commission: number;
  totalAmount: number;
  status: 'active' | 'overdue' | 'repaid' | 'blocked' | 'completed' | 'pending' | 'approved';
  dueDate: string;
  createdAt: string;
  repaidAt?: string;
  daysOverdue?: number;
  remainingBalance?: number;
  interestRate?: number;
  penaltyAmount?: number;
}

const LoansManagement: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedFilter, setSelectedFilter] = useState('all');
  const [selectedLoan, setSelectedLoan] = useState<Loan | null>(null);
  const [showDetailsModal, setShowDetailsModal] = useState(false);
  const [loans, setLoans] = useState<Loan[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Récupérer les prêts depuis l'API
  useEffect(() => {
    const fetchLoans = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await getLoans();

        // Transformer les données de l'API au format attendu
        const loansData = response.loans.map((loan: any) => {
          // Calculer la commission (6% du montant)
          const commission = loan.amount * (loan.interest_rate || 0.06);
          const totalAmount = loan.amount + commission;

          // Déterminer le statut
          let status: 'active' | 'overdue' | 'repaid' | 'blocked' | 'completed' | 'pending' | 'approved' = 'active';
          if (loan.status === 'completed' || loan.remaining_balance === 0) {
            status = 'repaid';
          } else if (loan.status === 'overdue' || loan.days_overdue > 0) {
            status = 'overdue';
          } else if (loan.status === 'pending') {
            status = 'pending';
          } else if (loan.status === 'approved' || loan.status === 'active') {
            status = 'active';
          }

          return {
            id: loan.id,
            userId: loan.user_id,
            userName: loan.user_name,
            userType: loan.tim_type?.toLowerCase() || 'tim_maxi',
            amount: loan.amount,
            commission: commission,
            totalAmount: totalAmount,
            status: status,
            dueDate: loan.due_date,
            createdAt: loan.created_at,
            repaidAt: loan.status === 'completed' ? loan.due_date : undefined,
            daysOverdue: loan.days_overdue,
            remainingBalance: loan.remaining_balance,
            interestRate: loan.interest_rate,
            penaltyAmount: loan.penalty_amount
          };
        });

        setLoans(loansData);
      } catch (err: any) {
        console.error('Erreur lors de la récupération des prêts:', err);
        setError(err.message || 'Erreur lors de la récupération des prêts');
      } finally {
        setLoading(false);
      }
    };

    fetchLoans();
  }, []);

  const filteredLoans = loans.filter(loan => {
    const matchesSearch = loan.userName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         loan.id.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesFilter = selectedFilter === 'all' || loan.status === selectedFilter;
    return matchesSearch && matchesFilter;
  });

  const loanStats = {
    total: loans.length,
    active: loans.filter(l => l.status === 'active').length,
    overdue: loans.filter(l => l.status === 'overdue').length,
    blocked: loans.filter(l => l.status === 'blocked').length,
    totalAmount: loans.reduce((sum, l) => sum + l.amount, 0),
    totalCommissions: loans.reduce((sum, l) => sum + l.commission, 0)
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'approved': return 'bg-green-100 text-green-800';
      case 'overdue': return 'bg-yellow-100 text-yellow-800';
      case 'repaid': return 'bg-blue-100 text-blue-800';
      case 'completed': return 'bg-blue-100 text-blue-800';
      case 'blocked': return 'bg-red-100 text-red-800';
      case 'pending': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case 'active': return 'Actif';
      case 'approved': return 'Approuvé';
      case 'overdue': return 'En retard';
      case 'repaid': return 'Remboursé';
      case 'completed': return 'Remboursé';
      case 'blocked': return 'Bloqué';
      case 'pending': return 'En attente';
      default: return status;
    }
  };

  const handleViewDetails = (loan: Loan) => {
    setSelectedLoan(loan);
    setShowDetailsModal(true);
  };

  return (
    <div className="p-6 max-w-7xl mx-auto">
      {/* Header avec thème marron doré */}
      <div className="mb-8 bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <h1 className="text-2xl font-bold text-amber-900 mb-2">💳 Gestion des Prêts</h1>
        <p className="text-amber-700">Suivi des prêts automatiques TIM MAXI et TIM BUSINESS</p>
      </div>

      {/* Statistiques */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-6 gap-4 mb-8">
        <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-lg p-4 border border-blue-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-blue-400 to-indigo-500 p-2 rounded-lg">
              <CreditCard className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-blue-700 font-medium">Total</p>
              <p className="text-xl font-semibold text-blue-900">{loanStats.total}</p>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-lg p-4 border border-green-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-green-400 to-emerald-500 p-2 rounded-lg">
              <CheckCircle className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-green-700 font-medium">Actifs</p>
              <p className="text-xl font-semibold text-green-900">{loanStats.active}</p>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-yellow-50 to-amber-50 rounded-lg p-4 border border-yellow-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-yellow-400 to-amber-500 p-2 rounded-lg">
              <Clock className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-yellow-700 font-medium">En retard</p>
              <p className="text-xl font-semibold text-yellow-900">{loanStats.overdue}</p>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-lg p-4 border border-red-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-red-400 to-rose-500 p-2 rounded-lg">
              <Ban className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-red-700 font-medium">Bloqués</p>
              <p className="text-xl font-semibold text-red-900">{loanStats.blocked}</p>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-purple-50 to-pink-50 rounded-lg p-4 border border-purple-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-purple-400 to-pink-500 p-2 rounded-lg">
              <DollarSign className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-purple-700 font-medium">Montant Total</p>
              <p className="text-lg font-semibold text-purple-900">{loanStats.totalAmount.toLocaleString()}</p>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-lg p-4 border border-amber-200 shadow-md">
          <div className="flex items-center space-x-3">
            <div className="bg-gradient-to-br from-amber-400 to-orange-500 p-2 rounded-lg">
              <DollarSign className="w-5 h-5 text-white" />
            </div>
            <div>
              <p className="text-sm text-amber-700 font-medium">Commissions</p>
              <p className="text-lg font-semibold text-amber-900">{loanStats.totalCommissions.toLocaleString()}</p>
            </div>
          </div>
        </div>
      </div>

      {/* Filtres et recherche avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/50 to-orange-50/50 rounded-lg border border-amber-200 shadow-lg p-6 mb-6">
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="flex-1">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
              <input
                type="text"
                placeholder="Rechercher par nom ou ID..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full pl-10 pr-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              />
            </div>
          </div>
          <div className="flex items-center space-x-3">
            <Filter className="w-4 h-4 text-gray-400" />
            <select
              value={selectedFilter}
              onChange={(e) => setSelectedFilter(e.target.value)}
              className="border border-amber-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
            >
              <option value="all">Tous les statuts</option>
              <option value="active">Actifs</option>
              <option value="overdue">En retard</option>
              <option value="repaid">Remboursés</option>
              <option value="blocked">Bloqués</option>
            </select>
          </div>
        </div>
      </div>

      {/* Liste des prêts avec thème marron doré */}
      <div className="bg-white rounded-lg border border-amber-200 shadow-lg">
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gradient-to-r from-amber-50 to-orange-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Emprunteur</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Montant</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Commission</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Total à rembourser</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Statut</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Échéance</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {loading ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center">
                    <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600 mb-4"></div>
                    <p className="text-amber-600">Chargement des prêts...</p>
                  </td>
                </tr>
              ) : error ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center">
                    <p className="text-red-600 font-medium mb-4">{error}</p>
                    <button
                      onClick={() => window.location.reload()}
                      className="px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700 transition-colors"
                    >
                      Réessayer
                    </button>
                  </td>
                </tr>
              ) : filteredLoans.length === 0 ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center">
                    <div className="inline-flex items-center justify-center w-16 h-16 bg-amber-100 rounded-full mb-4">
                      <CreditCard className="w-8 h-8 text-amber-600" />
                    </div>
                    <h3 className="text-lg font-semibold text-gray-900 mb-2">Aucun prêt</h3>
                    <p className="text-gray-600">
                      {searchTerm || selectedFilter !== 'all'
                        ? 'Aucun prêt ne correspond à vos critères de recherche'
                        : 'Les prêts automatiques TIM MAXI et TIM BUSINESS apparaîtront ici'}
                    </p>
                  </td>
                </tr>
              ) : (
                filteredLoans.map((loan) => (
                <tr key={loan.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div>
                      <p className="text-sm font-medium text-gray-900">{loan.userName}</p>
                      <p className="text-xs text-gray-500">{loan.userType.toUpperCase()}</p>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className="text-sm font-medium text-gray-900">{loan.amount.toLocaleString()} FCFA</span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className="text-sm text-gray-900">{loan.commission.toLocaleString()} FCFA</span>
                    <p className="text-xs text-gray-500">6%</p>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className="text-sm font-medium text-gray-900">{loan.totalAmount.toLocaleString()} FCFA</span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex items-center px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(loan.status)}`}>
                      {getStatusLabel(loan.status)}
                    </span>
                    {loan.daysOverdue && (
                      <p className="text-xs text-red-600 mt-1">{loan.daysOverdue} jours de retard</p>
                    )}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {new Date(loan.dueDate).toLocaleDateString('fr-FR')}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <button
                      onClick={() => handleViewDetails(loan)}
                      className="p-2 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors duration-200"
                      title="Voir les détails"
                    >
                      <Eye className="w-4 h-4" />
                    </button>
                  </td>
                </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Modal Détails */}
      {showDetailsModal && selectedLoan && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-2xl">
            <div className="p-6 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">Détails du Prêt</h3>
              <p className="text-gray-600">ID: {selectedLoan.id}</p>
            </div>
            <div className="p-6 space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700">Emprunteur</label>
                  <p className="text-gray-900">{selectedLoan.userName}</p>
                  <p className="text-sm text-gray-500">{selectedLoan.userType.toUpperCase()}</p>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700">Statut</label>
                  <span className={`inline-flex items-center px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(selectedLoan.status)}`}>
                    {getStatusLabel(selectedLoan.status)}
                  </span>
                </div>
              </div>
              <div className="grid grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700">Montant prêté</label>
                  <p className="text-gray-900">{selectedLoan.amount.toLocaleString()} FCFA</p>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700">Commission (6%)</label>
                  <p className="text-gray-900">{selectedLoan.commission.toLocaleString()} FCFA</p>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700">Total à rembourser</label>
                  <p className="text-gray-900 font-semibold">{selectedLoan.totalAmount.toLocaleString()} FCFA</p>
                </div>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700">Date de création</label>
                  <p className="text-gray-900">{new Date(selectedLoan.createdAt).toLocaleDateString('fr-FR')}</p>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700">Date d'échéance</label>
                  <p className="text-gray-900">{new Date(selectedLoan.dueDate).toLocaleDateString('fr-FR')}</p>
                </div>
              </div>
              {selectedLoan.repaidAt && (
                <div>
                  <label className="block text-sm font-medium text-gray-700">Date de remboursement</label>
                  <p className="text-gray-900">{new Date(selectedLoan.repaidAt).toLocaleDateString('fr-FR')}</p>
                </div>
              )}
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end">
              <button
                onClick={() => setShowDetailsModal(false)}
                className="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors duration-200"
              >
                Fermer
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default LoansManagement;
