import React, { useState, useEffect } from "react";
import {
  User,
  Phone,
  Mail,
  Calendar,
  MapPin,
  CreditCard,
  Clock,
  CheckCircle,
  XCircle,
  AlertCircle,
  Building2,
  Wallet,
  TrendingUp,
  FileText,
  Shield,
  Eye,
  Search,
  X,
  Image as ImageIcon,
} from "lucide-react";
import { getLoans } from "../../services/api";
import { API_CONFIG } from "../../config/api";
import AuthenticatedImage from "../../components/AuthenticatedImage";

interface LoanDetails {
  id: string;
  user_id: string;
  user_name: string;
  tim_type: string;
  amount: number;
  interest_rate: number;
  remaining_balance: number;
  status: string;
  due_date: string;
  created_at: string;
  document_recto?: string;
  document_verso?: string;
}

const LoansManagement: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedLoan, setSelectedLoan] = useState<LoanDetails | null>(null);
  const [isApproving, setIsApproving] = useState(false);
  const [isRejecting, setIsRejecting] = useState(false);
  const [loansData, setLoansData] = useState<LoanDetails[]>([]);
  const [loading, setLoading] = useState(true);

  // Récupérer les prêts depuis l'API
  useEffect(() => {
    const fetchLoans = async () => {
      try {
        setLoading(true);
        const response = await getLoans();
        // Filtrer uniquement les prêts en attente
        const pendingLoans = response.loans.filter((loan: any) => loan.status === 'pending');
        setLoansData(pendingLoans);
      } catch (error) {
        console.error('Erreur lors du chargement des prêts:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchLoans();
  }, []);

  const handleApprove = () => {
    setIsApproving(true);
    setTimeout(() => {
      alert("Prêt approuvé avec succès!");
      setIsApproving(false);
      setSelectedLoan(null);
    }, 1500);
  };

  const handleReject = () => {
    setIsRejecting(true);
    setTimeout(() => {
      alert("Prêt rejeté");
      setIsRejecting(false);
      setSelectedLoan(null);
    }, 1500);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("fr-FR", {
      day: "2-digit",
      month: "long",
      year: "numeric",
    });
  };

  const formatDateTime = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("fr-FR", {
      day: "2-digit",
      month: "long",
      year: "numeric",
      hour: "2-digit",
      minute: "2-digit",
    });
  };

  const formatCurrency = (amount: number, symbol: string) => {
    return `${amount.toLocaleString("fr-FR")} ${symbol}`;
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "pending":
        return "bg-yellow-100 text-yellow-800";
      case "approved":
        return "bg-green-100 text-green-800";
      case "rejected":
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case "pending":
        return "En attente";
      case "approved":
        return "Approuvé";
      case "rejected":
        return "Rejeté";
      default:
        return status;
    }
  };

  const getTimTypeColor = (type: string) => {
    switch (type) {
      case "TIM_MAXI":
        return "bg-purple-100 text-purple-800";
      case "TIM_BUSINESS":
        return "bg-orange-100 text-orange-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const filteredLoans = loansData.filter((loan) => {
    const searchLower = searchTerm.toLowerCase();
    return (
      loan.user_name.toLowerCase().includes(searchLower) ||
      loan.id.toLowerCase().includes(searchLower)
    );
  });

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50 p-6">
      <div className="max-w-7xl mx-auto space-y-6">
        {/* Header */}
        <div className="bg-gradient-to-br from-yellow-50 to-amber-50 rounded-2xl shadow-lg border border-yellow-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold mb-2 text-yellow-900">
                Gestion des Prêts
              </h1>
              <p className="text-yellow-600">
                {filteredLoans.length} demande
                {filteredLoans.length > 1 ? "s" : ""} de prêt
              </p>
            </div>
            <div className="bg-yellow-100 rounded-xl p-4 text-center">
              <p className="text-yellow-600 text-sm mb-1">Total</p>
              <p className="text-2xl font-bold text-yellow-900">
                {filteredLoans
                  .reduce((sum, loan) => sum + loan.amount, 0)
                  .toLocaleString()}{" "}
                FCFA
              </p>
            </div>
          </div>
        </div>

        {/* Barre de recherche */}
        <div className="bg-white rounded-xl shadow-md p-4 flex items-center space-x-3">
          <Search className="w-5 h-5 text-gray-400" />
          <input
            type="text"
            placeholder="Rechercher par nom, téléphone ou ID..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="flex-1 outline-none text-gray-700"
          />
        </div>

        {/* Tableau des prêts */}
        <div className="bg-white rounded-2xl shadow-lg overflow-hidden">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gradient-to-r from-gray-50 to-gray-100">
                <tr>
                  <th className="px-6 py-4 text-left text-sm font-semibold text-gray-700">
                    Client
                  </th>
                  <th className="px-6 py-4 text-left text-sm font-semibold text-gray-700">
                    Type de compte
                  </th>
                  <th className="px-6 py-4 text-left text-sm font-semibold text-gray-700">
                    Montant
                  </th>
                  <th className="px-6 py-4 text-left text-sm font-semibold text-gray-700">
                    Statut
                  </th>
                  <th className="px-6 py-4 text-left text-sm font-semibold text-gray-700">
                    Date
                  </th>
                  <th className="px-6 py-4 text-center text-sm font-semibold text-gray-700">
                    Action
                  </th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {loading ? (
                  <tr>
                    <td colSpan={6} className="px-6 py-12 text-center">
                      <Clock className="w-12 h-12 text-gray-300 mx-auto mb-3 animate-spin" />
                      <p className="text-gray-500">Chargement...</p>
                    </td>
                  </tr>
                ) : filteredLoans.length === 0 ? (
                  <tr>
                    <td colSpan={6} className="px-6 py-12 text-center">
                      <Clock className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                      <p className="text-gray-500">Aucun prêt trouvé</p>
                    </td>
                  </tr>
                ) : (
                  filteredLoans.map((loan) => (
                    <tr
                      key={loan.id}
                      className="hover:bg-gray-50 transition-colors"
                    >
                      <td className="px-6 py-4">
                        <div>
                          <p className="font-semibold text-gray-900">
                            {loan.user_name}
                          </p>
                          <p className="text-sm text-gray-500">
                            {loan.user_id}
                          </p>
                        </div>
                      </td>
                      <td className="px-6 py-4">
                        <span
                          className={`inline-flex px-3 py-1 rounded-full text-xs font-medium ${getTimTypeColor(
                            loan.tim_type
                          )}`}
                        >
                          {loan.tim_type}
                        </span>
                      </td>
                      <td className="px-6 py-4">
                        <p className="font-semibold text-gray-900">
                          {formatCurrency(loan.amount, "FCFA")}
                        </p>
                      </td>
                      <td className="px-6 py-4">
                        <span
                          className={`inline-flex px-3 py-1 rounded-full text-xs font-medium ${getStatusColor(
                            loan.status
                          )}`}
                        >
                          {getStatusLabel(loan.status)}
                        </span>
                      </td>
                      <td className="px-6 py-4">
                        <p className="text-sm text-gray-600">
                          {formatDate(loan.created_at)}
                        </p>
                      </td>
                      <td className="px-6 py-4 text-center">
                        <button
                          onClick={() => setSelectedLoan(loan)}
                          className="inline-flex items-center space-x-2 bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-lg transition-colors"
                        >
                          <Eye className="w-4 h-4" />
                          <span>Voir</span>
                        </button>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      </div>

      {/* Modal de détails */}
      {selectedLoan && (
        <div
          className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50"
          onClick={() => setSelectedLoan(null)}
        >
          <div
            className="bg-white rounded-2xl shadow-2xl max-w-4xl w-full max-h-[90vh] overflow-y-auto"
            onClick={(e) => e.stopPropagation()}
          >
            {/* Header du modal */}
            <div className="sticky top-0 bg-gradient-to-br from-amber-600 to-yellow-700 text-white p-6 flex items-center justify-between rounded-t-2xl">
              <div>
                <h2 className="text-2xl font-bold mb-1">Détails du Prêt</h2>
                <p className="text-amber-100">
                  Vérifiez les informations avant validation
                </p>
              </div>
              <button
                onClick={() => setSelectedLoan(null)}
                className="p-2 hover:bg-white hover:bg-opacity-20 rounded-lg transition-colors"
              >
                <X className="w-6 h-6" />
              </button>
            </div>

            <div className="p-6 space-y-6">
              {/* Montant principal */}
              <div className="bg-gradient-to-br from-amber-50 to-yellow-50 rounded-xl p-6">
                <div className="flex items-center justify-between flex-wrap gap-4">
                  <div>
                    <p className="text-amber-700 mb-2 font-medium">
                      Montant Demandé
                    </p>
                    <h3 className="text-4xl font-bold text-gray-900 mb-3">
                      {formatCurrency(selectedLoan.amount, "FCFA")}
                    </h3>
                    <div className="flex items-center space-x-4 text-sm">
                      <div className="flex items-center space-x-2 text-gray-600">
                        <TrendingUp className="w-4 h-4" />
                        <span>
                          Taux: {(selectedLoan.interest_rate * 100).toFixed(0)}%
                        </span>
                      </div>
                      <div className="flex items-center space-x-2 text-gray-600">
                        <Wallet className="w-4 h-4" />
                        <span>
                          À rembourser:{" "}
                          {formatCurrency(selectedLoan.remaining_balance, "FCFA")}
                        </span>
                      </div>
                    </div>
                  </div>
                  <div className="bg-white rounded-xl p-4 text-center shadow-sm">
                    <p className="text-gray-600 text-sm mb-1">Intérêts</p>
                    <p className="text-2xl font-bold text-amber-600">
                      +
                      {formatCurrency(
                        selectedLoan.remaining_balance - selectedLoan.amount,
                        "FCFA"
                      )}
                    </p>
                  </div>
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {/* Informations Client */}
                <div className="bg-gray-50 rounded-xl p-6 space-y-4">
                  <div className="flex items-center space-x-3 pb-3 border-b border-gray-200">
                    <div className="p-2 bg-amber-100 rounded-lg">
                      <User className="w-5 h-5 text-amber-700" />
                    </div>
                    <h4 className="text-lg font-bold text-gray-900">
                      Informations Client
                    </h4>
                  </div>

                  <div className="space-y-3">
                    <div className="flex items-start space-x-3">
                      <User className="w-5 h-5 text-gray-400 mt-1" />
                      <div className="flex-1">
                        <p className="text-xs text-gray-500">Nom complet</p>
                        <p className="text-gray-900 font-semibold">
                          {selectedLoan.user_name}
                        </p>
                      </div>
                    </div>

                    <div className="flex items-start space-x-3">
                      <CreditCard className="w-5 h-5 text-gray-400 mt-1" />
                      <div className="flex-1">
                        <p className="text-xs text-gray-500">ID Utilisateur</p>
                        <p className="text-gray-900 font-mono text-xs">
                          {selectedLoan.user_id}
                        </p>
                      </div>
                    </div>

                    <div className="flex items-start space-x-3">
                      <Building2 className="w-5 h-5 text-gray-400 mt-1" />
                      <div className="flex-1">
                        <p className="text-xs text-gray-500">Type de compte</p>
                        <span
                          className={`inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${getTimTypeColor(
                            selectedLoan.tim_type
                          )}`}
                        >
                          {selectedLoan.tim_type}
                        </span>
                      </div>
                    </div>
                  </div>
                </div>

                {/* Détails du Prêt */}
                <div className="bg-gray-50 rounded-xl p-6 space-y-4">
                  <div className="flex items-center space-x-3 pb-3 border-b border-gray-200">
                    <div className="p-2 bg-yellow-100 rounded-lg">
                      <FileText className="w-5 h-5 text-yellow-700" />
                    </div>
                    <h4 className="text-lg font-bold text-gray-900">
                      Détails de la Demande
                    </h4>
                  </div>

                  <div className="space-y-3">
                    <div className="flex items-start space-x-3">
                      <FileText className="w-5 h-5 text-gray-400 mt-1" />
                      <div className="flex-1">
                        <p className="text-xs text-gray-500">ID du prêt</p>
                        <p className="text-gray-900 font-mono text-xs break-all">
                          {selectedLoan.id}
                        </p>
                      </div>
                    </div>

                    <div className="flex items-start space-x-3">
                      <Clock className="w-5 h-5 text-gray-400 mt-1" />
                      <div className="flex-1">
                        <p className="text-xs text-gray-500">Date de demande</p>
                        <p className="text-gray-900 font-semibold">
                          {formatDateTime(selectedLoan.created_at)}
                        </p>
                      </div>
                    </div>

                    <div className="flex items-start space-x-3">
                      <Calendar className="w-5 h-5 text-gray-400 mt-1" />
                      <div className="flex-1">
                        <p className="text-xs text-gray-500">Date d'échéance</p>
                        <p className="text-gray-900 font-semibold">
                          {formatDateTime(selectedLoan.due_date)}
                        </p>
                      </div>
                    </div>

                    <div className="flex items-start space-x-3">
                      <Shield className="w-5 h-5 text-gray-400 mt-1" />
                      <div className="flex-1">
                        <p className="text-xs text-gray-500">Statut actuel</p>
                        <span
                          className={`inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${getStatusColor(
                            selectedLoan.status
                          )}`}
                        >
                          {getStatusLabel(selectedLoan.status)}
                        </span>
                      </div>
                    </div>
                  </div>

                  {/* Résumé Financier */}
                  <div className="bg-white rounded-lg p-4 space-y-2 mt-4">
                    <h5 className="font-semibold text-gray-900 mb-2 text-sm">
                      Résumé Financier
                    </h5>
                    <div className="flex justify-between items-center text-sm">
                      <span className="text-gray-600">Montant principal</span>
                      <span className="font-semibold text-gray-900">
                        {formatCurrency(selectedLoan.amount, "FCFA")}
                      </span>
                    </div>
                    <div className="flex justify-between items-center text-sm">
                      <span className="text-gray-600">
                        Intérêts (
                        {(selectedLoan.interest_rate * 100).toFixed(0)}%)
                      </span>
                      <span className="font-semibold text-gray-900">
                        +
                        {formatCurrency(
                          selectedLoan.remaining_balance - selectedLoan.amount,
                          "FCFA"
                        )}
                      </span>
                    </div>
                    <div className="border-t border-gray-200 pt-2 flex justify-between items-center">
                      <span className="font-semibold text-gray-900">
                        Total à rembourser
                      </span>
                      <span className="font-bold text-lg text-amber-600">
                        {formatCurrency(selectedLoan.remaining_balance, "FCFA")}
                      </span>
                    </div>
                  </div>
                </div>
              </div>

              {/* Documents d'identité */}
              {(selectedLoan.document_recto || selectedLoan.document_verso) && (
                <div className="bg-gray-50 rounded-xl p-6 space-y-4">
                  <div className="flex items-center space-x-3 pb-3 border-b border-gray-200">
                    <div className="p-2 bg-amber-100 rounded-lg">
                      <ImageIcon className="w-5 h-5 text-amber-700" />
                    </div>
                    <h4 className="text-lg font-bold text-gray-900">
                      Documents d'identité (CNI)
                    </h4>
                  </div>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {selectedLoan.document_recto && (
                      <div className="space-y-2">
                        <p className="text-xs text-gray-500 font-medium">Recto CNI</p>
                        <div className="relative group">
                          <AuthenticatedImage
                            src={`${API_CONFIG.BASE_URL}/admin/loans/${selectedLoan.id}/documents/recto`}
                            alt="Recto CNI"
                            className="w-full rounded-lg border-2 border-gray-300 hover:border-amber-500 transition-colors cursor-pointer shadow-sm"
                            onClick={() => {
                              // Ouvrir dans un nouvel onglet avec le token
                              const token = localStorage.getItem('tim_access_token');
                              const url = `${API_CONFIG.BASE_URL}/admin/loans/${selectedLoan.id}/documents/recto`;
                              fetch(url, {
                                headers: { 'Authorization': `Bearer ${token}` }
                              })
                              .then(res => res.blob())
                              .then(blob => {
                                const blobUrl = URL.createObjectURL(blob);
                                window.open(blobUrl, '_blank');
                              });
                            }}
                          />
                          <div className="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-10 rounded-lg transition-all flex items-center justify-center pointer-events-none">
                            <Eye className="w-8 h-8 text-white opacity-0 group-hover:opacity-100 transition-opacity" />
                          </div>
                        </div>
                      </div>
                    )}
                    {selectedLoan.document_verso && (
                      <div className="space-y-2">
                        <p className="text-xs text-gray-500 font-medium">Verso CNI</p>
                        <div className="relative group">
                          <AuthenticatedImage
                            src={`${API_CONFIG.BASE_URL}/admin/loans/${selectedLoan.id}/documents/verso`}
                            alt="Verso CNI"
                            className="w-full rounded-lg border-2 border-gray-300 hover:border-amber-500 transition-colors cursor-pointer shadow-sm"
                            onClick={() => {
                              // Ouvrir dans un nouvel onglet avec le token
                              const token = localStorage.getItem('tim_access_token');
                              const url = `${API_CONFIG.BASE_URL}/admin/loans/${selectedLoan.id}/documents/verso`;
                              fetch(url, {
                                headers: { 'Authorization': `Bearer ${token}` }
                              })
                              .then(res => res.blob())
                              .then(blob => {
                                const blobUrl = URL.createObjectURL(blob);
                                window.open(blobUrl, '_blank');
                              });
                            }}
                          />
                          <div className="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-10 rounded-lg transition-all flex items-center justify-center pointer-events-none">
                            <Eye className="w-8 h-8 text-white opacity-0 group-hover:opacity-100 transition-opacity" />
                          </div>
                        </div>
                      </div>
                    )}
                  </div>
                  <p className="text-xs text-gray-500 italic">
                    💡 Cliquez sur une image pour l'ouvrir en plein écran
                  </p>
                </div>
              )}

              {/* Alerte */}
              <div className="bg-amber-50 border border-amber-200 rounded-xl p-4 flex items-start space-x-3">
                <AlertCircle className="w-5 h-5 text-amber-700 mt-1 flex-shrink-0" />
                <div>
                  <h5 className="font-semibold text-amber-900 mb-1">
                    Vérification requise
                  </h5>
                  <p className="text-amber-800 text-sm">
                    Assurez-vous que toutes les informations sont correctes
                    avant d'approuver cette demande. Le montant sera crédité
                    immédiatement sur le compte du client.
                  </p>
                </div>
              </div>

              {/* Boutons d'action */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 pt-4">
                <button
                  onClick={handleReject}
                  disabled={isRejecting}
                  className="flex items-center justify-center space-x-2 bg-gray-600 hover:bg-gray-700 disabled:bg-gray-400 text-white px-6 py-3 rounded-xl font-semibold transition-all"
                >
                  <XCircle className="w-5 h-5" />
                  <span>{isRejecting ? "Rejet en cours..." : "Rejeter"}</span>
                </button>

                <button
                  onClick={handleApprove}
                  disabled={isApproving}
                  className="flex items-center justify-center space-x-2 bg-gradient-to-r from-amber-600 to-yellow-600 hover:from-amber-700 hover:to-yellow-700 disabled:from-amber-400 disabled:to-yellow-400 text-white px-6 py-3 rounded-xl font-semibold transition-all shadow-lg hover:shadow-xl"
                >
                  <CheckCircle className="w-5 h-5" />
                  <span>
                    {isApproving ? "Approbation en cours..." : "Approuver"}
                  </span>
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default LoansManagement;
