import React, { useState } from 'react';
import { 
  CreditCard, 
  Plus, 
  Calendar, 
  DollarSign, 
  Clock, 
  CheckCircle, 
  XCircle,
  AlertCircle,
  TrendingUp,
  Calculator,
  FileText,
  Filter
} from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';

const Loans: React.FC = () => {
  const { user } = useAuth();
  const [activeTab, setActiveTab] = useState('my-loans');
  const [showLoanCalculator, setShowLoanCalculator] = useState(false);
  const [loanAmount, setLoanAmount] = useState(5000);
  const [loanDuration, setLoanDuration] = useState(12);

  const isAdmin = user?.role === 'admin' || user?.role === 'manager';

  // Mock loan data
  const myLoans = [
    {
      id: 'loan-001',
      amount: 5000,
      interestRate: 8.5,
      duration: 12,
      monthlyPayment: 435.85,
      remainingBalance: 3500,
      nextPaymentDate: '2024-02-15',
      status: 'active',
      purpose: 'Développement d\'entreprise',
      createdAt: '2024-01-01'
    },
    {
      id: 'loan-002',
      amount: 2500,
      interestRate: 7.5,
      duration: 6,
      monthlyPayment: 435.12,
      remainingBalance: 0,
      nextPaymentDate: null,
      status: 'completed',
      purpose: 'Achat d\'équipement',
      createdAt: '2023-08-15'
    }
  ];

  const allLoans = [
    ...myLoans,
    {
      id: 'loan-003',
      userId: 'user-002',
      userName: 'Marie Dubois',
      amount: 10000,
      interestRate: 9.0,
      duration: 24,
      monthlyPayment: 456.79,
      remainingBalance: 8500,
      nextPaymentDate: '2024-02-20',
      status: 'active',
      purpose: 'Rénovation domiciliaire',
      createdAt: '2024-01-10'
    },
    {
      id: 'loan-004',
      userId: 'user-003',
      userName: 'Pierre Martin',
      amount: 3000,
      interestRate: 8.0,
      duration: 12,
      monthlyPayment: 260.45,
      remainingBalance: 3000,
      nextPaymentDate: null,
      status: 'pending',
      purpose: 'Formation professionnelle',
      createdAt: '2024-01-20'
    }
  ];

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'active':
        return <CheckCircle className="w-5 h-5 text-green-600" />;
      case 'completed':
        return <CheckCircle className="w-5 h-5 text-blue-600" />;
      case 'pending':
        return <Clock className="w-5 h-5 text-yellow-600" />;
      case 'overdue':
        return <AlertCircle className="w-5 h-5 text-red-600" />;
      default:
        return <XCircle className="w-5 h-5 text-gray-600" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
        return 'bg-green-100 text-green-800';
      case 'completed':
        return 'bg-blue-100 text-blue-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'overdue':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const calculateLoan = () => {
    const monthlyRate = 8.5 / 100 / 12;
    const monthlyPayment = (loanAmount * monthlyRate * Math.pow(1 + monthlyRate, loanDuration)) / 
                          (Math.pow(1 + monthlyRate, loanDuration) - 1);
    const totalPayment = monthlyPayment * loanDuration;
    const totalInterest = totalPayment - loanAmount;

    return {
      monthlyPayment: monthlyPayment.toFixed(2),
      totalPayment: totalPayment.toFixed(2),
      totalInterest: totalInterest.toFixed(2)
    };
  };

  const loanCalculation = calculateLoan();

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">
            {isAdmin ? 'Gestion des Prêts' : 'Mes Prêts'}
          </h1>
          <p className="text-gray-600 mt-2">
            {isAdmin 
              ? 'Gérez les demandes de prêts et suivez les remboursements' 
              : 'Suivez vos prêts et demandez de nouveaux financements'
            }
          </p>
        </div>
        <div className="flex space-x-4">
          <button 
            onClick={() => setShowLoanCalculator(!showLoanCalculator)}
            className="border border-gray-300 text-gray-700 hover:bg-gray-50 px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2"
          >
            <Calculator className="w-4 h-4" />
            <span>Calculateur</span>
          </button>
          <button className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2">
            <Plus className="w-4 h-4" />
            <span>{isAdmin ? 'Nouveau Prêt' : 'Demander un Prêt'}</span>
          </button>
        </div>
      </div>

      {/* Loan Calculator */}
      {showLoanCalculator && (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-6">Calculateur de Prêt</h3>
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <div className="space-y-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Montant du prêt
                </label>
                <div className="relative">
                  <DollarSign className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                  <input
                    type="number"
                    value={loanAmount}
                    onChange={(e) => setLoanAmount(Number(e.target.value))}
                    className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="5000"
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Durée (mois)
                </label>
                <select
                  value={loanDuration}
                  onChange={(e) => setLoanDuration(Number(e.target.value))}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value={6}>6 mois</option>
                  <option value={12}>12 mois</option>
                  <option value={18}>18 mois</option>
                  <option value={24}>24 mois</option>
                  <option value={36}>36 mois</option>
                </select>
              </div>
            </div>
            <div className="bg-gray-50 rounded-lg p-6">
              <h4 className="font-semibold text-gray-900 mb-4">Résumé du Prêt</h4>
              <div className="space-y-3">
                <div className="flex justify-between">
                  <span className="text-gray-600">Paiement mensuel:</span>
                  <span className="font-semibold">{loanCalculation.monthlyPayment} USD</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Total à rembourser:</span>
                  <span className="font-semibold">{loanCalculation.totalPayment} USD</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Total des intérêts:</span>
                  <span className="font-semibold text-red-600">{loanCalculation.totalInterest} USD</span>
                </div>
                <div className="flex justify-between pt-3 border-t border-gray-200">
                  <span className="text-gray-600">Taux d'intérêt:</span>
                  <span className="font-semibold">8.5% / an</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Prêts Actifs</p>
              <p className="text-3xl font-bold text-gray-900">
                {isAdmin ? '24' : myLoans.filter(l => l.status === 'active').length}
              </p>
            </div>
            <div className="w-12 h-12 bg-blue-50 rounded-lg flex items-center justify-center">
              <CreditCard className="w-6 h-6 text-blue-600" />
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Montant Total</p>
              <p className="text-3xl font-bold text-gray-900">
                {isAdmin ? '$847K' : '$7.5K'}
              </p>
            </div>
            <div className="w-12 h-12 bg-green-50 rounded-lg flex items-center justify-center">
              <DollarSign className="w-6 h-6 text-green-600" />
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">En Attente</p>
              <p className="text-3xl font-bold text-gray-900">
                {isAdmin ? '8' : '0'}
              </p>
            </div>
            <div className="w-12 h-12 bg-yellow-50 rounded-lg flex items-center justify-center">
              <Clock className="w-6 h-6 text-yellow-600" />
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Taux de Remboursement</p>
              <p className="text-3xl font-bold text-gray-900">98.5%</p>
            </div>
            <div className="w-12 h-12 bg-purple-50 rounded-lg flex items-center justify-center">
              <TrendingUp className="w-6 h-6 text-purple-600" />
            </div>
          </div>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6">
            {[
              { id: 'my-loans', label: isAdmin ? 'Tous les Prêts' : 'Mes Prêts' },
              ...(isAdmin ? [
                { id: 'pending', label: 'En Attente' },
                { id: 'overdue', label: 'En Retard' }
              ] : []),
              { id: 'history', label: 'Historique' }
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-200 ${
                  activeTab === tab.id
                    ? 'border-blue-500 text-blue-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
              >
                {tab.label}
              </button>
            ))}
          </nav>
        </div>

        <div className="p-6">
          {/* Filters */}
          <div className="flex items-center justify-between mb-6">
            <div className="flex items-center space-x-4">
              <button className="flex items-center space-x-2 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors duration-200">
                <Filter className="w-4 h-4" />
                <span>Filtrer</span>
              </button>
              <select className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                <option>Tous les statuts</option>
                <option>Actif</option>
                <option>Terminé</option>
                <option>En attente</option>
              </select>
            </div>
            <div className="text-sm text-gray-600">
              {isAdmin ? allLoans.length : myLoans.length} prêt(s) trouvé(s)
            </div>
          </div>

          {/* Loans List */}
          <div className="space-y-4">
            {(isAdmin ? allLoans : myLoans).map((loan) => (
              <div key={loan.id} className="border border-gray-200 rounded-lg p-6 hover:shadow-md transition-shadow duration-200">
                <div className="flex items-center justify-between mb-4">
                  <div className="flex items-center space-x-4">
                    <div className="w-12 h-12 bg-gray-100 rounded-lg flex items-center justify-center">
                      {getStatusIcon(loan.status)}
                    </div>
                    <div>
                      <h3 className="font-semibold text-gray-900">
                        {loan.purpose}
                        {isAdmin && 'userName' in loan && (
                          <span className="text-gray-500 ml-2">- {loan.userName}</span>
                        )}
                      </h3>
                      <p className="text-sm text-gray-600">
                        Prêt #{loan.id} • Créé le {new Date(loan.createdAt).toLocaleDateString('fr-FR')}
                      </p>
                    </div>
                  </div>
                  <span className={`inline-flex px-3 py-1 text-sm font-medium rounded-full ${getStatusColor(loan.status)}`}>
                    {loan.status === 'active' ? 'Actif' :
                     loan.status === 'completed' ? 'Terminé' :
                     loan.status === 'pending' ? 'En attente' : loan.status}
                  </span>
                </div>

                <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
                  <div>
                    <p className="text-sm text-gray-600">Montant</p>
                    <p className="font-semibold">{loan.amount.toLocaleString()} USD</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Paiement mensuel</p>
                    <p className="font-semibold">{loan.monthlyPayment.toFixed(2)} USD</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Solde restant</p>
                    <p className="font-semibold">{loan.remainingBalance.toLocaleString()} USD</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Prochain paiement</p>
                    <p className="font-semibold">
                      {loan.nextPaymentDate 
                        ? new Date(loan.nextPaymentDate).toLocaleDateString('fr-FR')
                        : 'N/A'
                      }
                    </p>
                  </div>
                </div>

                {loan.status === 'active' && (
                  <div className="bg-gray-50 rounded-lg p-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-2">
                        <Calendar className="w-4 h-4 text-gray-500" />
                        <span className="text-sm text-gray-600">
                          Durée: {loan.duration} mois • Taux: {loan.interestRate}%
                        </span>
                      </div>
                      <div className="flex space-x-2">
                        <button className="text-blue-600 hover:text-blue-700 text-sm font-medium">
                          Voir détails
                        </button>
                        {!isAdmin && (
                          <button className="text-green-600 hover:text-green-700 text-sm font-medium">
                            Rembourser
                          </button>
                        )}
                        {isAdmin && loan.status === 'pending' && (
                          <>
                            <button className="text-green-600 hover:text-green-700 text-sm font-medium">
                              Approuver
                            </button>
                            <button className="text-red-600 hover:text-red-700 text-sm font-medium">
                              Rejeter
                            </button>
                          </>
                        )}
                      </div>
                    </div>
                  </div>
                )}
              </div>
            ))}
          </div>

          {(isAdmin ? allLoans : myLoans).length === 0 && (
            <div className="text-center py-12">
              <CreditCard className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Aucun prêt trouvé</h3>
              <p className="text-gray-600">
                {isAdmin 
                  ? 'Aucune demande de prêt pour le moment'
                  : 'Vous n\'avez pas encore de prêt actif'
                }
              </p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default Loans;