import React, { useState, useEffect } from "react";
import {
  RefreshCw,
  Calculator,
  Clock,
  AlertTriangle,
  History,
  Eye,
  CheckCircle,
  DollarSign,
  TrendingUp,
  Search,
} from "lucide-react";
import { getLoanRepayments } from "../../services/api";

interface LoanRepayment {
  id: string;
  userId: string;
  userName: string;
  timType: "TIM_MAXI" | "TIM_BUSINESS" | "TIM_MINI";
  loanAmount: number;
  interestRate: number;
  penaltyRate: number;
  totalAmount: number;
  paidAmount: number;
  remainingAmount: number;
  dueDate: string;
  daysOverdue: number;
  penaltyAmount?: number;
  status: "pending" | "overdue" | "completed" | "partial";
  lastPayment?: string;
  paymentHistory: {
    date: string;
    amount: number;
    type: "payment" | "penalty";
    reference?: string;
  }[];
  createdAt: string;
}

const LoanRepayments: React.FC = () => {
  const [activeTab, setActiveTab] = useState<"pending" | "overdue" | "history">(
    "pending"
  );
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedLoan, setSelectedLoan] = useState<string | null>(null);
  const [calculatorAmount, setCalculatorAmount] = useState<number>(1000);
  const [repayments, setRepayments] = useState<LoanRepayment[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Récupérer les remboursements depuis l'API
  useEffect(() => {
    const fetchRepayments = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await getLoanRepayments();

        // Transformer les données de l'API au format attendu par le frontend
        const repaymentsData = (response ?? []).map((repayment: any) => ({
          id: repayment.loan_id,
          userId: repayment.user.id,
          userName: repayment.user.full_name,
          timType: repayment.user.account_type,
          loanAmount: repayment.amount,
          interestRate: repayment.interest_rate * 100, // 0.06 → 6
          penaltyRate: 15,
          totalAmount:
            repayment.amount + repayment.amount * repayment.interest_rate,
          paidAmount:
            repayment.amount +
            repayment.amount * repayment.interest_rate -
            repayment.remaining_balance,
          remainingAmount: repayment.remaining_balance,
          dueDate: repayment.due_date,
          daysOverdue:
            repayment.status === "overdue"
              ? Math.max(
                  0,
                  Math.floor(
                    (Date.now() - new Date(repayment.due_date).getTime()) /
                      (1000 * 60 * 60 * 24)
                  )
                )
              : 0,
          penaltyAmount:
            repayment.status === "overdue"
              ? (repayment.amount +
                  repayment.amount * repayment.interest_rate) *
                0.15
              : 0,
          status:
            repayment.status === "approved"
              ? "pending"
              : repayment.status === "active"
              ? "partial"
              : repayment.status,
          lastPayment: null,
          paymentHistory: [],
          createdAt: repayment.created_at,
        }));

        setRepayments(repaymentsData);
      } catch (err: any) {
        console.error(
          "Erreur lors de la récupération des remboursements:",
          err
        );
        setError(
          err.message || "Erreur lors de la récupération des remboursements"
        );
      } finally {
        setLoading(false);
      }
    };

    fetchRepayments();
  }, []);

  // Calculatrice de prêt SOS
  const calculateLoanDetails = (amount: number) => {
    const interest = amount * 0.06; // 6%
    const totalWithInterest = amount + interest;
    const penalty = totalWithInterest * 0.15; // 15%
    const totalWithPenalty = totalWithInterest + penalty;

    return {
      principal: amount,
      interest: interest,
      totalWithInterest: totalWithInterest,
      penalty: penalty,
      totalWithPenalty: totalWithPenalty,
    };
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "pending":
        return "bg-yellow-100 text-yellow-800";
      case "overdue":
        return "bg-red-100 text-red-800";
      case "completed":
        return "bg-green-100 text-green-800";
      case "partial":
        return "bg-blue-100 text-blue-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case "pending":
        return "En attente";
      case "overdue":
        return "En retard";
      case "completed":
        return "Terminé";
      case "partial":
        return "Partiel";
      default:
        return status;
    }
  };

  const getTabData = () => {
    switch (activeTab) {
      case "pending":
        return repayments.filter(
          (r) => r.status === "pending" || r.status === "partial"
        );
      case "overdue":
        return repayments.filter((r) => r.status === "overdue");
      case "history":
        return repayments.filter((r) => r.status === "completed");
      default:
        return [];
    }
  };

  const filteredData = getTabData().filter(
    (repayment) =>
      repayment.userName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      repayment.id.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const loanCalculation = calculateLoanDetails(calculatorAmount);

  const totalPending = repayments.filter(
    (r) => r.status === "pending" || r.status === "partial"
  ).length;
  const totalOverdue = repayments.filter((r) => r.status === "overdue").length;
  const totalCompleted = repayments.filter(
    (r) => r.status === "completed"
  ).length;
  const totalAmount = repayments.reduce((sum, r) => sum + r.remainingAmount, 0);

  return (
    <div className="space-y-6">
      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div className="p-3 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg shadow-md">
              <RefreshCw className="w-6 h-6 text-white" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-amber-900">
                💰 Remboursements Prêts SOS
              </h1>
              <p className="text-amber-700">
                Gestion des petits prêts d'urgence (500-10,000 FCFA)
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* Calculatrice de Prêt SOS avec thème marron doré */}
      <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-lg border border-amber-200 p-6">
        <div className="flex items-center space-x-2 mb-4">
          <Calculator className="w-5 h-5 text-amber-600" />
          <h3 className="text-lg font-semibold text-amber-900">
            Calculatrice Prêt SOS
          </h3>
        </div>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Montant du prêt (500 - 10,000 FCFA)
            </label>
            <input
              type="number"
              min="500"
              max="10000"
              step="100"
              value={calculatorAmount}
              onChange={(e) =>
                setCalculatorAmount(
                  Math.max(
                    500,
                    Math.min(10000, parseInt(e.target.value) || 500)
                  )
                )
              }
              className="w-full px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
            />
          </div>
          <div className="space-y-3">
            <div className="flex justify-between">
              <span className="text-sm text-gray-600">Montant principal:</span>
              <span className="font-medium">
                {loanCalculation.principal.toLocaleString()} FCFA
              </span>
            </div>
            <div className="flex justify-between">
              <span className="text-sm text-gray-600">Intérêts (6%):</span>
              <span className="font-medium text-blue-600">
                +{loanCalculation.interest.toLocaleString()} FCFA
              </span>
            </div>
            <div className="flex justify-between border-t pt-2">
              <span className="text-sm font-medium text-gray-900">
                Total à rembourser (3 jours):
              </span>
              <span className="font-bold text-green-600">
                {loanCalculation.totalWithInterest.toLocaleString()} FCFA
              </span>
            </div>
            <div className="flex justify-between text-red-600">
              <span className="text-sm">Pénalité après 3 jours (15%):</span>
              <span className="font-medium">
                +{loanCalculation.penalty.toLocaleString()} FCFA
              </span>
            </div>
            <div className="flex justify-between border-t pt-2 text-red-600">
              <span className="text-sm font-medium">Total avec pénalité:</span>
              <span className="font-bold">
                {loanCalculation.totalWithPenalty.toLocaleString()} FCFA
              </span>
            </div>
          </div>
        </div>
      </div>

      {/* Statistiques avec thème marron doré */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-gradient-to-br from-yellow-50 to-amber-50 rounded-xl shadow-lg border border-yellow-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-yellow-700 font-medium">En Attente</p>
              <p className="text-2xl font-bold text-yellow-900">
                {totalPending}
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-yellow-400 to-amber-500 rounded-lg flex items-center justify-center shadow-md">
              <Clock className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-xl shadow-lg border border-red-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-red-700 font-medium">En Retard</p>
              <p className="text-2xl font-bold text-red-900">{totalOverdue}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-red-400 to-rose-500 rounded-lg flex items-center justify-center shadow-md">
              <AlertTriangle className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-lg border border-green-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-green-700 font-medium">Terminés</p>
              <p className="text-2xl font-bold text-green-900">
                {totalCompleted}
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-green-400 to-emerald-500 rounded-lg flex items-center justify-center shadow-md">
              <CheckCircle className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-purple-50 to-indigo-50 rounded-xl shadow-lg border border-purple-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-purple-700 font-medium">
                Montant Restant
              </p>
              <p className="text-2xl font-bold text-purple-900">
                {totalAmount.toLocaleString()} FCFA
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-purple-400 to-indigo-500 rounded-lg flex items-center justify-center shadow-md">
              <DollarSign className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Onglets avec thème marron doré */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200">
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6">
            <button
              onClick={() => setActiveTab("pending")}
              className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors ${
                activeTab === "pending"
                  ? "border-amber-500 text-amber-600"
                  : "border-transparent text-gray-500 hover:text-gray-700 hover:border-amber-300"
              }`}
            >
              <div className="flex items-center space-x-2">
                <Clock className="w-4 h-4" />
                <span>En Attente ({totalPending})</span>
              </div>
            </button>
            <button
              onClick={() => setActiveTab("overdue")}
              className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors ${
                activeTab === "overdue"
                  ? "border-red-500 text-red-600"
                  : "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
              }`}
            >
              <div className="flex items-center space-x-2">
                <AlertTriangle className="w-4 h-4" />
                <span>En Retard ({totalOverdue})</span>
              </div>
            </button>
            <button
              onClick={() => setActiveTab("history")}
              className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors ${
                activeTab === "history"
                  ? "border-green-500 text-green-600"
                  : "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
              }`}
            >
              <div className="flex items-center space-x-2">
                <History className="w-4 h-4" />
                <span>Historique ({totalCompleted})</span>
              </div>
            </button>
          </nav>
        </div>

        {/* Filtres */}
        <div className="p-4 border-b border-gray-200">
          <div className="flex items-center space-x-4">
            <div className="relative flex-1">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
              <input
                type="text"
                placeholder="Rechercher par nom ou ID..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 pr-4 py-2 w-full border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              />
            </div>
          </div>
        </div>

        {/* Liste des remboursements */}
        <div className="divide-y divide-gray-200">
          {loading ? (
            <div className="p-12 text-center">
              <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600"></div>
              <p className="mt-4 text-amber-600">
                Chargement des remboursements...
              </p>
            </div>
          ) : error ? (
            <div className="p-12 text-center">
              <p className="text-red-600 font-medium">{error}</p>
              <button
                onClick={() => window.location.reload()}
                className="mt-4 px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700"
              >
                Réessayer
              </button>
            </div>
          ) : filteredData.length === 0 ? (
            <div className="p-12 text-center">
              <div className="inline-flex items-center justify-center w-16 h-16 bg-amber-100 rounded-full mb-4">
                <RefreshCw className="w-8 h-8 text-amber-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900 mb-2">
                Aucun remboursement
              </h3>
              <p className="text-gray-600">
                {searchTerm
                  ? "Aucun résultat pour votre recherche"
                  : "Aucun remboursement trouvé pour cette catégorie"}
              </p>
            </div>
          ) : (
            filteredData.map((repayment) => (
              <div
                key={repayment.id}
                className="p-6 hover:bg-amber-50/30 transition-colors"
              >
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-4">
                    <div
                      className={`w-12 h-12 rounded-full flex items-center justify-center ${
                        repayment.status === "overdue"
                          ? "bg-red-100"
                          : repayment.status === "completed"
                          ? "bg-green-100"
                          : repayment.status === "partial"
                          ? "bg-blue-100"
                          : "bg-yellow-100"
                      }`}
                    >
                      {repayment.status === "overdue" ? (
                        <AlertTriangle className="w-6 h-6 text-red-600" />
                      ) : repayment.status === "completed" ? (
                        <CheckCircle className="w-6 h-6 text-green-600" />
                      ) : repayment.status === "partial" ? (
                        <TrendingUp className="w-6 h-6 text-blue-600" />
                      ) : (
                        <Clock className="w-6 h-6 text-yellow-600" />
                      )}
                    </div>
                    <div>
                      <div className="flex items-center space-x-2 mb-1">
                        <h4 className="font-medium text-gray-900">
                          {repayment.userName}
                        </h4>
                        <span
                          className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${
                            repayment.timType === "TIM_MAXI"
                              ? "bg-purple-100 text-purple-800"
                              : "bg-orange-100 text-orange-800"
                          }`}
                        >
                          {repayment.timType}
                        </span>
                        <span
                          className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(
                            repayment.status
                          )}`}
                        >
                          {getStatusLabel(repayment.status)}
                        </span>
                        {repayment.daysOverdue > 0 && (
                          <span className="inline-flex px-2 py-1 text-xs font-medium rounded-full bg-red-100 text-red-800">
                            {repayment.daysOverdue} jours de retard
                          </span>
                        )}
                      </div>
                      <div className="flex items-center space-x-4 text-sm text-gray-500">
                        <span>ID: {repayment.id}</span>
                        <span>
                          Prêt: {repayment.loanAmount.toLocaleString()} FCFA
                        </span>
                        <span>
                          Échéance:{" "}
                          {new Date(repayment.dueDate).toLocaleDateString(
                            "fr-FR"
                          )}
                        </span>
                        <span>Intérêts: {repayment.interestRate}%</span>
                      </div>
                      {repayment.status === "partial" && (
                        <div className="mt-2">
                          <div className="flex items-center space-x-2 text-xs text-gray-500">
                            <span>Progression:</span>
                            <div className="w-32 bg-gray-200 rounded-full h-2">
                              <div
                                className="bg-blue-600 h-2 rounded-full"
                                style={{
                                  width: `${
                                    (repayment.paidAmount /
                                      repayment.totalAmount) *
                                    100
                                  }%`,
                                }}
                              />
                            </div>
                            <span>
                              {Math.round(
                                (repayment.paidAmount / repayment.totalAmount) *
                                  100
                              )}
                              %
                            </span>
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                  <div className="text-right">
                    <p className="text-lg font-bold text-gray-900">
                      {repayment.remainingAmount.toLocaleString()} FCFA
                    </p>
                    <p className="text-sm text-gray-500">
                      Payé: {repayment.paidAmount.toLocaleString()} FCFA
                    </p>
                    {repayment.lastPayment && (
                      <p className="text-xs text-gray-400">
                        Dernier paiement:{" "}
                        {new Date(repayment.lastPayment).toLocaleDateString(
                          "fr-FR"
                        )}
                      </p>
                    )}
                    <button
                      onClick={() =>
                        setSelectedLoan(
                          selectedLoan === repayment.id ? null : repayment.id
                        )
                      }
                      className="mt-2 flex items-center space-x-1 bg-gradient-to-r from-amber-600 to-orange-600 text-white px-3 py-1 rounded-lg hover:from-amber-700 hover:to-orange-700 transition-all shadow-md hover:shadow-lg text-sm"
                    >
                      <Eye className="w-3 h-3" />
                      <span>Voir détails</span>
                    </button>
                  </div>
                </div>

                {/* Détails étendus */}
                {selectedLoan === repayment.id && (
                  <div className="mt-4 p-4 bg-gradient-to-br from-amber-50/50 to-orange-50/50 rounded-lg border border-amber-200">
                    <h5 className="font-medium text-gray-900 mb-3">
                      Détails du Prêt SOS
                    </h5>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div>
                        <h6 className="text-sm font-medium text-gray-700 mb-2">
                          Informations Générales
                        </h6>
                        <div className="space-y-1 text-sm">
                          <div className="flex justify-between">
                            <span className="text-gray-600">
                              Montant initial:
                            </span>
                            <span>
                              {repayment.loanAmount.toLocaleString()} FCFA
                            </span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">
                              Intérêts (6%):
                            </span>
                            <span>
                              {(
                                repayment.totalAmount - repayment.loanAmount
                              ).toLocaleString()}{" "}
                              FCFA
                            </span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">
                              Total initial:
                            </span>
                            <span>
                              {repayment.totalAmount.toLocaleString()} FCFA
                            </span>
                          </div>
                          {repayment.daysOverdue > 0 && (
                            <div className="flex justify-between text-red-600">
                              <span>Pénalité (15%):</span>
                              <span>
                                +
                                {(
                                  repayment.remainingAmount -
                                  (repayment.totalAmount - repayment.paidAmount)
                                ).toLocaleString()}{" "}
                                FCFA
                              </span>
                            </div>
                          )}
                        </div>
                      </div>
                      <div>
                        <h6 className="text-sm font-medium text-gray-700 mb-2">
                          Historique des Paiements
                        </h6>
                        <div className="space-y-2">
                          {repayment.paymentHistory.length > 0 ? (
                            repayment.paymentHistory.map((payment, index) => (
                              <div
                                key={index}
                                className="flex justify-between text-sm"
                              >
                                <span className="text-gray-600">
                                  {new Date(payment.date).toLocaleDateString(
                                    "fr-FR"
                                  )}
                                </span>
                                <span
                                  className={
                                    payment.type === "penalty"
                                      ? "text-red-600"
                                      : "text-green-600"
                                  }
                                >
                                  {payment.type === "penalty" ? "+" : ""}
                                  {payment.amount.toLocaleString()} FCFA
                                  {payment.type === "penalty"
                                    ? " (Pénalité)"
                                    : " (Paiement)"}
                                </span>
                              </div>
                            ))
                          ) : (
                            <p className="text-sm text-gray-500">
                              Aucun paiement effectué
                            </p>
                          )}
                        </div>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            ))
          )}
        </div>
      </div>
    </div>
  );
};

export default LoanRepayments;
