import React, { useState, useEffect } from "react";
import {
  CreditCard,
  TrendingUp,
  AlertTriangle,
  Clock,
  Search,
  Filter,
  Eye,
  EyeOff,
  X,
  Info,
} from "lucide-react";
import { useLocation } from "react-router-dom";
import { getLoans, getOverdueLoans } from "../../services/api";

interface LoanData {
  id: string;
  userId: string;
  userName: string;
  timType: "TIM_MAXI" | "TIM_BUSINESS" | "TIM_MINI";
  amount: number;
  interestRate: number;
  penaltyRate: number;
  remainingBalance: number;
  monthlyPayment: number;
  nextPaymentDate?: string;
  daysOverdue: number;
  totalRepayment?: number;
  paymentsCompleted: number;
  totalPayments: number;
  penaltyAmount?: number;
  status: "active" | "overdue" | "completed" | "pending" | "approved";
  createdAt: string;
}

const ActiveLoans: React.FC = () => {
  const location = useLocation();
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedStatus, setSelectedStatus] = useState<string>("all");
  const [showAmounts, setShowAmounts] = useState(true);
  const [loans, setLoans] = useState<LoanData[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [selectedLoan, setSelectedLoan] = useState<LoanData | null>(null);

  // Déterminer si on affiche les prêts en retard ou tous les prêts
  const isOverduePage = location.pathname.includes("/overdue");

  // Récupérer les prêts depuis l'API
  useEffect(() => {
    const fetchLoans = async () => {
      try {
        setLoading(true);
        setError(null);

        // Appeler l'endpoint approprié selon la page
        const response = isOverduePage
          ? await getOverdueLoans()
          : await getOverdueLoans();

        // Transformer les données de l'API au format attendu par le frontend
        const loansData = response.map((loan: any) => {
          // Déterminer le nombre total de paiements (par défaut 12 mois)
          const totalPayments = loan.total_payments || 12;

          // Calculer la mensualité si elle n'existe pas
          const monthlyPayment =
            loan.monthly_payment || loan.amount / totalPayments;

          // Calculer les paiements complétés à partir du solde restant
          const paymentsCompleted =
            loan.payments_completed !== undefined
              ? loan.payments_completed
              : Math.max(
                  0,
                  totalPayments -
                    Math.ceil(loan.remaining_balance / monthlyPayment)
                );

          return {
            id: loan.loan_id || loan.id,
            userId: loan.user?.id || loan.user_id,
            userName:
              loan.user?.full_name || loan.user_name || "Utilisateur inconnu",
            timType: loan.user?.account_type || loan.tim_type || "TIM_MAXI",
            amount: loan.amount,
            interestRate: loan.interest_rate || 0,
            penaltyRate: loan.penalty_rate || 0.05,
            remainingBalance: loan.remaining_balance,
            monthlyPayment: monthlyPayment,
            nextPaymentDate: loan.due_date || loan.next_payment_date,
            daysOverdue: loan.days_overdue || 0,
            totalRepayment: loan.remaining_balance,
            paymentsCompleted: paymentsCompleted,
            totalPayments: totalPayments,
            penaltyAmount: loan.penalty_amount || 0,
            status:
              loan.status === "overdue" ||
              (loan.days_overdue && loan.days_overdue > 0)
                ? "overdue"
                : loan.status === "approved" || loan.status === "active"
                ? "active"
                : loan.status || "pending",
            createdAt: loan.created_at,
          };
        });

        setLoans(loansData);
      } catch (err: any) {
        console.error("Erreur lors de la récupération des prêts:", err);
        setError(err.message || "Erreur lors de la récupération des prêts");
      } finally {
        setLoading(false);
      }
    };

    fetchLoans();
  }, [isOverduePage]);

  const getStatusColor = (status: string) => {
    switch (status) {
      case "active":
        return "bg-amber-100 text-amber-800";
      case "overdue":
        return "bg-red-100 text-red-800";
      case "completed":
        return "bg-orange-100 text-orange-800";
      default:
        return "bg-amber-100 text-amber-800";
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case "active":
        return "Actif";
      case "overdue":
        return "En retard";
      case "completed":
        return "Terminé";
      default:
        return status;
    }
  };

  const getTimTypeColor = (type: string) => {
    switch (type) {
      case "TIM_MAXI":
        return "bg-orange-100 text-orange-800";
      case "TIM_BUSINESS":
        return "bg-yellow-100 text-yellow-800";
      default:
        return "bg-amber-100 text-amber-800";
    }
  };

  const filteredLoans = loans.filter((loan) => {
    const matchesSearch =
      loan.userName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      loan.id.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus =
      selectedStatus === "all" || loan.status === selectedStatus;
    return matchesSearch && matchesStatus;
  });

  const totalAmount = loans.reduce((sum, loan) => sum + loan.amount, 0);
  const totalRemaining = loans.reduce(
    (sum, loan) => sum + loan.remainingBalance,
    0
  );
  const activeLoans = loans.filter((l) => l.status === "active").length;
  const overdueLoans = loans.filter((l) => l.status === "overdue").length;

  const calculatePenalty = (loan: LoanData) => {
    if (loan.daysOverdue > 0) {
      return (loan.monthlyPayment * loan.penaltyRate) / 100;
    }
    return 0;
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-amber-100 rounded-lg">
              <CreditCard className="w-6 h-6 text-amber-600" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-amber-900">
                Prêts en Cours
              </h1>
              <p className="text-amber-600">
                Supervision des prêts actifs TIM MAXI et TIM BUSINESS
              </p>
            </div>
          </div>
          <button
            onClick={() => setShowAmounts(!showAmounts)}
            className="flex items-center space-x-2 bg-amber-100 text-amber-700 px-4 py-2 rounded-lg hover:bg-amber-200 transition-colors"
          >
            {showAmounts ? (
              <EyeOff className="w-4 h-4" />
            ) : (
              <Eye className="w-4 h-4" />
            )}
            <span>{showAmounts ? "Masquer" : "Afficher"} Montants</span>
          </button>
        </div>
      </div>

      {/* Statistiques */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Prêts Actifs</p>
              <p className="text-2xl font-bold text-gray-900">{loans.length}</p>
            </div>
            <CreditCard className="w-8 h-8 text-blue-600" />
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Montant Total</p>
              <p className="text-2xl font-bold text-green-600">
                {showAmounts
                  ? `${totalAmount.toLocaleString()} FCFA`
                  : "••••••••• FCFA"}
              </p>
            </div>
            <TrendingUp className="w-8 h-8 text-green-600" />
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Solde Restant</p>
              <p className="text-2xl font-bold text-purple-600">
                {showAmounts
                  ? `${totalRemaining.toLocaleString()} FCFA`
                  : "••••••••• FCFA"}
              </p>
            </div>
            <Clock className="w-8 h-8 text-purple-600" />
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">En Retard</p>
              <p className="text-2xl font-bold text-red-600">{overdueLoans}</p>
            </div>
            <AlertTriangle className="w-8 h-8 text-red-600" />
          </div>
        </div>
      </div>

      {/* Répartition par statut */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-gray-600">Actifs</span>
            <span className="text-lg font-bold text-green-600">
              {activeLoans}
            </span>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-2">
            <div
              className="bg-green-600 h-2 rounded-full"
              style={{
                width: `${
                  loans.length > 0 ? (activeLoans / loans.length) * 100 : 0
                }%`,
              }}
            ></div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-gray-600">En Retard</span>
            <span className="text-lg font-bold text-red-600">
              {overdueLoans}
            </span>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-2">
            <div
              className="bg-red-600 h-2 rounded-full"
              style={{
                width: `${
                  loans.length > 0 ? (overdueLoans / loans.length) * 100 : 0
                }%`,
              }}
            ></div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-4">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-gray-600">
              Taux de Remboursement
            </span>
            <span className="text-lg font-bold text-blue-600">
              {loans.length > 0
                ? Math.round(
                    (loans.reduce((sum, l) => sum + l.paymentsCompleted, 0) /
                      loans.reduce((sum, l) => sum + l.totalPayments, 0)) *
                      100
                  )
                : 0}
              %
            </span>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-2">
            <div
              className="bg-blue-600 h-2 rounded-full"
              style={{
                width: `${
                  loans.length > 0
                    ? (loans.reduce((sum, l) => sum + l.paymentsCompleted, 0) /
                        loans.reduce((sum, l) => sum + l.totalPayments, 0)) *
                      100
                    : 0
                }%`,
              }}
            ></div>
          </div>
        </div>
      </div>

      {/* Filtres */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-4">
        <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-4 md:space-y-0">
          <div className="flex items-center space-x-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
              <input
                type="text"
                placeholder="Rechercher par nom ou ID..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              />
            </div>
            <div className="flex items-center space-x-2">
              <Filter className="w-4 h-4 text-gray-400" />
              <select
                value={selectedStatus}
                onChange={(e) => setSelectedStatus(e.target.value)}
                className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              >
                <option value="all">Tous les statuts</option>
                <option value="active">Actif</option>
                <option value="overdue">En retard</option>
                <option value="completed">Terminé</option>
              </select>
            </div>
          </div>
        </div>
      </div>

      {/* Liste des prêts */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">
            {isOverduePage ? "Prêts en Retard" : "Prêts Actifs"} (
            {filteredLoans.length})
          </h3>
        </div>
        <div className="divide-y divide-gray-200">
          {loading ? (
            <div className="p-12 text-center">
              <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-purple-600"></div>
              <p className="mt-4 text-purple-600">Chargement des prêts...</p>
            </div>
          ) : error ? (
            <div className="p-12 text-center">
              <p className="text-red-600 font-medium">{error}</p>
              <button
                onClick={() => window.location.reload()}
                className="mt-4 px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700"
              >
                Réessayer
              </button>
            </div>
          ) : filteredLoans.length === 0 ? (
            <div className="p-12 text-center">
              <CreditCard className="w-16 h-16 text-purple-300 mx-auto mb-4" />
              <p className="text-purple-600 font-medium">Aucun prêt trouvé</p>
              <p className="text-purple-500 text-sm mt-2">
                {searchTerm || selectedStatus !== "all"
                  ? "Essayez de modifier vos filtres de recherche"
                  : isOverduePage
                  ? "Aucun prêt en retard pour le moment"
                  : "Les prêts créés depuis l'application mobile apparaîtront ici"}
              </p>
            </div>
          ) : (
            filteredLoans.map((loan) => (
              <div
                key={loan.id}
                className="p-6 hover:bg-gray-50 transition-colors"
              >
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-4">
                    <div
                      className={`w-12 h-12 rounded-full flex items-center justify-center ${
                        loan.status === "overdue"
                          ? "bg-red-100"
                          : "bg-purple-100"
                      }`}
                    >
                      {loan.status === "overdue" ? (
                        <AlertTriangle className="w-6 h-6 text-red-600" />
                      ) : (
                        <CreditCard className="w-6 h-6 text-purple-600" />
                      )}
                    </div>
                    <div>
                      <div className="flex items-center space-x-2 mb-1">
                        <h4 className="font-medium text-gray-900">
                          {loan.userName}
                        </h4>
                        <span
                          className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getTimTypeColor(
                            loan.timType
                          )}`}
                        >
                          {loan.timType}
                        </span>
                        <span
                          className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(
                            loan.status
                          )}`}
                        >
                          {getStatusLabel(loan.status)}
                        </span>
                        {loan.daysOverdue > 0 && (
                          <span className="inline-flex px-2 py-1 text-xs font-medium rounded-full bg-red-100 text-red-800">
                            {loan.daysOverdue} jours de retard
                          </span>
                        )}
                      </div>
                      <div className="flex items-center space-x-4 text-sm text-gray-500">
                        <span>ID: {loan.id}</span>
                        <span>
                          Créé:{" "}
                          {new Date(loan.createdAt).toLocaleDateString("fr-FR")}
                        </span>
                        <span>
                          Prochaine échéance:{" "}
                          {loan.nextPaymentDate
                            ? new Date(loan.nextPaymentDate).toLocaleDateString(
                                "fr-FR"
                              )
                            : "N/A"}
                        </span>
                        <span>
                          Taux: {loan.interestRate}% + Pénalité:{" "}
                          {loan.penaltyRate}%
                        </span>
                      </div>
                      <div className="mt-2">
                        <div className="flex items-center space-x-2 text-xs text-gray-500">
                          <span>Progression:</span>
                          <div className="w-32 bg-gray-200 rounded-full h-2">
                            <div
                              className="bg-purple-600 h-2 rounded-full"
                              style={{
                                width: `${
                                  loan.totalPayments > 0
                                    ? (loan.paymentsCompleted /
                                        loan.totalPayments) *
                                      100
                                    : 0
                                }%`,
                              }}
                            ></div>
                          </div>
                          <span>
                            {loan.paymentsCompleted}/{loan.totalPayments}{" "}
                            paiements
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>
                  <div className="text-right">
                    <p className="text-lg font-bold text-gray-900">
                      {showAmounts
                        ? `${loan.amount.toLocaleString()} FCFA`
                        : "••••••••• FCFA"}
                    </p>
                    <p className="text-sm text-gray-500">
                      Restant:{" "}
                      {showAmounts
                        ? `${loan.remainingBalance.toLocaleString()} FCFA`
                        : "••••••••• FCFA"}
                    </p>
                    <p className="text-sm text-purple-600">
                      Mensualité:{" "}
                      {showAmounts
                        ? `${loan.monthlyPayment.toLocaleString()} FCFA`
                        : "••••••••• FCFA"}
                    </p>
                    {loan.daysOverdue > 0 && (
                      <p className="text-sm text-red-600">
                        Pénalité:{" "}
                        {showAmounts
                          ? `${calculatePenalty(loan).toLocaleString()} FCFA`
                          : "••••••••• FCFA"}
                      </p>
                    )}
                    <button
                      onClick={() => setSelectedLoan(loan)}
                      className="mt-3 px-4 py-2 bg-blue-600 text-white text-sm rounded-lg hover:bg-blue-700 transition-colors flex items-center space-x-1"
                    >
                      <Info className="w-4 h-4" />
                      <span>Détails</span>
                    </button>
                  </div>
                </div>
              </div>
            ))
          )}
        </div>
      </div>

      {/* Modal Détails */}
      {selectedLoan && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-xl shadow-2xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            {/* En-tête du modal */}
            <div className="sticky top-0 bg-gradient-to-r from-blue-600 to-purple-600 p-6 flex items-center justify-between">
              <div className="flex items-center space-x-3">
                <CreditCard className="w-6 h-6 text-white" />
                <h2 className="text-xl font-bold text-white">
                  Détails du Prêt
                </h2>
              </div>
              <button
                onClick={() => setSelectedLoan(null)}
                className="p-1 hover:bg-white/20 rounded-lg transition-colors"
              >
                <X className="w-6 h-6 text-white" />
              </button>
            </div>

            {/* Contenu du modal */}
            <div className="p-6 space-y-6">
              {/* Section 1: Informations Emprunteur */}
              <div className="border-b pb-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center space-x-2">
                  <div className="w-2 h-2 bg-blue-600 rounded-full"></div>
                  <span>Informations Emprunteur</span>
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <p className="text-sm text-gray-600">Nom Complet</p>
                    <p className="text-base font-medium text-gray-900">
                      {selectedLoan.userName}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">ID Emprunteur</p>
                    <p className="text-base font-medium text-gray-900">
                      {selectedLoan.userId}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Type de Compte</p>
                    <span
                      className={`inline-flex px-3 py-1 text-sm font-medium rounded-full ${getTimTypeColor(
                        selectedLoan.timType
                      )}`}
                    >
                      {selectedLoan.timType}
                    </span>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Statut</p>
                    <span
                      className={`inline-flex px-3 py-1 text-sm font-medium rounded-full ${getStatusColor(
                        selectedLoan.status
                      )}`}
                    >
                      {getStatusLabel(selectedLoan.status)}
                    </span>
                  </div>
                </div>
              </div>

              {/* Section 2: Détails du Prêt */}
              <div className="border-b pb-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center space-x-2">
                  <div className="w-2 h-2 bg-green-600 rounded-full"></div>
                  <span>Détails du Prêt</span>
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="bg-gradient-to-br from-green-50 to-emerald-50 p-4 rounded-lg">
                    <p className="text-sm text-gray-600 mb-1">
                      Montant Initial
                    </p>
                    <p className="text-2xl font-bold text-green-600">
                      {showAmounts
                        ? `${selectedLoan.amount.toLocaleString()} FCFA`
                        : "••••••••• FCFA"}
                    </p>
                  </div>
                  <div className="bg-gradient-to-br from-purple-50 to-pink-50 p-4 rounded-lg">
                    <p className="text-sm text-gray-600 mb-1">Solde Restant</p>
                    <p className="text-2xl font-bold text-purple-600">
                      {showAmounts
                        ? `${selectedLoan.remainingBalance.toLocaleString()} FCFA`
                        : "••••••••• FCFA"}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Taux d'Intérêt</p>
                    <p className="text-base font-medium text-gray-900">
                      {selectedLoan.interestRate}%
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Taux de Pénalité</p>
                    <p className="text-base font-medium text-gray-900">
                      {selectedLoan.penaltyRate}%
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Date de Création</p>
                    <p className="text-base font-medium text-gray-900">
                      {new Date(selectedLoan.createdAt).toLocaleDateString(
                        "fr-FR"
                      )}
                    </p>
                  </div>
                </div>
              </div>

              {/* Section 3: Paiements */}
              <div className="border-b pb-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center space-x-2">
                  <div className="w-2 h-2 bg-orange-600 rounded-full"></div>
                  <span>Paiements</span>
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="bg-gradient-to-br from-orange-50 to-amber-50 p-4 rounded-lg">
                    <p className="text-sm text-gray-600 mb-1">Mensualité</p>
                    <p className="text-2xl font-bold text-orange-600">
                      {showAmounts
                        ? `${selectedLoan.monthlyPayment.toLocaleString()} FCFA`
                        : "••••••••• FCFA"}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600 mb-2">Progression</p>
                    <div className="flex items-center space-x-2">
                      <div className="flex-1 bg-gray-200 rounded-full h-3">
                        <div
                          className="bg-blue-600 h-3 rounded-full transition-all"
                          style={{
                            width: `${
                              selectedLoan.totalPayments > 0
                                ? (selectedLoan.paymentsCompleted /
                                    selectedLoan.totalPayments) *
                                  100
                                : 0
                            }%`,
                          }}
                        ></div>
                      </div>
                      <span className="text-sm font-medium text-gray-900">
                        {Math.round(
                          selectedLoan.totalPayments > 0
                            ? (selectedLoan.paymentsCompleted /
                                selectedLoan.totalPayments) *
                                100
                            : 0
                        )}
                        %
                      </span>
                    </div>
                    <p className="text-xs text-gray-600 mt-2">
                      {selectedLoan.paymentsCompleted} /{" "}
                      {selectedLoan.totalPayments} paiements
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Prochaine Échéance</p>
                    <p className="text-base font-medium text-gray-900">
                      {selectedLoan.nextPaymentDate
                        ? new Date(
                            selectedLoan.nextPaymentDate
                          ).toLocaleDateString("fr-FR")
                        : "N/A"}
                    </p>
                  </div>
                  {selectedLoan.daysOverdue > 0 && (
                    <div className="bg-red-50 p-4 rounded-lg">
                      <p className="text-sm text-gray-600 mb-1">Retard</p>
                      <p className="text-lg font-bold text-red-600">
                        {selectedLoan.daysOverdue} jours
                      </p>
                    </div>
                  )}
                </div>
              </div>

              {/* Section 4: Pénalités */}
              {selectedLoan.daysOverdue > 0 && (
                <div className="bg-red-50 border border-red-200 p-4 rounded-lg">
                  <h3 className="text-lg font-semibold text-red-900 mb-4 flex items-center space-x-2">
                    <AlertTriangle className="w-5 h-5" />
                    <span>Pénalités</span>
                  </h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm text-red-700">
                        Montant de la Pénalité
                      </p>
                      <p className="text-2xl font-bold text-red-600">
                        {showAmounts
                          ? `${calculatePenalty(
                              selectedLoan
                            ).toLocaleString()} FCFA`
                          : "••••••••• FCFA"}
                      </p>
                    </div>
                    <div>
                      <p className="text-sm text-red-700">Jours de Retard</p>
                      <p className="text-2xl font-bold text-red-600">
                        {selectedLoan.daysOverdue}
                      </p>
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Pied de page du modal */}
            <div className="sticky bottom-0 bg-gray-50 border-t p-6 flex justify-end space-x-3">
              <button
                onClick={() => setSelectedLoan(null)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-100 transition-colors font-medium"
              >
                Fermer
              </button>
              <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors font-medium">
                Actions
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default ActiveLoans;
