import React, { useState, useEffect } from 'react';
import { 
  Users, 
  Wallet, 
  CreditCard, 
  TrendingUp, 
  DollarSign, 
  Activity,
  Eye,
  EyeOff,
  RefreshCw,
  Calendar,
  Clock,
  Shield,
  FileText
} from 'lucide-react';
import { Search } from 'lucide-react';
import Widget from '../../components/Dashboard/Widget';
import QuickActions from '../../components/Dashboard/QuickActions';
import AlertsWidget from '../../components/Dashboard/AlertsWidget';
import PerformanceChart from '../../components/Dashboard/PerformanceChart';
import RecentTransactions from '../../components/Dashboard/RecentTransactions';
import ReportGenerator from '../../components/Dashboard/ReportGenerator';
import { useAuth } from '../../contexts/AuthContext';
import { Transaction } from '../../types';
import { useAdminStock } from '../../hooks/useAdminStock';
import { useDashboardStats } from '../../hooks/useDashboardStats';
import { useRecentTransactions } from '../../hooks/useRecentTransactions';
import { useFinancialPerformance } from '../../hooks/useFinancialPerformance';

const Dashboard: React.FC = () => {
  const { user } = useAuth();
  const isAdmin = user?.role === 'admin' || user?.role === 'manager';
  const [balanceVisible, setBalanceVisible] = useState(true);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [currentTime, setCurrentTime] = useState(new Date());
  const [searchQuery, setSearchQuery] = useState('');
  const [showReportGenerator, setShowReportGenerator] = useState(false);
  
  // Récupérer le solde du stock admin avec auto-refresh
  const { stock, loading: stockLoading, refetch: refetchStock } = useAdminStock(isAdmin, 30000);
  
  // Récupérer les statistiques du dashboard avec auto-refresh
  const { stats: dashStats, refetch: refetchDashStats } = useDashboardStats(isAdmin, 60000);
  
  // Récupérer les transactions récentes avec auto-refresh
  const { transactions: recentTransactions, refetch: refetchTransactions } = useRecentTransactions(10, isAdmin, 30000);
  
  // Récupérer les données de performance financière avec auto-refresh
  const { chartData: performanceChartData, metrics: performanceMetrics, refetch: refetchPerformance } = useFinancialPerformance("12months", isAdmin, 60000);
  
  // Debug: Afficher le stock dans la console
  useEffect(() => {
    if (isAdmin) {
      console.log('📊 Dashboard: Stock admin =', stock);
      console.log('📊 Dashboard: Stats =', dashStats);
    }
  }, [stock, dashStats, isAdmin]);

  useEffect(() => {
    const timer = setInterval(() => setCurrentTime(new Date()), 1000);
    return () => clearInterval(timer);
  }, []);

  const handleRefresh = async () => {
    setIsRefreshing(true);
    if (isAdmin) {
      if (refetchStock) await refetchStock();
      if (refetchDashStats) await refetchDashStats();
      if (refetchTransactions) await refetchTransactions();
      if (refetchPerformance) await refetchPerformance();
    }
    await new Promise(resolve => setTimeout(resolve, 1000));
    setIsRefreshing(false);
  };

  // Utiliser les vraies transactions pour les admins, sinon liste vide
  const transactionsToShow = isAdmin ? recentTransactions : [];

  const adminStats = [
    {
      title: 'Utilisateurs Totaux',
      value: dashStats?.total_users?.toString() || '0',
      change: 0,
      changeType: 'increase' as const,
      icon: Users,
      color: 'blue' as const,
      trend: [0, 0, 0, 0, 0],
      description: `${dashStats?.total_users || 0} utilisateurs inscrits`
    },
    {
      title: 'Prêts Actifs',
      value: dashStats?.active_loans?.toString() || '0',
      change: 0,
      changeType: 'increase' as const,
      icon: CreditCard,
      color: 'purple' as const,
      trend: [0, 0, 0, 0, 0],
      description: `${dashStats?.active_loans || 0} prêts en cours`
    },
    {
      title: 'Volume Total',
      value: `${(dashStats?.total_volume || 0).toLocaleString('fr-FR')} FCFA`,
      change: 0,
      changeType: 'increase' as const,
      icon: DollarSign,
      color: 'green' as const,
      trend: [0, 0, 0, 0, 0],
      description: 'Volume total des transactions'
    },
    {
      title: 'Transactions NFC',
      value: dashStats?.nfc_transactions?.toString() || '0',
      change: 0,
      changeType: 'increase' as const,
      icon: Activity,
      color: 'indigo' as const,
      trend: [0, 0, 0, 0, 0],
      description: `${dashStats?.nfc_transactions || 0} paiements NFC`
    }
  ];

  // Simuler le type d'utilisateur (normalement vient du contexte auth)
  const userTimType = user?.timType || 'TIM_MAXI';
  
  const getUserStats = () => {
    const baseStats = [
      {
        title: `Mon Portefeuille ${userTimType}`,
        value: balanceVisible ? '0 FCFA' : '••••••••• FCFA',
        change: 0,
        changeType: 'increase' as const,
        icon: Wallet,
        color: 'blue' as const,
        trend: [0, 0, 0, 0, 0],
        description: userTimType === 'TIM_MINI' ? 'Limite: 500K FCFA' : 
                    userTimType === 'TIM_MAXI' ? 'Limite: 12M FCFA' : 
                    'Limite: Illimitée'
      },
      {
        title: 'Transactions NFC',
        value: balanceVisible ? '0 FCFA' : '••••••••• FCFA',
        change: 0,
        changeType: 'increase' as const,
        icon: Activity,
        color: 'indigo' as const,
        trend: [0, 0, 0, 0, 0],
        description: 'Aucune transaction NFC'
      }
    ];

    // Ajouter les stats spécifiques selon le type TIM
    if (userTimType === 'TIM_MAXI' || userTimType === 'TIM_BUSINESS') {
      baseStats.push({
        title: 'Mes Prêts Actifs',
        value: '0',
        change: 0,
        changeType: 'increase' as const,
        icon: CreditCard,
        color: 'blue' as const,
        trend: [0, 0, 0, 0, 0],
        description: 'Aucun prêt actif'
      });
    }

    if (userTimType === 'TIM_BUSINESS') {
      baseStats.push({
        title: 'Recharges Effectuées',
        value: balanceVisible ? '0 FCFA' : '••••••••• FCFA',
        change: 0,
        changeType: 'increase' as const,
        icon: DollarSign,
        color: 'blue' as const,
        trend: [0, 0, 0, 0, 0],
        description: 'Aucune recharge effectuée'
      });
    }

    return baseStats;
  };

  const userStats = getUserStats();

  const getQuickActions = () => {
    if (isAdmin) {
      return [
        { name: 'Gestion Utilisateurs', icon: Users, color: 'blue' as const, href: '/users', description: 'Tous les comptes TIM' },
        { name: 'Prêts en Attente', icon: CreditCard, color: 'purple' as const, href: '/loans/pending', description: 'Demandes de prêts à traiter' },
        { name: 'Stock Admin', icon: Shield, color: 'orange' as const, href: '/admin-stock', description: 'Gestion des fonds' },
        { name: 'Analytics NFC', icon: Activity, color: 'green' as const, href: '/analytics/nfc', description: 'Transactions sans contact' },
      ];
    }

    // Actions selon le type d'utilisateur
    const baseActions = [
      { name: 'Paiement NFC', icon: Activity, color: 'blue' as const, href: '/nfc-payment', description: 'Paiement sans contact' },
      { name: 'Mon Wallet', icon: Wallet, color: 'green' as const, href: '/wallet', description: 'Gérer mon portefeuille' },
    ];

    if (userTimType === 'TIM_MAXI' || userTimType === 'TIM_BUSINESS') {
      baseActions.push(
        { name: 'Mes Prêts', icon: CreditCard, color: 'blue' as const, href: '/loans', description: 'Gérer mes prêts' }
      );
    }

    if (userTimType === 'TIM_BUSINESS') {
      baseActions.push(
        { name: 'Recharges', icon: Wallet, color: 'green' as const, href: '/recharges', description: 'Recharger des comptes' },
        { name: 'Transactions', icon: FileText, color: 'blue' as const, href: '/transactions', description: 'Historique complet' }
      );
    }
    
    // Actions communes
    baseActions.push(
      { name: 'Recharge Wallet', icon: Wallet, color: 'blue' as const, href: '/wallet/reload', description: 'Recharger mon compte' },
      { name: 'Transfert', icon: Activity, color: 'green' as const, href: '/wallet/transfer', description: 'Envoyer de l\'argent' }
    );

    return baseActions;
  };

  const quickActions = getQuickActions();

  // Fonctions d'action pour les alertes
  const handleAlertAction = (alertId: number, actionType: string) => {
    console.log(`Action ${actionType} pour l'alerte ${alertId}`);
    
    switch (actionType) {
      case 'Superviser':
        window.location.href = '/loans/active';
        break;
      case 'Voir détails':
        window.location.href = '/transactions';
        break;
      case 'Valider':
        alert('Virement validé avec succès !');
        break;
      case 'OK':
        alert('Notification prise en compte');
        break;
      default:
        console.log('Action non définie');
    }
  };

  const handleDismissAlert = (alertId: number) => {
    console.log(`Alerte ${alertId} fermée`);
    // Ici on pourrait mettre à jour l'état pour supprimer l'alerte
  };

  const getAlerts = () => {
    // Aucune alerte
    return [];
  };

  const alerts = getAlerts();

  // Utiliser les vraies données de performance pour les admins, sinon données vides
  const chartDataToShow = isAdmin && performanceChartData.length > 0 ? performanceChartData : [
    { label: 'Jan', value: 0 },
    { label: 'Fév', value: 0 },
    { label: 'Mar', value: 0 },
    { label: 'Avr', value: 0 },
    { label: 'Mai', value: 0 },
    { label: 'Jun', value: 0 },
    { label: 'Jul', value: 0 },
    { label: 'Aoû', value: 0 },
    { label: 'Sep', value: 0 },
    { label: 'Oct', value: 0 },
    { label: 'Nov', value: 0 },
    { label: 'Déc', value: 0 }
  ];

  const statsToShow = isAdmin ? adminStats : userStats;

  return (
    <div className="space-y-8 animate-fadeIn">
      {/* Enhanced Header avec Logo TIM CASH */}
      <div className="bg-gradient-to-r from-amber-50/80 via-orange-50/80 to-red-50/80 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-4">
        <div className="flex items-center justify-between">
          {/* Left Section - User Info avec Logo */}
          <div className="flex items-center space-x-4">
            {/* Logo TIM CASH */}
            <div className="flex items-center space-x-2">
              <img 
                src="/assets/logoTim-2.png" 
                alt="TIM CASH Logo" 
                className="w-10 h-10 object-contain"
              />
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-amber-500 via-orange-500 to-red-500 rounded-xl flex items-center justify-center shadow-lg">
              <span className="text-white font-bold text-lg">
                {user?.full_name ? 
                  user.full_name.split(' ').map(name => name.charAt(0)).slice(0, 2).join('') : 
                  (user?.firstName?.charAt(0) || '') + (user?.lastName?.charAt(0) || '')
                }
              </span>
            </div>
            <div>
              <h1 className="text-2xl font-bold bg-gradient-to-r from-amber-700 via-orange-600 to-red-600 bg-clip-text text-transparent">
                Bonjour, {user?.full_name ? user.full_name.split(' ')[0] : user?.firstName || 'Utilisateur'}!
              </h1>
              <p className="text-amber-700 mt-1 text-sm font-medium">
                {isAdmin 
                  ? 'Votre vue d\'ensemble de la plateforme TIM CASH' 
                  : 'Votre aperçu financier TIM CASH'
                }
              </p>
            </div>
          </div>
          
          {/* Right Section - Search & Controls */}
          <div className="flex items-center space-x-4">
            {/* Enhanced Search */}
            <form onSubmit={handleRefresh} className="relative">
              <Search className="w-4 h-4 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="Rechercher..."
                className="pl-9 pr-4 py-2 w-64 bg-amber-50 border border-amber-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-amber-500 focus:border-transparent focus:bg-white transition-all duration-200 text-sm"
              />
            </form>
            
            {/* Quick Stats */}
            <div className="hidden lg:flex items-center space-x-3">
              <div className="bg-gradient-to-r from-amber-50 to-yellow-50 px-3 py-1.5 rounded-lg border border-amber-200">
                <div className="flex items-center space-x-2">
                  <Wallet className="w-3 h-3 text-amber-600" />
                  <div>
                    <p className="text-xs text-amber-600 font-medium">
                      {isAdmin ? 'Stock Admin' : 'Solde'}
                    </p>
                    <p className="text-xs font-bold text-amber-700">
                      {balanceVisible
                        ? isAdmin
                          ? stockLoading
                            ? '...'
                            : `${(stock?.balance || 0).toLocaleString('fr-FR')} FCFA`
                          : '0 FCFA'
                        : '••••••••• FCFA'}
                    </p>
                  </div>
                </div>
              </div>

              <div className="bg-gradient-to-r from-orange-50 to-red-50 px-3 py-1.5 rounded-lg border border-orange-200">
                <div className="flex items-center space-x-2">
                  <TrendingUp className={`w-3 h-3 ${
                    (stock?.variation_24h || 0) >= 0 ? 'text-green-600' : 'text-red-600'
                  }`} />
                  <div>
                    <p className="text-xs text-orange-600 font-medium">Variation 24h</p>
                    <p className={`text-xs font-bold ${
                      (stock?.variation_24h || 0) >= 0 ? 'text-green-700' : 'text-red-700'
                    }`}>
                      {balanceVisible
                        ? isAdmin
                          ? stockLoading
                            ? '...'
                            : `${(stock?.variation_24h || 0) >= 0 ? '+' : ''}${(stock?.variation_24h || 0).toLocaleString('fr-FR')} FCFA`
                          : '0 FCFA'
                        : '•••••• FCFA'}
                    </p>
                  </div>
                </div>
              </div>
            </div>
            
            {/* Action Buttons */}
            <div className="flex items-center space-x-2">
              {!isAdmin && (
                <button
                  onClick={() => setBalanceVisible(!balanceVisible)}
                  className="p-2 text-amber-600 hover:text-amber-700 hover:bg-amber-50 rounded-lg transition-all duration-200"
                  title={balanceVisible ? 'Masquer les montants' : 'Afficher les montants'}
                >
                  {balanceVisible ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                </button>
              )}
              
              <button
                onClick={handleRefresh}
                disabled={isRefreshing}
                className="p-2 text-amber-600 hover:text-orange-600 hover:bg-orange-50 rounded-lg transition-all duration-200 disabled:opacity-50"
                title="Actualiser les données"
              >
                <RefreshCw className={`w-4 h-4 ${isRefreshing ? 'animate-spin' : ''}`} />
              </button>
              
              <div className="hidden lg:flex items-center space-x-2 text-xs text-amber-600">
                <Calendar className="w-3 h-3" />
                <span>{currentTime.toLocaleDateString('fr-FR', { weekday: 'short', day: 'numeric', month: 'short' })}</span>
                <Clock className="w-3 h-3 ml-1" />
                <span>{currentTime.toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' })}</span>
              </div>
              
              <button 
                onClick={() => {
                  if (isAdmin) {
                    setShowReportGenerator(true);
                  } else {
                    // Action pour nouvelle transaction
                    console.log('Nouvelle transaction');
                  }
                }}
                className="bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg font-medium transition-all duration-200 shadow-md hover:shadow-lg transform hover:scale-105 text-sm"
              >
                {isAdmin ? 'Générer Rapport' : 'Nouvelle Transaction'}
              </button>
            </div>
          </div>
        </div>
      </div>

      {/* Enhanced Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        {statsToShow.map((stat, index) => (
          <Widget key={index} {...stat} className="scale-90" />
        ))}
      </div>

      {/* Main Content Grid */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-4">
        {/* Left Column - Charts & Transactions */}
        <div className="lg:col-span-2 space-y-4">
          <div className="scale-95">
            <RecentTransactions transactions={transactionsToShow} />
          </div>
          <div className="scale-95">
            <PerformanceChart 
              title="Performance Financière"
              data={chartDataToShow}
              color="blue"
              metrics={performanceMetrics}
            />
          </div>
        </div>
        
        {/* Right Column - Widgets */}
        <div className="space-y-4">
          {/* Quick Actions */}
          <div className="scale-95">
            <QuickActions actions={quickActions} />
          </div>
          
          {/* Alerts */}
          <div className="scale-95">
            <AlertsWidget alerts={alerts} />
          </div>
          
          {/* Financial Tip */}
          <div className="bg-gradient-to-br from-blue-500 via-purple-500 to-pink-500 rounded-xl shadow-lg p-4 text-white scale-95">
            <h3 className="text-base font-bold mb-2">
              {isAdmin ? 'Insights Plateforme' : 'Conseil Financier'}
            </h3>
            <p className="text-white/90 mb-4 text-xs leading-relaxed">
              {isAdmin 
                ? 'TIM CASH Mobile: Aucune transaction • Système prêt pour les opérations'
                : 'TIM CASH: Payez sans contact avec NFC, demandez des prêts jusqu\'à 10K FCFA, rechargez via CinetPay.'
              }
            </p>
            <button 
              onClick={() => {
                if (isAdmin) {
                  // Redirection vers analytics NFC pour les insights plateforme
                  window.location.href = '/analytics/nfc';
                } else {
                  // Simulation d'ouverture de l'app mobile
                  alert('Redirection vers l\'application mobile TIM CASH...');
                }
              }}
              className="bg-white/20 backdrop-blur-sm text-white px-3 py-1.5 rounded-lg font-medium hover:bg-white/30 transition-all duration-200 border border-white/20 text-xs"
            >
              {isAdmin ? 'Dashboard Mobile' : 'Ouvrir App Mobile'}
            </button>
          </div>
        </div>
      </div>

      {/* Report Generator Modal */}
      <ReportGenerator
        isOpen={showReportGenerator}
        onClose={() => setShowReportGenerator(false)}
        isAdmin={isAdmin}
      />
    </div>
  );
};

export default Dashboard;