import React, { useState } from 'react';
import { 
  Globe, 
  MapPin, 
  Plus, 
  Search, 
  Filter, 
  Edit3, 
  Trash2, 
  Eye,
  Building2,
  Users,
  Activity,
  DollarSign,
  Flag,
  Map,
  X
} from 'lucide-react';

interface Country {
  id: string;
  name: string;
  code: string;
  currency: string;
  currencySymbol: string;
  flag: string;
  isActive: boolean;
  cities: City[];
  totalUsers: number;
  totalVolume: number;
  createdAt: string;
}

interface City {
  id: string;
  name: string;
  countryId: string;
  isActive: boolean;
  branchCode: string;
  users: number;
  volume: number;
  createdAt: string;
}

const Countries: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedFilter, setSelectedFilter] = useState('all');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showCityModal, setShowCityModal] = useState(false);
  const [selectedCountry, setSelectedCountry] = useState<Country | null>(null);
  const [viewMode, setViewMode] = useState<'countries' | 'cities'>('countries');
  const [countries, setCountries] = useState<Country[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  // Modals pour les actions
  const [showViewModal, setShowViewModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [selectedItem, setSelectedItem] = useState<any>(null);
  const [itemType, setItemType] = useState<'country' | 'city'>('country');
  const [editFormData, setEditFormData] = useState<any>({});

  // Charger les pays depuis l'API
  React.useEffect(() => {
    const fetchCountries = async () => {
      try {
        setLoading(true);
        const response = await fetch('http://localhost:8001/api/financials/mobiles/countries');
        
        if (!response.ok) {
          throw new Error('Erreur lors du chargement des pays');
        }
        
        const data = await response.json();
        console.log('Pays reçus:', data);
        
        // Transformer les données de l'API
        const transformedCountries: Country[] = await Promise.all(
          data.map(async (country: any) => {
            // Charger les villes pour chaque pays
            const citiesResponse = await fetch(`http://localhost:8001/api/financials/cities/country/details/${country.id}`);
            let cities: City[] = [];
            
            if (citiesResponse.ok) {
              const citiesData = await citiesResponse.json();
              cities = citiesData.map((city: any) => ({
                id: city.id,
                name: city.name,
                countryId: city.country_id,
                isActive: city.is_active,
                branchCode: city.branch_code || '',
                users: city.users || 0,
                volume: city.volume || 0,
                createdAt: city.created_at
              }));
            }
            
            return {
              id: country.id,
              name: country.name,
              code: country.code,
              currency: country.currency,
              currencySymbol: country.currency_symbol,
              flag: country.flag || '🏳️',
              isActive: country.is_active,
              cities: cities,
              totalUsers: cities.reduce((sum, c) => sum + c.users, 0),
              totalVolume: cities.reduce((sum, c) => sum + c.volume, 0),
              createdAt: country.created_at
            };
          })
        );
        
        setCountries(transformedCountries);
        setError(null);
      } catch (err) {
        console.error('Erreur:', err);
        setError('Impossible de charger les pays');
      } finally {
        setLoading(false);
      }
    };
    
    fetchCountries();
  }, []);

  const filteredCountries = countries.filter(country => {
    const matchesSearch = country.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         country.code.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesFilter = selectedFilter === 'all' || 
                         (selectedFilter === 'active' && country.isActive) ||
                         (selectedFilter === 'inactive' && !country.isActive);
    return matchesSearch && matchesFilter;
  });

  const allCities = countries.flatMap(country => 
    country.cities.map(city => ({
      ...city,
      countryName: country.name,
      countryFlag: country.flag,
      currency: country.currency
    }))
  );

  const filteredCities = allCities.filter(city => {
    const matchesSearch = city.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         city.branchCode.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         city.countryName.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesFilter = selectedFilter === 'all' || 
                         (selectedFilter === 'active' && city.isActive) ||
                         (selectedFilter === 'inactive' && !city.isActive);
    return matchesSearch && matchesFilter;
  });

  const totalStats = {
    countries: countries.length,
    activeCountries: countries.filter(c => c.isActive).length,
    cities: allCities.length,
    activeCities: allCities.filter(c => c.isActive).length,
    totalUsers: countries.reduce((sum, c) => sum + c.totalUsers, 0),
    totalVolume: countries.reduce((sum, c) => sum + c.totalVolume, 0)
  };

  const handleAddCountry = () => {
    setShowAddModal(true);
  };

  const handleAddCity = (country: Country) => {
    setSelectedCountry(country);
    setShowCityModal(true);
  };

  const handleViewCountry = (country: Country) => {
    setSelectedItem(country);
    setItemType('country');
    setShowViewModal(true);
  };

  const handleEditCountry = (country: Country) => {
    setSelectedItem(country);
    setItemType('country');
    setEditFormData({
      name: country.name,
      code: country.code,
      currency: country.currency,
      isActive: country.isActive
    });
    setShowEditModal(true);
  };

  const handleDeleteCountry = (country: Country) => {
    setSelectedItem(country);
    setItemType('country');
    setShowDeleteModal(true);
  };

  const handleViewCity = (city: any) => {
    setSelectedItem(city);
    setItemType('city');
    setShowViewModal(true);
  };

  const handleEditCity = (city: any) => {
    setSelectedItem(city);
    setItemType('city');
    setEditFormData({
      name: city.name,
      branchCode: city.branchCode,
      isActive: city.isActive
    });
    setShowEditModal(true);
  };

  const handleDeleteCity = (city: any) => {
    setSelectedItem(city);
    setItemType('city');
    setShowDeleteModal(true);
  };

  const confirmDelete = () => {
    console.log(`Suppression de ${itemType}:`, selectedItem);
    // TODO: Appeler l'API pour supprimer
    setShowDeleteModal(false);
    setSelectedItem(null);
  };

  const handleSaveEdit = async () => {
    try {
      console.log('Sauvegarde des modifications:', editFormData);
      
      if (itemType === 'country') {
        // Mettre à jour le pays localement
        setCountries(prevCountries => 
          prevCountries.map(country => 
            country.id === selectedItem.id 
              ? { ...country, ...editFormData }
              : country
          )
        );
        alert(`✅ Pays "${editFormData.name}" modifié avec succès !`);
      } else {
        // Mettre à jour la ville localement
        setCountries(prevCountries => 
          prevCountries.map(country => ({
            ...country,
            cities: country.cities.map(city => 
              city.id === selectedItem.id 
                ? { ...city, ...editFormData }
                : city
            )
          }))
        );
        alert(`✅ Ville "${editFormData.name}" modifiée avec succès !`);
      }
      
      // TODO: Appeler l'API pour sauvegarder en base de données
      // await fetch(`http://localhost:8001/api/...`, { method: 'PUT', body: JSON.stringify(editFormData) });
      
      setShowEditModal(false);
      setSelectedItem(null);
      setEditFormData({});
    } catch (error) {
      console.error('Erreur lors de la sauvegarde:', error);
      alert('❌ Erreur lors de la sauvegarde des modifications');
    }
  };

  // Afficher un loader pendant le chargement
  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600 mx-auto"></div>
          <p className="mt-4 text-amber-700 font-medium">Chargement des pays...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-8">
      {/* Afficher l'erreur si présente */}
      {error && (
        <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
          <p className="font-semibold">Erreur</p>
          <p className="text-sm mt-1">{error}</p>
        </div>
      )}

      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-amber-900">🌍 Gestion des Pays et Villes</h1>
            <p className="text-amber-700 mt-2">
              Gérez les branches TIM CASH dans le monde entier
            </p>
          </div>
          <div className="flex space-x-4">
            <button
              onClick={() => setViewMode(viewMode === 'countries' ? 'cities' : 'countries')}
              className="border border-amber-300 text-amber-700 hover:bg-amber-50 px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2"
            >
              {viewMode === 'countries' ? <MapPin className="w-4 h-4" /> : <Globe className="w-4 h-4" />}
              <span>{viewMode === 'countries' ? 'Voir Villes' : 'Voir Pays'}</span>
            </button>
            <button
              onClick={handleAddCountry}
              className="bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md hover:shadow-lg transform hover:scale-105 flex items-center space-x-2"
            >
              <Plus className="w-4 h-4" />
              <span>Nouveau Pays</span>
            </button>
          </div>
        </div>
      </div>

      {/* Stats Cards avec thème marron doré */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl shadow-lg border border-blue-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-blue-700">Pays Total</p>
              <p className="text-3xl font-bold text-blue-900">{totalStats.countries}</p>
              <p className="text-sm text-green-600">{totalStats.activeCountries} actifs</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-blue-400 to-indigo-500 rounded-lg flex items-center justify-center shadow-md">
              <Globe className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-purple-50 to-pink-50 rounded-xl shadow-lg border border-purple-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-purple-700">Villes Total</p>
              <p className="text-3xl font-bold text-purple-900">{totalStats.cities}</p>
              <p className="text-sm text-green-600">{totalStats.activeCities} actives</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-purple-400 to-pink-500 rounded-lg flex items-center justify-center shadow-md">
              <MapPin className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-lg border border-green-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-green-700">Utilisateurs Total</p>
              <p className="text-3xl font-bold text-green-900">{totalStats.totalUsers.toLocaleString()}</p>
              <p className="text-sm text-emerald-600">Toutes branches</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-green-400 to-emerald-500 rounded-lg flex items-center justify-center shadow-md">
              <Users className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-lg border border-amber-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-amber-700">Volume Total</p>
              <p className="text-3xl font-bold text-amber-900">{(totalStats.totalVolume / 1000000).toFixed(0)}M</p>
              <p className="text-sm text-orange-600">FCFA</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg flex items-center justify-center shadow-md">
              <DollarSign className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Filters avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/50 to-orange-50/50 rounded-xl shadow-lg border border-amber-200 p-6">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
          <div className="flex items-center space-x-4">
            <div className="relative">
              <Search className="w-4 h-4 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
              <input
                type="text"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                placeholder={`Rechercher ${viewMode === 'countries' ? 'pays' : 'villes'}...`}
                className="pl-9 pr-4 py-2 w-64 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              />
            </div>
            <div className="flex items-center space-x-2">
              <Filter className="w-4 h-4 text-gray-400" />
              <select
                value={selectedFilter}
                onChange={(e) => setSelectedFilter(e.target.value)}
                className="border border-amber-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              >
                <option value="all">Tous</option>
                <option value="active">Actifs</option>
                <option value="inactive">Inactifs</option>
              </select>
            </div>
          </div>
        </div>
      </div>

      {/* Content */}
      {viewMode === 'countries' ? (
        /* Countries View */
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="p-6 border-b border-gray-200">
            <h3 className="text-lg font-semibold text-gray-900">Pays TIM CASH</h3>
            <p className="text-gray-600">Gérez les pays où TIM CASH est disponible</p>
          </div>
          <div className="p-6">
            <div className="space-y-4">
              {filteredCountries.map((country) => (
                <div key={country.id} className="border border-gray-200 rounded-lg p-6 hover:shadow-md transition-shadow duration-200">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4">
                      <div className="text-4xl">{country.flag}</div>
                      <div>
                        <div className="flex items-center space-x-2">
                          <h4 className="text-lg font-semibold text-gray-900">{country.name}</h4>
                          <span className="text-sm text-gray-500">({country.code})</span>
                          <span className={`inline-flex items-center px-2 py-1 text-xs font-medium rounded-full ${
                            country.isActive 
                              ? 'bg-green-100 text-green-800' 
                              : 'bg-red-100 text-red-800'
                          }`}>
                            {country.isActive ? 'Actif' : 'Inactif'}
                          </span>
                        </div>
                        <p className="text-sm text-gray-600">
                          Devise: {country.currency} • {country.cities.length} villes • {country.totalUsers.toLocaleString()} utilisateurs
                        </p>
                      </div>
                    </div>
                    <div className="flex items-center space-x-2">
                      <div className="text-right mr-4">
                        <p className="text-lg font-semibold text-gray-900">
                          {(country.totalVolume / 1000000).toFixed(1)}M {country.currencySymbol}
                        </p>
                        <p className="text-sm text-gray-600">Volume total</p>
                      </div>
                      <button
                        onClick={() => handleAddCity(country)}
                        className="p-2 text-amber-600 hover:bg-amber-50 rounded-lg transition-colors duration-200"
                        title="Ajouter ville"
                      >
                        <Plus className="w-4 h-4" />
                      </button>
                      <button 
                        onClick={() => handleViewCountry(country)}
                        className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors duration-200"
                        title="Voir détails"
                      >
                        <Eye className="w-4 h-4" />
                      </button>
                      <button 
                        onClick={() => handleEditCountry(country)}
                        className="p-2 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors duration-200"
                        title="Éditer"
                      >
                        <Edit3 className="w-4 h-4" />
                      </button>
                      <button 
                        onClick={() => handleDeleteCountry(country)}
                        className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors duration-200"
                        title="Supprimer"
                      >
                        <Trash2 className="w-4 h-4" />
                      </button>
                    </div>
                  </div>
                  
                  {/* Cities Preview */}
                  {country.cities.length > 0 && (
                    <div className="mt-4 pt-4 border-t border-gray-100">
                      <div className="flex items-center space-x-4">
                        <span className="text-sm font-medium text-gray-700">Villes:</span>
                        <div className="flex flex-wrap gap-2">
                          {country.cities.slice(0, 3).map((city) => (
                            <span key={city.id} className="inline-flex items-center px-2 py-1 text-xs bg-gray-100 text-gray-700 rounded-full">
                              <MapPin className="w-3 h-3 mr-1" />
                              {city.name} ({city.users.toLocaleString()})
                            </span>
                          ))}
                          {country.cities.length > 3 && (
                            <span className="text-xs text-gray-500">+{country.cities.length - 3} autres</span>
                          )}
                        </div>
                      </div>
                    </div>
                  )}
                </div>
              ))}
            </div>
          </div>
        </div>
      ) : (
        /* Cities View */
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="p-6 border-b border-gray-200">
            <h3 className="text-lg font-semibold text-gray-900">Villes TIM CASH</h3>
            <p className="text-gray-600">Gérez les branches TIM CASH par ville</p>
          </div>
          <div className="p-6">
            <div className="space-y-4">
              {filteredCities.map((city) => (
                <div key={city.id} className="border border-gray-200 rounded-lg p-6 hover:shadow-md transition-shadow duration-200">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4">
                      <div className="w-12 h-12 bg-blue-50 rounded-lg flex items-center justify-center">
                        <Building2 className="w-6 h-6 text-blue-600" />
                      </div>
                      <div>
                        <div className="flex items-center space-x-2">
                          <h4 className="text-lg font-semibold text-gray-900">{city.name}</h4>
                          <span className="text-2xl">{city.countryFlag}</span>
                          <span className="text-sm text-gray-500">{city.countryName}</span>
                          <span className={`inline-flex items-center px-2 py-1 text-xs font-medium rounded-full ${
                            city.isActive 
                              ? 'bg-green-100 text-green-800' 
                              : 'bg-red-100 text-red-800'
                          }`}>
                            {city.isActive ? 'Actif' : 'Inactif'}
                          </span>
                        </div>
                        <p className="text-sm text-gray-600">
                          Code: {city.branchCode} • {city.users.toLocaleString()} utilisateurs
                        </p>
                      </div>
                    </div>
                    <div className="flex items-center space-x-2">
                      <div className="text-right mr-4">
                        <p className="text-lg font-semibold text-gray-900">
                          {(city.volume / 1000000).toFixed(1)}M {city.currency}
                        </p>
                        <p className="text-sm text-gray-600">Volume</p>
                      </div>
                      <button 
                        onClick={() => handleViewCity(city)}
                        className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors duration-200"
                        title="Voir détails"
                      >
                        <Eye className="w-4 h-4" />
                      </button>
                      <button 
                        onClick={() => handleEditCity(city)}
                        className="p-2 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors duration-200"
                        title="Éditer"
                      >
                        <Edit3 className="w-4 h-4" />
                      </button>
                      <button 
                        onClick={() => handleDeleteCity(city)}
                        className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors duration-200"
                        title="Supprimer"
                      >
                        <Trash2 className="w-4 h-4" />
                      </button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}

      {/* Add Country Modal */}
      {showAddModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-md">
            <div className="p-6 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">Nouveau Pays</h3>
              <p className="text-gray-600">Ajouter un nouveau pays pour TIM CASH</p>
            </div>
            <div className="p-6 space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Nom du pays</label>
                <input
                  type="text"
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Ex: Burkina Faso"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Code pays (ISO)</label>
                <input
                  type="text"
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Ex: BF"
                  maxLength={2}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Devise</label>
                <input
                  type="text"
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Ex: FCFA"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Symbole devise</label>
                <input
                  type="text"
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Ex: FCFA"
                />
              </div>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => setShowAddModal(false)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors duration-200"
              >
                Annuler
              </button>
              <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors duration-200">
                Créer Pays
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Add City Modal */}
      {showCityModal && selectedCountry && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-md">
            <div className="p-6 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">Nouvelle Ville</h3>
              <p className="text-gray-600">Ajouter une ville à {selectedCountry.name} {selectedCountry.flag}</p>
            </div>
            <div className="p-6 space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Nom de la ville</label>
                <input
                  type="text"
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Ex: Ouagadougou"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Code branche</label>
                <input
                  type="text"
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder={`Ex: TIM-${selectedCountry.code}-OUA`}
                />
              </div>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => setShowCityModal(false)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors duration-200"
              >
                Annuler
              </button>
              <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors duration-200">
                Créer Ville
              </button>
            </div>
          </div>
        </div>
      )}

      {/* View Modal */}
      {showViewModal && selectedItem && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50/30 to-orange-50/30 flex items-center justify-between">
              <h3 className="text-lg font-semibold text-amber-900">
                {itemType === 'country' ? '🌍 Détails du Pays' : '📍 Détails de la Ville'}
              </h3>
              <button
                onClick={() => setShowViewModal(false)}
                className="p-1 hover:bg-amber-100 rounded-lg transition-colors"
                title="Fermer"
              >
                <X className="w-5 h-5 text-gray-500" />
              </button>
            </div>
            <div className="p-6 space-y-4">
              {itemType === 'country' ? (
                <>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Nom</label>
                      <p className="text-lg font-semibold text-gray-900">{selectedItem.flag} {selectedItem.name}</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Code</label>
                      <p className="text-lg font-semibold text-gray-900">{selectedItem.code}</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Devise</label>
                      <p className="text-lg font-semibold text-gray-900">{selectedItem.currency} ({selectedItem.currencySymbol})</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Statut</label>
                      <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                        selectedItem.isActive ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                      }`}>
                        {selectedItem.isActive ? 'Actif' : 'Inactif'}
                      </span>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Nombre de villes</label>
                      <p className="text-lg font-semibold text-gray-900">{selectedItem.cities?.length || 0}</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Total utilisateurs</label>
                      <p className="text-lg font-semibold text-gray-900">{selectedItem.totalUsers?.toLocaleString() || 0}</p>
                    </div>
                    <div className="col-span-2">
                      <label className="block text-sm font-medium text-gray-600">Volume total</label>
                      <p className="text-lg font-semibold text-amber-900">{((selectedItem.totalVolume || 0) / 1000000).toFixed(2)}M {selectedItem.currencySymbol}</p>
                    </div>
                  </div>
                  {selectedItem.cities && selectedItem.cities.length > 0 && (
                    <div className="mt-4 pt-4 border-t border-gray-200">
                      <label className="block text-sm font-medium text-gray-600 mb-2">Villes ({selectedItem.cities.length})</label>
                      <div className="flex flex-wrap gap-2">
                        {selectedItem.cities.map((city: City) => (
                          <span key={city.id} className="inline-flex items-center px-3 py-1 text-sm bg-amber-100 text-amber-800 rounded-full">
                            <MapPin className="w-3 h-3 mr-1" />
                            {city.name}
                          </span>
                        ))}
                      </div>
                    </div>
                  )}
                </>
              ) : (
                <>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Nom</label>
                      <p className="text-lg font-semibold text-gray-900">{selectedItem.name}</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Code branche</label>
                      <p className="text-lg font-semibold text-gray-900">{selectedItem.branchCode}</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Pays</label>
                      <p className="text-lg font-semibold text-gray-900">{selectedItem.countryFlag} {selectedItem.countryName}</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Statut</label>
                      <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                        selectedItem.isActive ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                      }`}>
                        {selectedItem.isActive ? 'Actif' : 'Inactif'}
                      </span>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Utilisateurs</label>
                      <p className="text-lg font-semibold text-gray-900">{selectedItem.users?.toLocaleString() || 0}</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-600">Volume</label>
                      <p className="text-lg font-semibold text-amber-900">{((selectedItem.volume || 0) / 1000000).toFixed(2)}M {selectedItem.currency}</p>
                    </div>
                  </div>
                </>
              )}
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end">
              <button
                onClick={() => setShowViewModal(false)}
                className="px-4 py-2 bg-gradient-to-r from-amber-600 to-orange-600 text-white rounded-lg hover:from-amber-700 hover:to-orange-700 transition-all shadow-md"
              >
                Fermer
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Edit Modal */}
      {showEditModal && selectedItem && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50/30 to-orange-50/30 flex items-center justify-between">
              <h3 className="text-lg font-semibold text-amber-900">
                {itemType === 'country' ? '✏️ Modifier le Pays' : '✏️ Modifier la Ville'}
              </h3>
              <button
                onClick={() => setShowEditModal(false)}
                className="p-1 hover:bg-amber-100 rounded-lg transition-colors"
                title="Fermer"
              >
                <X className="w-5 h-5 text-gray-500" />
              </button>
            </div>
            <div className="p-6 space-y-4">
              {itemType === 'country' ? (
                <>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Nom du pays</label>
                    <input
                      type="text"
                      value={editFormData.name || ''}
                      onChange={(e) => setEditFormData({...editFormData, name: e.target.value})}
                      className="w-full border border-amber-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Code pays</label>
                      <input
                        type="text"
                        value={editFormData.code || ''}
                        onChange={(e) => setEditFormData({...editFormData, code: e.target.value})}
                        className="w-full border border-amber-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Devise</label>
                      <input
                        type="text"
                        value={editFormData.currency || ''}
                        onChange={(e) => setEditFormData({...editFormData, currency: e.target.value})}
                        className="w-full border border-amber-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                      />
                    </div>
                  </div>
                  <div>
                    <label className="flex items-center space-x-2">
                      <input
                        type="checkbox"
                        checked={editFormData.isActive || false}
                        onChange={(e) => setEditFormData({...editFormData, isActive: e.target.checked})}
                        className="rounded border-amber-300 text-amber-600 focus:ring-amber-500"
                      />
                      <span className="text-sm font-medium text-gray-700">Pays actif</span>
                    </label>
                  </div>
                </>
              ) : (
                <>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Nom de la ville</label>
                    <input
                      type="text"
                      value={editFormData.name || ''}
                      onChange={(e) => setEditFormData({...editFormData, name: e.target.value})}
                      className="w-full border border-amber-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Code branche</label>
                    <input
                      type="text"
                      value={editFormData.branchCode || ''}
                      onChange={(e) => setEditFormData({...editFormData, branchCode: e.target.value})}
                      className="w-full border border-amber-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                    />
                  </div>
                  <div>
                    <label className="flex items-center space-x-2">
                      <input
                        type="checkbox"
                        checked={editFormData.isActive || false}
                        onChange={(e) => setEditFormData({...editFormData, isActive: e.target.checked})}
                        className="rounded border-amber-300 text-amber-600 focus:ring-amber-500"
                      />
                      <span className="text-sm font-medium text-gray-700">Ville active</span>
                    </label>
                  </div>
                </>
              )}
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => setShowEditModal(false)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Annuler
              </button>
              <button 
                onClick={handleSaveEdit}
                className="px-4 py-2 bg-gradient-to-r from-amber-600 to-orange-600 text-white rounded-lg hover:from-amber-700 hover:to-orange-700 transition-all shadow-md"
              >
                Enregistrer
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Delete Modal */}
      {showDeleteModal && selectedItem && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-md">
            <div className="p-6 border-b border-red-200 bg-red-50 flex items-center justify-between">
              <h3 className="text-lg font-semibold text-red-900">
                🗑️ Confirmer la suppression
              </h3>
              <button
                onClick={() => setShowDeleteModal(false)}
                className="p-1 hover:bg-red-100 rounded-lg transition-colors"
                title="Fermer"
              >
                <X className="w-5 h-5 text-gray-500" />
              </button>
            </div>
            <div className="p-6">
              <p className="text-gray-700">
                Êtes-vous sûr de vouloir supprimer {itemType === 'country' ? 'le pays' : 'la ville'} <strong>{selectedItem.name}</strong> ?
              </p>
              {itemType === 'country' && selectedItem.cities && selectedItem.cities.length > 0 && (
                <div className="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                  <p className="text-sm text-yellow-800">
                    ⚠️ Attention : Ce pays contient {selectedItem.cities.length} ville(s). Elles seront également supprimées.
                  </p>
                </div>
              )}
              <p className="text-sm text-gray-500 mt-4">
                Cette action est irréversible.
              </p>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => setShowDeleteModal(false)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Annuler
              </button>
              <button 
                onClick={confirmDelete}
                className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors shadow-md"
              >
                Supprimer
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Countries;
