import React, { useState } from 'react';
import { useForm } from 'react-hook-form';
import { yupResolver } from '@hookform/resolvers/yup';
import * as yup from 'yup';
import { Eye, EyeOff, Shield, Zap, CreditCard, Smartphone, ArrowRight, CheckCircle } from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';
import { useNavigate, Navigate } from 'react-router-dom';

const schema = yup.object({
  email: yup.string().email('Email invalide').required('Email requis'),
  password: yup.string().min(6, 'Le mot de passe doit contenir au moins 6 caractères').required('Mot de passe requis'),
});

type LoginFormData = {
  email: string;
  password: string;
};

const LoginNew: React.FC = () => {
  const [showPassword, setShowPassword] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const { login, isAuthenticated } = useAuth();
  const navigate = useNavigate();

  const { register, handleSubmit, formState: { errors }, setError } = useForm<LoginFormData>({
    resolver: yupResolver(schema)
  });

  if (isAuthenticated) {
    return <Navigate to="/dashboard" replace />;
  }

  const onSubmit = async (data: LoginFormData) => {
    setIsLoading(true);
    try {
      await login(data.email, data.password);
      navigate('/dashboard');
    } catch (error) {
      setError('root', {
        type: 'manual',
        message: 'Identifiants invalides. Veuillez réessayer.'
      });
    } finally {
      setIsLoading(false);
    }
  };

  const features = [
    {
      icon: Shield,
      title: 'Sécurité Bancaire',
      description: 'Vos données financières sont protégées par un chiffrement de niveau militaire'
    },
    {
      icon: Zap,
      title: 'Paiements NFC',
      description: 'Transactions sans contact instantanées avec la technologie NFC avancée'
    },
    {
      icon: CreditCard,
      title: 'Prêts SOS',
      description: 'Prêts d\'urgence automatiques de 500 à 10,000 FCFA en quelques secondes'
    },
    {
      icon: Smartphone,
      title: 'TIM Mobile',
      description: 'Portefeuilles TIM MINI, MAXI et BUSINESS adaptés à vos besoins'
    }
  ];

  return (
    <div className="min-h-screen bg-gradient-to-br from-stone-50 via-amber-50 to-orange-50">
      <div className="flex min-h-screen">
        {/* Left Side - Login Form */}
        <div className="flex-1 flex items-center justify-center px-8 py-12">
          <div className="w-full max-w-md">
            {/* Logo et Header */}
            <div className="text-center mb-8">
              <div className="flex items-center justify-center mb-6">
                <div className="relative">
                  <img 
                    src="/assets/logoTim-2.png" 
                    alt="TIM CASH Logo" 
                    className="w-20 h-20 object-contain"
                  />
                  <div className="absolute inset-0 bg-gradient-to-br from-amber-400/20 to-orange-500/20 rounded-2xl"></div>
                </div>
              </div>
              <h1 className="text-3xl font-bold bg-gradient-to-r from-amber-700 via-orange-600 to-red-600 bg-clip-text text-transparent mb-2">
                Bienvenue sur TIM CASH
              </h1>
              <p className="text-amber-700 font-medium">Connectez-vous à votre plateforme financière</p>
            </div>

            {/* Card de connexion */}
            <div className="bg-white/80 backdrop-blur-sm rounded-2xl shadow-xl border border-amber-200/50 p-8">
              <form onSubmit={handleSubmit(onSubmit)} className="space-y-6">
                {errors.root && (
                  <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-xl text-sm">
                    {errors.root.message}
                  </div>
                )}

                <div>
                  <label className="block text-sm font-semibold text-amber-800 mb-3">
                    Adresse Email
                  </label>
                  <input
                    {...register('email')}
                    type="email"
                    className="w-full px-4 py-3 border border-amber-200 rounded-xl focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-amber-50/50 transition-all duration-200 placeholder-amber-400"
                    placeholder="admin@timcash.ci"
                  />
                  {errors.email && (
                    <p className="mt-2 text-sm text-red-600">{errors.email.message}</p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-semibold text-amber-800 mb-3">
                    Mot de Passe
                  </label>
                  <div className="relative">
                    <input
                      {...register('password')}
                      type={showPassword ? 'text' : 'password'}
                      className="w-full px-4 py-3 pr-12 border border-amber-200 rounded-xl focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-amber-50/50 transition-all duration-200 placeholder-amber-400"
                      placeholder="••••••••"
                    />
                    <button
                      type="button"
                      onClick={() => setShowPassword(!showPassword)}
                      className="absolute right-4 top-1/2 transform -translate-y-1/2 text-amber-500 hover:text-amber-700 transition-colors"
                    >
                      {showPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
                    </button>
                  </div>
                  {errors.password && (
                    <p className="mt-2 text-sm text-red-600">{errors.password.message}</p>
                  )}
                </div>

                <div className="flex items-center justify-between">
                  <label className="flex items-center">
                    <input type="checkbox" className="rounded border-amber-300 text-amber-600 focus:ring-amber-500 bg-amber-50" />
                    <span className="ml-2 text-sm text-amber-700 font-medium">Se souvenir de moi</span>
                  </label>
                  <a href="#" className="text-sm text-amber-600 hover:text-amber-800 font-medium transition-colors">
                    Mot de passe oublié ?
                  </a>
                </div>

                <button
                  type="submit"
                  disabled={isLoading}
                  className="w-full bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white font-semibold py-3 px-4 rounded-xl transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed shadow-lg hover:shadow-xl transform hover:scale-[1.02] flex items-center justify-center space-x-2"
                >
                  {isLoading ? (
                    <>
                      <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                      <span>Connexion...</span>
                    </>
                  ) : (
                    <>
                      <span>Se Connecter</span>
                      <ArrowRight className="w-5 h-5" />
                    </>
                  )}
                </button>
              </form>

              {/* Informations de démonstration */}
              <div className="mt-8 p-4 bg-gradient-to-r from-amber-50 to-orange-50 rounded-xl border border-amber-200">
                <div className="flex items-center space-x-2 mb-2">
                  <CheckCircle className="w-5 h-5 text-amber-600" />
                  <span className="text-sm font-semibold text-amber-800">Compte de Démonstration</span>
                </div>
                <p className="text-sm text-amber-700">
                  <strong>Email:</strong> admin@timcash.ci<br />
                  <strong>Mot de passe:</strong> admin123
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* Right Side - Features TIM CASH */}
        <div className="hidden lg:flex flex-1 bg-gradient-to-br from-amber-900 via-orange-900 to-red-900 items-center justify-center p-12 relative overflow-hidden">
          {/* Motif de fond */}
          <div className="absolute inset-0 opacity-10">
            <div className="absolute top-20 left-20 w-32 h-32 bg-amber-400 rounded-full blur-3xl"></div>
            <div className="absolute bottom-20 right-20 w-40 h-40 bg-orange-400 rounded-full blur-3xl"></div>
            <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-64 h-64 bg-red-400 rounded-full blur-3xl"></div>
          </div>
          
          <div className="max-w-lg text-white relative z-10">
            {/* Logo et titre */}
            <div className="flex items-center space-x-4 mb-8">
              <img 
                src="/assets/logoTim-2.png" 
                alt="TIM CASH" 
                className="w-16 h-16 object-contain"
              />
              <div>
                <h2 className="text-4xl font-bold bg-gradient-to-r from-amber-200 to-orange-200 bg-clip-text text-transparent">
                  TIM CASH
                </h2>
                <p className="text-amber-200 text-lg font-medium">Super-App Financière</p>
              </div>
            </div>
            
            <h3 className="text-2xl font-bold text-amber-100 mb-6">
              Votre Solution Financière Mobile Complète
            </h3>
            <p className="text-xl text-amber-200 mb-12 leading-relaxed">
              Gérez vos portefeuilles TIM, effectuez des paiements NFC, obtenez des prêts SOS et analysez vos données financières depuis une seule plateforme puissante.
            </p>
            
            <div className="space-y-6">
              {features.map((feature, index) => {
                const IconComponent = feature.icon;
                return (
                  <div key={index} className="flex items-start space-x-4 group">
                    <div className="w-14 h-14 bg-gradient-to-br from-amber-500 to-orange-600 rounded-xl flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform duration-300">
                      <IconComponent className="w-7 h-7 text-white" />
                    </div>
                    <div className="flex-1">
                      <h4 className="font-bold text-lg mb-2 text-amber-100 group-hover:text-white transition-colors">
                        {feature.title}
                      </h4>
                      <p className="text-amber-200 leading-relaxed group-hover:text-amber-100 transition-colors">
                        {feature.description}
                      </p>
                    </div>
                  </div>
                );
              })}
            </div>

            {/* Statistiques */}
            <div className="mt-12 grid grid-cols-3 gap-6">
              <div className="text-center">
                <div className="text-2xl font-bold text-amber-200">2,847</div>
                <div className="text-sm text-amber-300">Utilisateurs</div>
              </div>
              <div className="text-center">
                <div className="text-2xl font-bold text-amber-200">847M</div>
                <div className="text-sm text-amber-300">FCFA Stock</div>
              </div>
              <div className="text-center">
                <div className="text-2xl font-bold text-amber-200">18,947</div>
                <div className="text-sm text-amber-300">Paiements NFC</div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default LoginNew;
