import React, { useState, useEffect } from 'react';
import {
  Wifi,
  WifiOff,
  AlertCircle,
  RefreshCw
} from 'lucide-react';
import { getNFCAnalytics } from '../../services/api';

interface DeviceCapability {
  id: string;
  deviceModel: string;
  brand: string;
  hasNFC: boolean;
  nfcEnabled: boolean;
  osVersion: string;
  userCount: number;
  transactionCount: number;
  successRate: number;
  lastSeen: string;
  location: string;
  country: string;
  city: string;
}

interface LocationAnalytics {
  zone: string;
  country: string;
  city: string;
  totalUsers: number;
  nfcCapableUsers: number;
  nfcEnabledUsers: number;
  transactionsToday: number;
  transactionsNFC: number;
  transactionsTraditional: number;
  averageAmount: number;
  peakHour: string;
  adoptionRate: number;
}

interface RealTimeTransaction {
  id: string;
  deviceModel: string;
  amount: number;
  timestamp: string;
  location: string;
  status: 'success' | 'failed' | 'pending';
  processingTime: number;
}

const NFCAnalytics: React.FC = () => {
  const [selectedPeriod] = useState<string>('7d');
  const [selectedView] = useState<string>('devices');
  const [deviceCapabilities, setDeviceCapabilities] = useState<DeviceCapability[]>([]);
  const [locationAnalytics, setLocationAnalytics] = useState<LocationAnalytics[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Statistiques
  const [totalUsers, setTotalUsers] = useState(0);
  const [nfcCapableDevices, setNfcCapableDevices] = useState(0);
  const [nfcEnabledDevices, setNfcEnabledDevices] = useState(0);
  const [totalTransactions, setTotalTransactions] = useState(0);
  const [nfcTransactions, setNfcTransactions] = useState(0);
  const [traditionalTransactions, setTraditionalTransactions] = useState(0);
  const [totalVolume, setTotalVolume] = useState(0);

  // Récupérer les analytics NFC depuis l'API
  useEffect(() => {
    const fetchNFCAnalytics = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await getNFCAnalytics();

        // Transformer les données des appareils
        const devicesData = response.device_capabilities.map((device: any) => ({
          id: device.id,
          deviceModel: device.device_model,
          brand: device.brand,
          hasNFC: device.has_nfc,
          nfcEnabled: device.nfc_enabled,
          osVersion: device.os_version,
          userCount: device.user_count,
          transactionCount: device.transaction_count,
          successRate: device.success_rate,
          lastSeen: device.last_seen,
          location: device.location,
          country: device.country,
          city: device.city
        }));

        // Transformer les données de localisation
        const locationsData = response.location_analytics.map((loc: any) => ({
          zone: loc.zone,
          country: loc.country,
          city: loc.city,
          totalUsers: loc.total_users,
          nfcCapableUsers: loc.nfc_capable_users,
          nfcEnabledUsers: loc.nfc_enabled_users,
          transactionsToday: loc.transactions_today,
          transactionsNFC: loc.transactions_nfc,
          transactionsTraditional: loc.transactions_traditional,
          averageAmount: loc.average_amount,
          peakHour: loc.peak_hour,
          adoptionRate: loc.adoption_rate
        }));

        setDeviceCapabilities(devicesData);
        setLocationAnalytics(locationsData);

        // Mettre à jour les statistiques
        setTotalUsers(response.stats.total_users);
        setNfcCapableDevices(response.stats.nfc_capable_devices);
        setNfcEnabledDevices(response.stats.nfc_enabled_devices);
        setTotalTransactions(response.stats.total_transactions);
        setNfcTransactions(response.stats.nfc_transactions);
        setTraditionalTransactions(response.stats.traditional_transactions);
        setTotalVolume(response.stats.total_volume);
      } catch (err: any) {
        console.error('Erreur lors de la récupération des analytics NFC:', err);
        setError(err.message || 'Erreur lors de la récupération des analytics NFC');
      } finally {
        setLoading(false);
      }
    };

    fetchNFCAnalytics();
  }, []);

  return (
    <div className="p-6 space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div className="p-2 bg-amber-100 rounded-lg">
              <Wifi className="w-6 h-6 text-amber-600" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-amber-900">Analytics NFC</h1>
              <p className="text-amber-700">Analyse des paiements sans contact</p>
            </div>
          </div>
        </div>
      </div>

      {/* Statistiques principales */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-700">Appareils NFC</p>
              <p className="text-2xl font-bold text-amber-800">{nfcCapableDevices}</p>
              <p className="text-xs text-amber-600 mt-1">Sur {totalUsers} utilisateurs</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg flex items-center justify-center shadow-md">
              <Wifi className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-yellow-50 to-amber-50 rounded-xl shadow-sm border border-yellow-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-yellow-700">NFC Activé</p>
              <p className="text-2xl font-bold text-yellow-800">{nfcEnabledDevices}</p>
              <p className="text-xs text-yellow-600 mt-1">Prêts à payer</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-yellow-400 to-amber-500 rounded-lg flex items-center justify-center shadow-md">
              <Wifi className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-orange-50 to-red-50 rounded-xl shadow-sm border border-orange-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-orange-700">Transactions NFC</p>
              <p className="text-2xl font-bold text-orange-800">{nfcTransactions}</p>
              <p className="text-xs text-orange-600 mt-1">Sur {totalTransactions} total</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-orange-400 to-red-500 rounded-lg flex items-center justify-center shadow-md">
              <Wifi className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-stone-50 to-amber-50 rounded-xl shadow-sm border border-stone-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-stone-700">Sans NFC</p>
              <p className="text-2xl font-bold text-stone-800">{traditionalTransactions}</p>
              <p className="text-xs text-stone-600 mt-1">Méthodes traditionnelles</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-stone-400 to-amber-500 rounded-lg flex items-center justify-center shadow-md">
              <WifiOff className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Contenu principal */}
      {loading ? (
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-12 text-center">
          <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600 mb-4"></div>
          <p className="text-amber-600">Chargement des analytics NFC...</p>
        </div>
      ) : error ? (
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-12 text-center">
          <p className="text-red-600 font-medium mb-4">{error}</p>
          <button
            onClick={() => window.location.reload()}
            className="px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700 transition-colors"
          >
            Réessayer
          </button>
        </div>
      ) : deviceCapabilities.length === 0 && locationAnalytics.length === 0 ? (
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-sm border border-amber-200 p-12 text-center">
          <div className="w-16 h-16 bg-gradient-to-br from-amber-200 to-orange-300 rounded-full flex items-center justify-center mx-auto mb-4">
            <AlertCircle className="w-8 h-8 text-amber-700" />
          </div>
          <h3 className="text-xl font-semibold text-amber-900 mb-2">Aucune donnée NFC</h3>
          <p className="text-amber-700">
            Les analytics NFC apparaîtront ici lorsque les utilisateurs effectueront des paiements sans contact.
          </p>
        </div>
      ) : (
        <div className="space-y-6">
          {/* Tableau des appareils */}
          {deviceCapabilities.length > 0 && (
            <div className="bg-white rounded-xl shadow-sm border border-amber-200">
              <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50 to-orange-50">
                <h3 className="text-lg font-semibold text-amber-900">Capacités des Appareils</h3>
                <p className="text-sm text-amber-700 mt-1">{deviceCapabilities.length} appareil(s)</p>
              </div>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50 border-b border-gray-200">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Appareil</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">NFC</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Utilisateurs</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Transactions</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Taux de succès</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Localisation</th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {deviceCapabilities.map((device) => (
                      <tr key={device.id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div>
                            <p className="text-sm font-medium text-gray-900">{device.deviceModel}</p>
                            <p className="text-xs text-gray-500">{device.brand} - {device.osVersion}</p>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          {device.hasNFC ? (
                            <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                              device.nfcEnabled ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'
                            }`}>
                              <Wifi className="w-3 h-3 mr-1" />
                              {device.nfcEnabled ? 'Activé' : 'Désactivé'}
                            </span>
                          ) : (
                            <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                              <WifiOff className="w-3 h-3 mr-1" />
                              Non supporté
                            </span>
                          )}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          {device.userCount.toLocaleString()}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          {device.transactionCount.toLocaleString()}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={`inline-flex px-2 py-1 rounded-full text-xs font-medium ${
                            device.successRate >= 95 ? 'bg-green-100 text-green-800' :
                            device.successRate >= 85 ? 'bg-yellow-100 text-yellow-800' :
                            'bg-red-100 text-red-800'
                          }`}>
                            {device.successRate.toFixed(1)}%
                          </span>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div>
                            <p className="text-sm text-gray-900">{device.city}</p>
                            <p className="text-xs text-gray-500">{device.country}</p>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
          )}

          {/* Analytics par zone */}
          {locationAnalytics.length > 0 && (
            <div className="bg-white rounded-xl shadow-sm border border-amber-200">
              <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50 to-orange-50">
                <h3 className="text-lg font-semibold text-amber-900">Analytics par Zone Géographique</h3>
                <p className="text-sm text-amber-700 mt-1">{locationAnalytics.length} zone(s)</p>
              </div>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50 border-b border-gray-200">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Zone</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Utilisateurs</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">NFC Capable</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">NFC Activé</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Transactions</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Taux d'adoption</th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {locationAnalytics.map((location, index) => (
                      <tr key={index} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div>
                            <p className="text-sm font-medium text-gray-900">{location.zone}</p>
                            <p className="text-xs text-gray-500">{location.city}, {location.country}</p>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          {location.totalUsers.toLocaleString()}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          {location.nfcCapableUsers.toLocaleString()}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          {location.nfcEnabledUsers.toLocaleString()}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div>
                            <p className="text-sm text-gray-900">Total: {location.transactionsToday.toLocaleString()}</p>
                            <p className="text-xs text-gray-500">
                              NFC: {location.transactionsNFC.toLocaleString()} |
                              Trad: {location.transactionsTraditional.toLocaleString()}
                            </p>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={`inline-flex px-2 py-1 rounded-full text-xs font-medium ${
                            location.adoptionRate >= 50 ? 'bg-green-100 text-green-800' :
                            location.adoptionRate >= 25 ? 'bg-yellow-100 text-yellow-800' :
                            'bg-red-100 text-red-800'
                          }`}>
                            {location.adoptionRate.toFixed(1)}%
                          </span>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );
};

export default NFCAnalytics;
