import React, { useState } from 'react';
import { 
  BarChart3, 
  PieChart, 
  TrendingUp, 
  TrendingDown,
  Users,
  DollarSign,
  CreditCard,
  Activity,
  Calendar,
  Filter,
  Download,
  RefreshCw,
  Eye,
  ArrowUpRight,
  ArrowDownLeft
} from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';

const Analytics: React.FC = () => {
  const { user } = useAuth();
  const [selectedPeriod, setSelectedPeriod] = useState('30');
  const [selectedMetric, setSelectedMetric] = useState('revenue');
  const [viewType, setViewType] = useState('chart');

  const isAdmin = user?.role === 'admin' || user?.role === 'manager';

  // Mock analytics data - Toutes les données à zéro
  const analyticsData = {
    revenue: {
      current: 0,
      previous: 0,
      growth: 0,
      trend: 'up'
    },
    users: {
      current: 0,
      previous: 0,
      growth: 0,
      trend: 'up'
    },
    transactions: {
      current: 0,
      previous: 0,
      growth: 0,
      trend: 'up'
    },
    loans: {
      current: 0,
      previous: 0,
      growth: 0,
      trend: 'up'
    }
  };

  const chartData = [
    { month: 'Jan', revenue: 0, users: 0, transactions: 0, loans: 0 },
    { month: 'Fév', revenue: 0, users: 0, transactions: 0, loans: 0 },
    { month: 'Mar', revenue: 0, users: 0, transactions: 0, loans: 0 },
    { month: 'Avr', revenue: 0, users: 0, transactions: 0, loans: 0 },
    { month: 'Mai', revenue: 0, users: 0, transactions: 0, loans: 0 },
    { month: 'Jun', revenue: 0, users: 0, transactions: 0, loans: 0 }
  ];

  const topMetrics = [
    {
      title: 'Taux de Conversion',
      value: '0%',
      change: '0%',
      trend: 'up',
      description: 'Visiteurs → Utilisateurs inscrits'
    },
    {
      title: 'Valeur Moyenne Transaction',
      value: '0 FCFA',
      change: '0 FCFA',
      trend: 'up',
      description: 'Montant moyen par transaction'
    },
    {
      title: 'Taux de Rétention',
      value: '0%',
      change: '0%',
      trend: 'up',
      description: 'Utilisateurs actifs après 30 jours'
    },
    {
      title: 'Temps de Session',
      value: '0m 0s',
      change: '0m 0s',
      trend: 'up',
      description: 'Durée moyenne de session'
    }
  ];

  const userSegments = [
    { name: 'Nouveaux Utilisateurs', value: 0, color: 'bg-blue-500' },
    { name: 'Utilisateurs Actifs', value: 0, color: 'bg-green-500' },
    { name: 'Utilisateurs Premium', value: 0, color: 'bg-purple-500' },
    { name: 'Utilisateurs Inactifs', value: 0, color: 'bg-gray-400' }
  ];

  const transactionTypes = [
    { name: 'Transferts', value: 0, amount: 0, color: 'bg-blue-500' },
    { name: 'Prêts', value: 0, amount: 0, color: 'bg-green-500' },
    { name: 'Remboursements', value: 0, amount: 0, color: 'bg-yellow-500' },
    { name: 'Recharges', value: 0, amount: 0, color: 'bg-purple-500' }
  ];

  const getMetricIcon = (metric: string) => {
    switch (metric) {
      case 'revenue':
        return DollarSign;
      case 'users':
        return Users;
      case 'transactions':
        return Activity;
      case 'loans':
        return CreditCard;
      default:
        return BarChart3;
    }
  };

  const formatValue = (value: number, metric: string) => {
    switch (metric) {
      case 'revenue':
        return `$${(value / 1000).toFixed(0)}K`;
      case 'users':
        return value.toLocaleString();
      case 'transactions':
        return `${(value / 1000).toFixed(1)}K`;
      case 'loans':
        return value.toLocaleString();
      default:
        return value.toString();
    }
  };

  return (
    <div className="space-y-8">
      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-amber-900">📊 Analytics Dashboard</h1>
            <p className="text-amber-700 mt-2">
              Analyses approfondies et métriques de performance de la plateforme
            </p>
          </div>
          <div className="flex space-x-4">
            <select
              value={selectedPeriod}
              onChange={(e) => setSelectedPeriod(e.target.value)}
              className="px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
            >
              <option value="7">7 derniers jours</option>
              <option value="30">30 derniers jours</option>
              <option value="90">90 derniers jours</option>
              <option value="365">Cette année</option>
            </select>
            <button className="border border-amber-300 text-amber-700 hover:bg-amber-50 px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2">
              <Download className="w-4 h-4" />
              <span>Exporter</span>
            </button>
            <button className="bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md hover:shadow-lg transform hover:scale-105 flex items-center space-x-2">
              <RefreshCw className="w-4 h-4" />
              <span>Actualiser</span>
            </button>
          </div>
        </div>
      </div>

      {/* Key Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {Object.entries(analyticsData).map(([key, data]) => {
          const IconComponent = getMetricIcon(key);
          return (
            <div key={key} className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
              <div className="flex items-center justify-between mb-4">
                <div className="w-12 h-12 bg-gradient-to-br from-amber-100 to-orange-100 rounded-lg flex items-center justify-center">
                  <IconComponent className="w-6 h-6 text-amber-600" />
                </div>
                <div className={`flex items-center space-x-1 text-sm font-medium ${
                  data.trend === 'up' ? 'text-green-600' : 'text-red-600'
                }`}>
                  {data.trend === 'up' ? (
                    <TrendingUp className="w-4 h-4" />
                  ) : (
                    <TrendingDown className="w-4 h-4" />
                  )}
                  <span>{data.growth}%</span>
                </div>
              </div>
              <div>
                <p className="text-sm font-medium text-amber-700 capitalize mb-1">
                  {key === 'revenue' ? 'Revenus' :
                   key === 'users' ? 'Utilisateurs' :
                   key === 'transactions' ? 'Transactions' :
                   key === 'loans' ? 'Prêts' : key}
                </p>
                <p className="text-3xl font-bold text-amber-900">
                  {formatValue(data.current, key)}
                </p>
                <p className="text-sm text-gray-500 mt-1">
                  vs {formatValue(data.previous, key)} période précédente
                </p>
              </div>
            </div>
          );
        })}
      </div>

      {/* Chart Section */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200">
        <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50 to-orange-50">
          <div className="flex items-center justify-between">
            <h3 className="text-lg font-semibold text-amber-900">Tendances Temporelles</h3>
            <div className="flex items-center space-x-4">
              <select
                value={selectedMetric}
                onChange={(e) => setSelectedMetric(e.target.value)}
                className="px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 text-sm"
              >
                <option value="revenue">Revenus</option>
                <option value="users">Utilisateurs</option>
                <option value="transactions">Transactions</option>
                <option value="loans">Prêts</option>
              </select>
              <div className="flex border border-gray-300 rounded-lg">
                <button
                  onClick={() => setViewType('chart')}
                  className={`px-3 py-2 text-sm font-medium rounded-l-lg transition-colors duration-200 ${
                    viewType === 'chart'
                      ? 'bg-gradient-to-r from-amber-600 to-orange-600 text-white'
                      : 'text-gray-600 hover:text-amber-700'
                  }`}
                >
                  <BarChart3 className="w-4 h-4" />
                </button>
                <button
                  onClick={() => setViewType('table')}
                  className={`px-3 py-2 text-sm font-medium rounded-r-lg transition-colors duration-200 ${
                    viewType === 'table'
                      ? 'bg-gradient-to-r from-amber-600 to-orange-600 text-white'
                      : 'text-gray-600 hover:text-amber-700'
                  }`}
                >
                  <Eye className="w-4 h-4" />
                </button>
              </div>
            </div>
          </div>
        </div>
        
        <div className="p-6">
          {viewType === 'chart' ? (
            <div className="h-80 flex items-end justify-between space-x-2">
              {chartData.map((data, index) => {
                const value = data[selectedMetric as keyof typeof data] as number;
                const maxValue = Math.max(...chartData.map(d => d[selectedMetric as keyof typeof d] as number));
                const height = (value / maxValue) * 100;
                
                return (
                  <div key={index} className="flex-1 flex flex-col items-center">
                    <div
                      className="w-full bg-gradient-to-t from-amber-500 to-orange-500 rounded-t-lg transition-all duration-300 hover:from-amber-600 hover:to-orange-600"
                      style={{ height: `${height}%` }}
                    ></div>
                    <div className="mt-2 text-center">
                      <p className="text-sm font-medium text-gray-900">
                        {formatValue(value, selectedMetric)}
                      </p>
                      <p className="text-xs text-gray-500">{data.month}</p>
                    </div>
                  </div>
                );
              })}
            </div>
          ) : (
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b border-gray-200">
                    <th className="text-left py-3 px-4 font-medium text-gray-900">Mois</th>
                    <th className="text-left py-3 px-4 font-medium text-gray-900">Revenus</th>
                    <th className="text-left py-3 px-4 font-medium text-gray-900">Utilisateurs</th>
                    <th className="text-left py-3 px-4 font-medium text-gray-900">Transactions</th>
                    <th className="text-left py-3 px-4 font-medium text-gray-900">Prêts</th>
                  </tr>
                </thead>
                <tbody>
                  {chartData.map((data, index) => (
                    <tr key={index} className="border-b border-gray-100 hover:bg-gray-50">
                      <td className="py-3 px-4 font-medium text-gray-900">{data.month}</td>
                      <td className="py-3 px-4 text-gray-600">${data.revenue.toLocaleString()}</td>
                      <td className="py-3 px-4 text-gray-600">{data.users.toLocaleString()}</td>
                      <td className="py-3 px-4 text-gray-600">{data.transactions.toLocaleString()}</td>
                      <td className="py-3 px-4 text-gray-600">{data.loans.toLocaleString()}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* User Segments */}
        <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
          <h3 className="text-lg font-semibold text-amber-900 mb-6">Segmentation Utilisateurs</h3>
          <div className="space-y-4">
            {userSegments.map((segment, index) => (
              <div key={index} className="flex items-center justify-between">
                <div className="flex items-center space-x-3">
                  <div className={`w-4 h-4 rounded-full ${segment.color}`}></div>
                  <span className="font-medium text-gray-900">{segment.name}</span>
                </div>
                <div className="flex items-center space-x-3">
                  <div className="w-32 bg-gray-200 rounded-full h-2">
                    <div
                      className={`h-2 rounded-full ${segment.color}`}
                      style={{ width: `${segment.value}%` }}
                    ></div>
                  </div>
                  <span className="text-sm font-medium text-gray-600 w-8">{segment.value}%</span>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Transaction Distribution */}
        <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
          <h3 className="text-lg font-semibold text-amber-900 mb-6">Répartition des Transactions</h3>
          <div className="space-y-4">
            {transactionTypes.map((type, index) => (
              <div key={index} className="flex items-center justify-between">
                <div className="flex items-center space-x-3">
                  <div className={`w-4 h-4 rounded-full ${type.color}`}></div>
                  <span className="font-medium text-gray-900">{type.name}</span>
                </div>
                <div className="text-right">
                  <p className="font-medium text-gray-900">${type.amount.toLocaleString()}</p>
                  <p className="text-sm text-gray-500">{type.value}%</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Top Metrics */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200 p-6">
        <h3 className="text-lg font-semibold text-amber-900 mb-6">Métriques Clés</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {topMetrics.map((metric, index) => (
            <div key={index} className="text-center">
              <div className="flex items-center justify-center mb-2">
                <span className="text-2xl font-bold text-gray-900">{metric.value}</span>
                <div className={`ml-2 flex items-center text-sm font-medium ${
                  metric.trend === 'up' ? 'text-green-600' : 'text-red-600'
                }`}>
                  {metric.trend === 'up' ? (
                    <ArrowUpRight className="w-4 h-4" />
                  ) : (
                    <ArrowDownLeft className="w-4 h-4" />
                  )}
                  <span>{metric.change}</span>
                </div>
              </div>
              <h4 className="font-medium text-gray-900 mb-1">{metric.title}</h4>
              <p className="text-sm text-gray-600">{metric.description}</p>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
};

export default Analytics;