import React, { useState } from 'react';
import { 
  Image, 
  Plus, 
  Search, 
  Filter, 
  Edit3, 
  Trash2, 
  Eye,
  Upload,
  Play,
  Pause,
  BarChart3,
  MousePointer,
  DollarSign
} from 'lucide-react';

interface Advertisement {
  id: string;
  title: string;
  description: string;
  imageUrl: string;
  type: 'banner' | 'popup' | 'video' | 'native';
  status: 'active' | 'paused' | 'scheduled' | 'expired';
  targetCountries: string[];
  targetCities: string[];
  startDate: string;
  endDate: string;
  budget: number;
  spent: number;
  impressions: number;
  clicks: number;
  ctr: number;
  createdAt: string;
  createdBy: string;
}

const Advertising: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedFilter, setSelectedFilter] = useState('all');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showDetailsModal, setShowDetailsModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [selectedAd, setSelectedAd] = useState<Advertisement | null>(null);
  // Liste vide des publicités TIM CASH
  const [advertisements, setAdvertisements] = useState<Advertisement[]>([]);

  // Fonctions pour les actions des boutons
  const handleViewDetails = (ad: Advertisement) => {
    setSelectedAd(ad);
    setShowDetailsModal(true);
  };

  const handleEditAd = (ad: Advertisement) => {
    setSelectedAd(ad);
    setShowEditModal(true);
  };

  const handleToggleStatus = (ad: Advertisement) => {
    const newStatus = ad.status === 'active' ? 'paused' : 'active';
    setAdvertisements(prev => 
      prev.map(a => 
        a.id === ad.id ? { ...a, status: newStatus } : a
      )
    );
    console.log(`Publicité ${ad.title} ${newStatus === 'active' ? 'activée' : 'mise en pause'}`);
  };

  const handleDeleteAd = (ad: Advertisement) => {
    if (window.confirm(`Êtes-vous sûr de vouloir supprimer la publicité "${ad.title}" ?`)) {
      setAdvertisements(prev => prev.filter(a => a.id !== ad.id));
      console.log(`Publicité ${ad.title} supprimée`);
    }
  };

  const filteredAds = advertisements.filter(ad => {
    const matchesSearch = ad.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         ad.description.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesFilter = selectedFilter === 'all' || ad.status === selectedFilter;
    return matchesSearch && matchesFilter;
  });

  const totalStats = {
    totalAds: advertisements.length,
    activeAds: advertisements.filter(ad => ad.status === 'active').length,
    totalBudget: advertisements.reduce((sum, ad) => sum + ad.budget, 0),
    totalSpent: advertisements.reduce((sum, ad) => sum + ad.spent, 0),
    totalImpressions: advertisements.reduce((sum, ad) => sum + ad.impressions, 0),
    totalClicks: advertisements.reduce((sum, ad) => sum + ad.clicks, 0),
    avgCTR: advertisements.reduce((sum, ad) => sum + ad.ctr, 0) / advertisements.length
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'paused': return 'bg-yellow-100 text-yellow-800';
      case 'scheduled': return 'bg-blue-100 text-blue-800';
      case 'expired': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case 'banner': return <Image className="w-4 h-4" />;
      case 'popup': return <MousePointer className="w-4 h-4" />;
      case 'video': return <Play className="w-4 h-4" />;
      case 'native': return <BarChart3 className="w-4 h-4" />;
      default: return <Image className="w-4 h-4" />;
    }
  };

  return (
    <div className="space-y-8">
      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-amber-900">📢 Gestion des Publicités</h1>
            <p className="text-amber-700 mt-2">
              Créez et gérez les publicités pour l'application mobile TIM CASH
            </p>
          </div>
          <button
            onClick={() => setShowAddModal(true)}
            className="bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg font-medium transition-all shadow-md hover:shadow-lg transform hover:scale-105 flex items-center space-x-2"
          >
            <Plus className="w-4 h-4" />
            <span>Nouvelle Publicité</span>
          </button>
        </div>
      </div>

      {/* Stats Cards avec thème marron doré */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-lg border border-amber-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-amber-700">Publicités Total</p>
              <p className="text-3xl font-bold text-amber-900">{totalStats.totalAds}</p>
              <p className="text-sm text-green-600">{totalStats.activeAds} actives</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg flex items-center justify-center shadow-md">
              <Image className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-lg border border-green-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-green-700">Budget Total</p>
              <p className="text-3xl font-bold text-green-900">{(totalStats.totalBudget / 1000).toFixed(0)}K</p>
              <p className="text-sm text-emerald-600">FCFA</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-green-400 to-emerald-500 rounded-lg flex items-center justify-center shadow-md">
              <DollarSign className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-purple-50 to-pink-50 rounded-xl shadow-lg border border-purple-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-purple-700">Impressions</p>
              <p className="text-3xl font-bold text-purple-900">{(totalStats.totalImpressions / 1000).toFixed(0)}K</p>
              <p className="text-sm text-pink-600">Vues totales</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-purple-400 to-pink-500 rounded-lg flex items-center justify-center shadow-md">
              <Eye className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-orange-50 to-red-50 rounded-xl shadow-lg border border-orange-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-orange-700">CTR Moyen</p>
              <p className="text-3xl font-bold text-orange-900">{totalStats.avgCTR.toFixed(1)}%</p>
              <p className="text-sm text-red-600">Taux de clic</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-orange-400 to-red-500 rounded-lg flex items-center justify-center shadow-md">
              <MousePointer className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Filters avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/50 to-orange-50/50 rounded-xl shadow-lg border border-amber-200 p-6">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
          <div className="flex items-center space-x-4">
            <div className="relative">
              <Search className="w-4 h-4 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
              <input
                type="text"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                placeholder="Rechercher publicités..."
                className="pl-9 pr-4 py-2 w-64 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              />
            </div>
            <div className="flex items-center space-x-2">
              <Filter className="w-4 h-4 text-gray-400" />
              <select
                value={selectedFilter}
                onChange={(e) => setSelectedFilter(e.target.value)}
                className="border border-amber-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              >
                <option value="all">Tous</option>
                <option value="active">Actives</option>
                <option value="paused">En pause</option>
                <option value="scheduled">Programmées</option>
                <option value="expired">Expirées</option>
              </select>
            </div>
          </div>
        </div>
      </div>

      {/* Advertisements List avec thème marron doré */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200">
        <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50/30 to-orange-50/30">
          <h3 className="text-lg font-semibold text-amber-900">Publicités TIM CASH</h3>
          <p className="text-amber-700">Gérez vos campagnes publicitaires</p>
        </div>
        <div className="p-6">
          {filteredAds.length === 0 ? (
            <div className="p-12 text-center">
              <div className="inline-flex items-center justify-center w-16 h-16 bg-amber-100 rounded-full mb-4">
                <Image className="w-8 h-8 text-amber-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Aucune publicité</h3>
              <p className="text-gray-600 mb-4">
                Commencez par créer votre première campagne publicitaire
              </p>
              <button
                onClick={() => setShowAddModal(true)}
                className="inline-flex items-center space-x-2 bg-gradient-to-r from-amber-600 to-orange-600 text-white px-4 py-2 rounded-lg hover:from-amber-700 hover:to-orange-700 transition-all shadow-md"
              >
                <Plus className="w-4 h-4" />
                <span>Créer une publicité</span>
              </button>
            </div>
          ) : (
            <div className="space-y-6">
              {filteredAds.map((ad) => (
              <div key={ad.id} className="border border-gray-200 rounded-lg p-6 hover:shadow-md transition-shadow duration-200">
                <div className="flex items-start space-x-4">
                  {/* Ad Preview */}
                  <div className="w-24 h-16 bg-gray-100 rounded-lg flex items-center justify-center overflow-hidden">
                    <img 
                      src={ad.imageUrl} 
                      alt={ad.title}
                      className="w-full h-full object-cover"
                      onError={(e) => {
                        const target = e.currentTarget as HTMLImageElement;
                        target.style.display = 'none';
                        const nextElement = target.nextElementSibling as HTMLElement;
                        if (nextElement) nextElement.style.display = 'flex';
                      }}
                    />
                    <div className="hidden w-full h-full items-center justify-center">
                      <Image className="w-8 h-8 text-gray-400" />
                    </div>
                  </div>

                  {/* Ad Info */}
                  <div className="flex-1">
                    <div className="flex items-start justify-between">
                      <div>
                        <div className="flex items-center space-x-2 mb-2">
                          <h4 className="text-lg font-semibold text-gray-900">{ad.title}</h4>
                          <div className="flex items-center space-x-1">
                            {getTypeIcon(ad.type)}
                            <span className="text-sm text-gray-500 capitalize">{ad.type}</span>
                          </div>
                          <span className={`inline-flex items-center px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(ad.status)}`}>
                            {ad.status === 'active' ? 'Active' : 
                             ad.status === 'paused' ? 'En pause' :
                             ad.status === 'scheduled' ? 'Programmée' : 'Expirée'}
                          </span>
                        </div>
                        <p className="text-gray-600 mb-2">{ad.description}</p>
                        <div className="flex items-center space-x-4 text-sm text-gray-500">
                          <span>Cibles: {ad.targetCountries.join(', ')}</span>
                          <span>•</span>
                          <span>Budget: {ad.budget.toLocaleString()} FCFA</span>
                          <span>•</span>
                          <span>CTR: {ad.ctr}%</span>
                        </div>
                      </div>
                      <div className="flex items-center space-x-2">
                        <button 
                          onClick={() => handleViewDetails(ad)}
                          className="p-2 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors duration-200"
                          title="Voir les détails"
                        >
                          <Eye className="w-4 h-4" />
                        </button>
                        <button 
                          onClick={() => handleEditAd(ad)}
                          className="p-2 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors duration-200"
                          title="Modifier"
                        >
                          <Edit3 className="w-4 h-4" />
                        </button>
                        <button 
                          onClick={() => handleToggleStatus(ad)}
                          className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors duration-200"
                          title={ad.status === 'active' ? 'Mettre en pause' : 'Activer'}
                        >
                          {ad.status === 'active' ? <Pause className="w-4 h-4" /> : <Play className="w-4 h-4" />}
                        </button>
                        <button 
                          onClick={() => handleDeleteAd(ad)}
                          className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors duration-200"
                          title="Supprimer"
                        >
                          <Trash2 className="w-4 h-4" />
                        </button>
                      </div>
                    </div>

                    {/* Performance Metrics */}
                    <div className="mt-4 pt-4 border-t border-gray-100">
                      <div className="grid grid-cols-4 gap-4">
                        <div className="text-center">
                          <p className="text-lg font-semibold text-gray-900">{ad.impressions.toLocaleString()}</p>
                          <p className="text-sm text-gray-600">Impressions</p>
                        </div>
                        <div className="text-center">
                          <p className="text-lg font-semibold text-gray-900">{ad.clicks.toLocaleString()}</p>
                          <p className="text-sm text-gray-600">Clics</p>
                        </div>
                        <div className="text-center">
                          <p className="text-lg font-semibold text-gray-900">{ad.spent.toLocaleString()}</p>
                          <p className="text-sm text-gray-600">Dépensé (FCFA)</p>
                        </div>
                        <div className="text-center">
                          <div className="w-full bg-gray-200 rounded-full h-2 mb-1">
                            <div 
                              className="bg-blue-600 h-2 rounded-full" 
                              style={{ width: `${(ad.spent / ad.budget) * 100}%` }}
                            ></div>
                          </div>
                          <p className="text-sm text-gray-600">{((ad.spent / ad.budget) * 100).toFixed(0)}% Budget</p>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
              ))}
            </div>
          )}
        </div>
      </div>

      {/* Add Advertisement Modal */}
      {showAddModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">Nouvelle Publicité</h3>
              <p className="text-gray-600">Créer une nouvelle campagne publicitaire</p>
            </div>
            <div className="p-6 space-y-6">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Titre</label>
                  <input
                    type="text"
                    className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="Ex: Promo TIM MAXI"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Type</label>
                  <select className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    <option value="banner">Bannière</option>
                    <option value="popup">Pop-up</option>
                    <option value="video">Vidéo</option>
                    <option value="native">Native</option>
                  </select>
                </div>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Description</label>
                <textarea
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  rows={3}
                  placeholder="Description de la publicité..."
                ></textarea>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Image/Vidéo</label>
                <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                  <Upload className="w-8 h-8 text-gray-400 mx-auto mb-2" />
                  <p className="text-gray-600">Glissez votre fichier ici ou cliquez pour sélectionner</p>
                  <p className="text-sm text-gray-500 mt-1">PNG, JPG, MP4 jusqu'à 10MB</p>
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Budget (FCFA)</label>
                  <input
                    type="number"
                    className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="500000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Durée</label>
                  <select className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    <option value="7">7 jours</option>
                    <option value="14">14 jours</option>
                    <option value="30">30 jours</option>
                    <option value="custom">Personnalisé</option>
                  </select>
                </div>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Ciblage géographique</label>
                <div className="space-y-2">
                  <div className="flex items-center space-x-2">
                    <input type="checkbox" id="ci" className="rounded" />
                    <label htmlFor="ci">🇨🇮 Côte d'Ivoire</label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <input type="checkbox" id="sn" className="rounded" />
                    <label htmlFor="sn">🇸🇳 Sénégal</label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <input type="checkbox" id="ml" className="rounded" />
                    <label htmlFor="ml">🇲🇱 Mali</label>
                  </div>
                </div>
              </div>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => setShowAddModal(false)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors duration-200"
              >
                Annuler
              </button>
              <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors duration-200">
                Créer Publicité
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Modal Détails */}
      {showDetailsModal && selectedAd && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">Détails de la Publicité</h3>
              <p className="text-gray-600">{selectedAd.title}</p>
            </div>
            <div className="p-6 space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700">Type</label>
                  <p className="text-gray-900 capitalize">{selectedAd.type}</p>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700">Statut</label>
                  <span className={`inline-flex items-center px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(selectedAd.status)}`}>
                    {selectedAd.status === 'active' ? 'Active' : 
                     selectedAd.status === 'paused' ? 'En pause' :
                     selectedAd.status === 'scheduled' ? 'Programmée' : 'Expirée'}
                  </span>
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700">Description</label>
                <p className="text-gray-900">{selectedAd.description}</p>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700">Budget</label>
                  <p className="text-gray-900">{selectedAd.budget.toLocaleString()} FCFA</p>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700">Dépensé</label>
                  <p className="text-gray-900">{selectedAd.spent.toLocaleString()} FCFA</p>
                </div>
              </div>
              <div className="grid grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700">Impressions</label>
                  <p className="text-gray-900">{selectedAd.impressions.toLocaleString()}</p>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700">Clics</label>
                  <p className="text-gray-900">{selectedAd.clicks.toLocaleString()}</p>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700">CTR</label>
                  <p className="text-gray-900">{selectedAd.ctr}%</p>
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700">Pays ciblés</label>
                <p className="text-gray-900">{selectedAd.targetCountries.join(', ')}</p>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700">Villes ciblées</label>
                <p className="text-gray-900">{selectedAd.targetCities.join(', ')}</p>
              </div>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end">
              <button
                onClick={() => setShowDetailsModal(false)}
                className="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors duration-200"
              >
                Fermer
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Modal Édition */}
      {showEditModal && selectedAd && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">Modifier la Publicité</h3>
              <p className="text-gray-600">Édition de "{selectedAd.title}"</p>
            </div>
            <div className="p-6 space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Titre</label>
                <input
                  type="text"
                  defaultValue={selectedAd.title}
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Description</label>
                <textarea
                  defaultValue={selectedAd.description}
                  className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  rows={3}
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Budget (FCFA)</label>
                  <input
                    type="number"
                    defaultValue={selectedAd.budget}
                    className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Type</label>
                  <select 
                    defaultValue={selectedAd.type}
                    className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="banner">Bannière</option>
                    <option value="popup">Pop-up</option>
                    <option value="video">Vidéo</option>
                    <option value="native">Native</option>
                  </select>
                </div>
              </div>
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => setShowEditModal(false)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors duration-200"
              >
                Annuler
              </button>
              <button 
                onClick={() => {
                  console.log('Modifications sauvegardées pour:', selectedAd.title);
                  setShowEditModal(false);
                }}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors duration-200"
              >
                Sauvegarder
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Advertising;
