import React, { useState } from 'react';
import { 
  TrendingUp, 
  DollarSign, 
  ArrowUpRight, 
  ArrowDownLeft,
  BarChart3,
  Calendar,
  Download,
  RefreshCw,
  AlertCircle,
  CheckCircle,
  Clock,
  Zap,
  Activity,
  Building2,
  CreditCard,
  Users
} from 'lucide-react';
import { useAdminStock } from '../../hooks/useAdminStock';

const AdminStock: React.FC = () => {
  // const { user } = useAuth(); // Unused for now
  const [selectedPeriod, setSelectedPeriod] = useState('30');
  const [activeTab, setActiveTab] = useState('overview');

  // Récupérer les données réelles du stock admin avec auto-refresh
  const { stock, loading: stockLoading, refetch } = useAdminStock(true, 30000);

  // Admin stock data - TIM CASH (DONNÉES RÉELLES)
  const adminStockData = {
    currentBalance: stock?.balance || 0,
    totalCredits: stock?.total_credits || 0,
    totalDebits: stock?.total_debits || 0,
    totalCommissions: stock?.total_commissions || 0,
    monthlyGrowth: 0,
    lastUpdated: stock?.last_updated || new Date().toISOString(),
    activeUsers: 0,
    nfcTransactions: 0,
    loansSOS: 0,
    rechargeVolume: 0
  };

  // Liste vide des transactions
  const recentTransactions: any[] = [];

  // Données mensuelles vides
  const monthlyData: any[] = [];

  const getTransactionIcon = (type: string) => {
    switch (type) {
      case 'nfc_payment':
        return <Zap className="w-5 h-5 text-amber-600" />;
      case 'recharge_cinetpay':
        return <Activity className="w-5 h-5 text-orange-600" />;
      case 'loan_sos':
        return <CreditCard className="w-5 h-5 text-yellow-600" />;
      case 'business_recharge':
        return <Building2 className="w-5 h-5 text-amber-700" />;
      case 'virement_bancaire':
        return <ArrowUpRight className="w-5 h-5 text-orange-700" />;
      default:
        return <RefreshCw className="w-5 h-5 text-amber-600" />;
    }
  };

  const getTransactionTypeLabel = (type: string) => {
    switch (type) {
      case 'nfc_payment':
        return 'Paiement NFC';
      case 'recharge_cinetpay':
        return 'Recharge CinetPay';
      case 'loan_sos':
        return 'Prêt SOS';
      case 'business_recharge':
        return 'Recharge TIM BUSINESS';
      case 'virement_bancaire':
        return 'Virement Bancaire';
      default:
        return type;
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="w-4 h-4 text-amber-600" />;
      case 'pending':
        return <Clock className="w-4 h-4 text-yellow-600" />;
      case 'failed':
        return <AlertCircle className="w-4 h-4 text-red-600" />;
      default:
        return <Clock className="w-4 h-4 text-amber-600" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'bg-amber-100 text-amber-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-amber-100 text-amber-800';
    }
  };

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-amber-900">Compte Stock Administrateur</h1>
          <p className="text-amber-600 mt-2">
            Gestion centralisée des fonds de la plateforme et suivi des mouvements financiers
          </p>
        </div>
        <div className="flex space-x-4">
          <select
            value={selectedPeriod}
            onChange={(e) => setSelectedPeriod(e.target.value)}
            className="px-4 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent"
          >
            <option value="7">7 derniers jours</option>
            <option value="30">30 derniers jours</option>
            <option value="90">90 derniers jours</option>
            <option value="365">Cette année</option>
          </select>
          <button className="border border-amber-300 text-amber-700 hover:bg-amber-50 px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2">
            <Download className="w-4 h-4" />
            <span>Exporter</span>
          </button>
          <button 
            onClick={() => refetch()}
            disabled={stockLoading}
            className="bg-amber-600 hover:bg-amber-700 text-white px-4 py-2 rounded-lg font-medium transition-colors duration-200 flex items-center space-x-2 disabled:opacity-50"
          >
            <RefreshCw className={`w-4 h-4 ${stockLoading ? 'animate-spin' : ''}`} />
            <span>Actualiser</span>
          </button>
        </div>
      </div>

      {/* Balance Overview */}
      <div className="bg-gradient-to-br from-amber-600 to-orange-800 rounded-2xl shadow-xl p-8 text-white">
        <div className="flex items-center justify-between mb-6">
          <div>
            <h2 className="text-2xl font-bold mb-2">Solde du Compte Stock</h2>
            <p className="text-amber-100">
              Dernière mise à jour: {new Date(adminStockData.lastUpdated).toLocaleString('fr-FR')}
            </p>
          </div>
          <div className="w-16 h-16 bg-white bg-opacity-20 rounded-xl flex items-center justify-center">
            <TrendingUp className="w-8 h-8 text-white" />
          </div>
        </div>
        
        <div className="text-5xl font-bold mb-6">
          {(adminStockData.currentBalance / 1000000).toFixed(0)}M FCFA
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div className="bg-white bg-opacity-10 rounded-lg p-4">
            <div className="flex items-center justify-between mb-2">
              <span className="text-amber-100">Total Crédits</span>
              <ArrowDownLeft className="w-5 h-5 text-yellow-300" />
            </div>
            <div className="text-2xl font-semibold">
              +{(adminStockData.totalCredits / 1000000).toFixed(0)}M FCFA
            </div>
          </div>
          
          <div className="bg-white bg-opacity-10 rounded-lg p-4">
            <div className="flex items-center justify-between mb-2">
              <span className="text-amber-100">Total Débits</span>
              <ArrowUpRight className="w-5 h-5 text-red-300" />
            </div>
            <div className="text-2xl font-semibold">
              -{(adminStockData.totalDebits / 1000000).toFixed(0)}M FCFA
            </div>
          </div>
          
          <div className="bg-white bg-opacity-10 rounded-lg p-4">
            <div className="flex items-center justify-between mb-2">
              <span className="text-amber-100">Commissions</span>
              <DollarSign className="w-5 h-5 text-yellow-300" />
            </div>
            <div className="text-2xl font-semibold">
              +{(adminStockData.totalCommissions / 1000000).toFixed(1)}M FCFA
            </div>
          </div>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-amber-600">Croissance Mensuelle</p>
              <p className="text-3xl font-bold text-amber-700">+{adminStockData.monthlyGrowth}%</p>
            </div>
            <div className="w-12 h-12 bg-amber-100 rounded-lg flex items-center justify-center">
              <TrendingUp className="w-6 h-6 text-amber-600" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-amber-600">Utilisateurs Actifs</p>
              <p className="text-3xl font-bold text-amber-900">{adminStockData.activeUsers.toLocaleString()}</p>
            </div>
            <div className="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
              <Users className="w-6 h-6 text-orange-600" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-amber-600">Transactions NFC</p>
              <p className="text-3xl font-bold text-amber-900">{adminStockData.nfcTransactions.toLocaleString()}</p>
            </div>
            <div className="w-12 h-12 bg-yellow-100 rounded-lg flex items-center justify-center">
              <Zap className="w-6 h-6 text-yellow-600" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-amber-600">Prêts SOS</p>
              <p className="text-3xl font-bold text-amber-900">{adminStockData.loansSOS.toLocaleString()}</p>
            </div>
            <div className="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
              <CreditCard className="w-6 h-6 text-orange-600" />
            </div>
          </div>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30">
        <div className="border-b border-amber-200/50">
          <nav className="flex space-x-8 px-6">
            {[
              { id: 'overview', label: 'Vue d\'ensemble' },
              { id: 'transactions', label: 'Transactions' },
              { id: 'analytics', label: 'Analyses' },
              { id: 'reports', label: 'Rapports' }
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-200 ${
                  activeTab === tab.id
                    ? 'border-amber-500 text-amber-600'
                    : 'border-transparent text-amber-500 hover:text-amber-700 hover:border-amber-300'
                }`}
              >
                {tab.label}
              </button>
            ))}
          </nav>
        </div>

        <div className="p-6">
          {activeTab === 'overview' && (
            <div className="space-y-6">
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                {/* Recent Transactions */}
                <div>
                  <h3 className="text-lg font-semibold text-amber-900 mb-4">Transactions Récentes</h3>
                  <div className="space-y-4">
                    {recentTransactions.slice(0, 5).map((transaction) => (
                      <div key={transaction.id} className="flex items-center justify-between p-4 bg-amber-50/50 rounded-lg">
                        <div className="flex items-center space-x-4">
                          <div className="w-10 h-10 bg-amber-100 rounded-lg flex items-center justify-center shadow-sm">
                            {getTransactionIcon(transaction.type)}
                          </div>
                          <div>
                            <p className="font-medium text-amber-900">
                              {getTransactionTypeLabel(transaction.type)}
                            </p>
                            <p className="text-sm text-amber-600">{transaction.description}</p>
                            <p className="text-xs text-amber-500">
                              {new Date(transaction.date).toLocaleDateString('fr-FR')} à{' '}
                              {new Date(transaction.date).toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' })}
                            </p>
                          </div>
                        </div>
                        <div className="text-right">
                          <p className={`font-semibold ${
                            transaction.type === 'recharge_cinetpay' || transaction.type === 'nfc_payment'
                              ? 'text-amber-600' 
                              : transaction.type === 'virement_bancaire' || transaction.type === 'loan_sos'
                              ? 'text-red-600'
                              : 'text-orange-600'
                          }`}>
                            {transaction.type === 'recharge_cinetpay' || transaction.type === 'nfc_payment' ? '+' : 
                             transaction.type === 'virement_bancaire' || transaction.type === 'loan_sos' ? '-' : '+'}
                            {transaction.amount.toLocaleString()} FCFA
                          </p>
                          <span className={`inline-flex items-center px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(transaction.status)}`}>
                            {getStatusIcon(transaction.status)}
                            <span className="ml-1">
                              {transaction.status === 'completed' ? 'Terminé' :
                               transaction.status === 'pending' ? 'En attente' : transaction.status}
                            </span>
                          </span>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>

                {/* Monthly Performance */}
                <div>
                  <h3 className="text-lg font-semibold text-amber-900 mb-4">Performance Mensuelle</h3>
                  <div className="bg-amber-50/50 rounded-lg p-6">
                    <div className="space-y-4">
                      {monthlyData.slice(-3).map((month) => (
                        <div key={month.month} className="flex items-center justify-between">
                          <div className="flex items-center space-x-3">
                            <div className="w-8 h-8 bg-amber-100 rounded-lg flex items-center justify-center">
                              <Calendar className="w-4 h-4 text-amber-600" />
                            </div>
                            <span className="font-medium text-amber-900">{month.month} 2024</span>
                          </div>
                          <div className="text-right">
                            <p className="font-semibold text-amber-900">
                              {((month.credits - month.debits + month.commissions) / 1000000).toFixed(1)}M FCFA
                            </p>
                            <p className="text-sm text-amber-600">
                              Net: +{((month.credits - month.debits) / 1000000).toFixed(1)}M FCFA
                            </p>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}

          {activeTab === 'transactions' && (
            <div className="text-center py-12">
              <RefreshCw className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Historique Complet des Transactions</h3>
              <p className="text-gray-600">Fonctionnalité en cours de développement</p>
            </div>
          )}

          {activeTab === 'analytics' && (
            <div className="text-center py-12">
              <BarChart3 className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Analyses Avancées</h3>
              <p className="text-gray-600">Graphiques et métriques détaillées en cours de développement</p>
            </div>
          )}

          {activeTab === 'reports' && (
            <div className="text-center py-12">
              <Download className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Rapports Financiers</h3>
              <p className="text-gray-600">Génération de rapports automatisés en cours de développement</p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default AdminStock;