import React, { useState, useEffect } from 'react';
import { 
  CreditCard, 
  Search, 
  Filter, 
  RefreshCw,
  CheckCircle,
  XCircle,
  Clock,
  AlertCircle,
  DollarSign,
  Users,
  TrendingUp,
  Calendar
} from 'lucide-react';
import apiService from '../../services/api';

interface VirtualCardPurchase {
  purchase_id: string;
  user_id: string;
  user_phone: string;
  user_name: string;
  user_account_type: string;
  card_color: string;
  amount: number;
  commission_amount: number;
  total_amount: number;
  status: string;
  payment_method: string;
  payment_transaction_id: string;
  card_id: string | null;
  device_info: string;
  ip_address: string;
  failure_reason: string | null;
  created_at: string;
  payment_initiated_at: string | null;
  payment_completed_at: string | null;
  card_created_at: string | null;
  card?: {
    card_uid: string;
    card_number: string;
    balance: number;
    status: string;
  };
}

const VirtualCardPurchases: React.FC = () => {
  const [purchases, setPurchases] = useState<VirtualCardPurchase[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [colorFilter, setColorFilter] = useState('all');
  const [totalCount, setTotalCount] = useState(0);
  const [totalSpent, setTotalSpent] = useState(0);

  const fetchPurchases = async () => {
    try {
      setLoading(true);
      const params: any = { limit: 1000 };
      if (statusFilter !== 'all') params.status_filter = statusFilter;
      if (colorFilter !== 'all') params.color_filter = colorFilter;

      const response = await apiService.get('/admin/virtual-cards/purchases', { params });
      setPurchases(response.purchases || []);
      setTotalCount(response.total_count || 0);
      setTotalSpent(response.total_spent || 0);
    } catch (error) {
      console.error('Erreur chargement achats:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchPurchases();
  }, [statusFilter, colorFilter]);

  const getStatusBadge = (status: string) => {
    const statusConfig: Record<string, { color: string; icon: any; label: string }> = {
      pending: { color: 'bg-gray-100 text-gray-800', icon: Clock, label: 'En attente' },
      payment_initiated: { color: 'bg-blue-100 text-blue-800', icon: Clock, label: 'Paiement initié' },
      payment_completed: { color: 'bg-green-100 text-green-800', icon: CheckCircle, label: 'Paiement complété' },
      card_created: { color: 'bg-emerald-100 text-emerald-800', icon: CheckCircle, label: 'Carte créée' },
      failed: { color: 'bg-red-100 text-red-800', icon: XCircle, label: 'Échec' },
      cancelled: { color: 'bg-orange-100 text-orange-800', icon: AlertCircle, label: 'Annulé' },
    };

    const config = statusConfig[status] || statusConfig.pending;
    const Icon = config.icon;

    return (
      <span className={`inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${config.color}`}>
        <Icon className="w-3 h-3 mr-1" />
        {config.label}
      </span>
    );
  };

  const getColorBadge = (color: string) => {
    const colorConfig: Record<string, { bg: string; text: string; label: string }> = {
      gold: { bg: 'bg-yellow-100', text: 'text-yellow-800', label: '🟡 GOLD' },
      amber: { bg: 'bg-amber-100', text: 'text-amber-800', label: '🟠 AMBER' },
      green: { bg: 'bg-green-100', text: 'text-green-800', label: '🟢 GREEN' },
      infinite: { bg: 'bg-gray-900', text: 'text-white', label: '⚫ INFINITE' },
    };

    const config = colorConfig[color] || { bg: 'bg-gray-100', text: 'text-gray-800', label: color };

    return (
      <span className={`inline-flex items-center px-3 py-1 rounded-lg text-xs font-bold ${config.bg} ${config.text}`}>
        {config.label}
      </span>
    );
  };

  const filteredPurchases = purchases.filter(purchase => {
    const matchesSearch = 
      purchase.user_phone?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      purchase.user_name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      purchase.payment_transaction_id?.toLowerCase().includes(searchTerm.toLowerCase());
    
    return matchesSearch;
  });

  const stats = [
    {
      title: 'Total Achats',
      value: totalCount.toString(),
      icon: CreditCard,
      color: 'from-blue-500 to-blue-600'
    },
    {
      title: 'Revenus Totaux',
      value: `${totalSpent.toLocaleString()} FCFA`,
      icon: DollarSign,
      color: 'from-green-500 to-green-600'
    },
    {
      title: 'Cartes Créées',
      value: purchases.filter(p => p.status === 'card_created').length.toString(),
      icon: CheckCircle,
      color: 'from-emerald-500 to-emerald-600'
    },
    {
      title: 'En Attente',
      value: purchases.filter(p => p.status === 'payment_initiated' || p.status === 'pending').length.toString(),
      icon: Clock,
      color: 'from-orange-500 to-orange-600'
    },
  ];

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-amber-50 via-orange-50 to-red-50 rounded-xl shadow-lg border border-amber-200 p-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-amber-900 flex items-center space-x-3">
              <CreditCard className="w-8 h-8" />
              <span>💳 Achats Cartes Virtuelles</span>
            </h1>
            <p className="text-amber-700 mt-2">
              Gérez et surveillez tous les achats de cartes virtuelles de la plateforme
            </p>
          </div>
          <button
            onClick={fetchPurchases}
            className="flex items-center space-x-2 px-4 py-2 bg-white border border-amber-300 rounded-lg hover:bg-amber-50 transition-colors"
          >
            <RefreshCw className={`w-4 h-4 ${loading ? 'animate-spin' : ''}`} />
            <span>Actualiser</span>
          </button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        {stats.map((stat, index) => (
          <div key={index} className="bg-white rounded-xl shadow-md border border-gray-200 p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 font-medium">{stat.title}</p>
                <p className="text-2xl font-bold text-gray-900 mt-1">{stat.value}</p>
              </div>
              <div className={`w-12 h-12 rounded-lg bg-gradient-to-br ${stat.color} flex items-center justify-center`}>
                <stat.icon className="w-6 h-6 text-white" />
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Filters */}
      <div className="bg-white rounded-xl shadow-md border border-gray-200 p-6">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              <Search className="w-4 h-4 inline mr-1" />
              Rechercher
            </label>
            <input
              type="text"
              placeholder="Téléphone, nom, transaction..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              <Filter className="w-4 h-4 inline mr-1" />
              Statut
            </label>
            <select
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent"
            >
              <option value="all">Tous les statuts</option>
              <option value="pending">En attente</option>
              <option value="payment_initiated">Paiement initié</option>
              <option value="payment_completed">Paiement complété</option>
              <option value="card_created">Carte créée</option>
              <option value="failed">Échec</option>
              <option value="cancelled">Annulé</option>
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              <CreditCard className="w-4 h-4 inline mr-1" />
              Couleur
            </label>
            <select
              value={colorFilter}
              onChange={(e) => setColorFilter(e.target.value)}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent"
            >
              <option value="all">Toutes les couleurs</option>
              <option value="gold">🟡 GOLD</option>
              <option value="amber">🟠 AMBER</option>
              <option value="green">🟢 GREEN</option>
              <option value="infinite">⚫ INFINITE</option>
            </select>
          </div>
        </div>
      </div>

      {/* Table */}
      <div className="bg-white rounded-xl shadow-md border border-gray-200 overflow-hidden">
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gradient-to-r from-amber-100 to-orange-100">
              <tr>
                <th className="px-6 py-4 text-left text-xs font-semibold text-amber-900 uppercase">Date</th>
                <th className="px-6 py-4 text-left text-xs font-semibold text-amber-900 uppercase">Utilisateur</th>
                <th className="px-6 py-4 text-left text-xs font-semibold text-amber-900 uppercase">Couleur</th>
                <th className="px-6 py-4 text-left text-xs font-semibold text-amber-900 uppercase">Montant</th>
                <th className="px-6 py-4 text-left text-xs font-semibold text-amber-900 uppercase">Statut</th>
                <th className="px-6 py-4 text-left text-xs font-semibold text-amber-900 uppercase">Transaction</th>
                <th className="px-6 py-4 text-left text-xs font-semibold text-amber-900 uppercase">Carte</th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {loading ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center">
                    <RefreshCw className="w-8 h-8 animate-spin text-amber-500 mx-auto" />
                    <p className="text-gray-600 mt-2">Chargement...</p>
                  </td>
                </tr>
              ) : filteredPurchases.length === 0 ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center">
                    <CreditCard className="w-12 h-12 text-gray-300 mx-auto" />
                    <p className="text-gray-600 mt-2">Aucun achat trouvé</p>
                  </td>
                </tr>
              ) : (
                filteredPurchases.map((purchase) => (
                  <tr key={purchase.purchase_id} className="hover:bg-amber-50 transition-colors">
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {new Date(purchase.created_at).toLocaleDateString('fr-FR', {
                        day: '2-digit',
                        month: '2-digit',
                        year: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit'
                      })}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="text-sm font-medium text-gray-900">{purchase.user_name || 'N/A'}</div>
                      <div className="text-sm text-gray-500">{purchase.user_phone}</div>
                      <div className="text-xs text-gray-400">{purchase.user_account_type}</div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      {getColorBadge(purchase.card_color)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="text-sm font-bold text-gray-900">{purchase.total_amount} FCFA</div>
                      {purchase.commission_amount > 0 && (
                        <div className="text-xs text-gray-500">Commission: {purchase.commission_amount} FCFA</div>
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      {getStatusBadge(purchase.status)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="text-xs font-mono text-gray-600">{purchase.payment_transaction_id}</div>
                      {purchase.payment_method && (
                        <div className="text-xs text-gray-500 mt-1">{purchase.payment_method}</div>
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      {purchase.card ? (
                        <div className="text-xs">
                          <div className="font-mono text-gray-900">{purchase.card.card_number}</div>
                          <div className="text-gray-500">{purchase.card.balance} FCFA</div>
                        </div>
                      ) : (
                        <span className="text-xs text-gray-400">-</span>
                      )}
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Footer */}
      <div className="text-center text-sm text-gray-500">
        {filteredPurchases.length} achat(s) affiché(s) sur {totalCount} au total
      </div>
    </div>
  );
};

export default VirtualCardPurchases;

