import React, { useState, useEffect } from 'react';
import { 
  Settings, 
  Plus, 
  Edit3, 
  Trash2, 
  Save, 
  X, 
  DollarSign,
  Percent,
  AlertCircle,
  CheckCircle,
  RefreshCw,
  Smartphone,
  Wifi,
  Calculator
} from 'lucide-react';
import { CommissionRate } from '../../types';
import { apiService } from '../../services/apiService';
import { sweetAlert } from '../../services/sweetAlertService';

interface CommissionManagementProps {}

const CommissionManagement: React.FC<CommissionManagementProps> = () => {
  const [commissionRates, setCommissionRates] = useState<CommissionRate[]>([]);
  const [isEditing, setIsEditing] = useState<string | null>(null);
  const [isCreating, setIsCreating] = useState(false);
  const [loading, setLoading] = useState(true);
  const [mobileStatus, setMobileStatus] = useState<any>(null);
  const [activatingMobile, setActivatingMobile] = useState(false);
  const [rechargeAmount, setRechargeAmount] = useState<number>(10000);
  const [rechargeBreakdown, setRechargeBreakdown] = useState<any>(null);

  // Charger les commissions depuis l'API
  const fetchCommissions = async () => {
    try {
      setLoading(true);
      console.log('🔄 Chargement des commissions...');
      const response = await apiService.getCommissionRates();
      console.log('✅ Commissions chargées:', response);
      
      // Mapper les données de l'API (snake_case) vers le format frontend (camelCase)
      const mappedCommissions = (response.commission_rates || []).map((rate: any) => ({
        id: rate.id,
        name: rate.name,
        type: rate.rate_type,
        value: rate.value,
        category: rate.category,
        description: rate.description,
        isActive: rate.is_active, // Mapping snake_case vers camelCase
        minAmount: rate.min_amount,
        maxAmount: rate.max_amount,
        updatedBy: rate.updated_by,
        createdAt: rate.created_at,
        updatedAt: rate.updated_at
      }));
      
      console.log('📊 Commissions mappées:', mappedCommissions);
      setCommissionRates(mappedCommissions);
    } catch (err: any) {
      console.error('❌ Erreur chargement commissions:', err);
      setCommissionRates([]);
    } finally {
      setLoading(false);
    }
  };

  // Calculer la répartition des commissions de recharge
  const calculateRechargeBreakdown = async (amount: number) => {
    try {
      console.log('🔄 Calcul répartition recharge pour:', amount, 'FCFA');
      const response = await apiService.getRechargeCommissionBreakdown(amount);
      console.log('✅ Répartition calculée:', response);
      setRechargeBreakdown(response);
    } catch (err: any) {
      console.error('❌ Erreur calcul répartition:', err);
      setRechargeBreakdown(null);
    }
  };

  // Charger le statut mobile
  const fetchMobileStatus = async () => {
    try {
      const status = await apiService.getMobileCommissionStatus();
      setMobileStatus(status);
    } catch (err: any) {
      console.error('❌ Erreur chargement statut mobile:', err);
    }
  };

  // Activer toutes les commissions
  const handleActivateAll = async () => {
    const result = await sweetAlert.confirm(
      'Activer Toutes les Commissions',
      'Voulez-vous activer toutes les commissions inactives ? Elles seront immédiatement disponibles pour les transactions.',
      {
        confirmButtonText: 'Oui, activer toutes',
        cancelButtonText: 'Annuler',
        confirmButtonColor: '#10B981'
      }
    );
    
    if (!result.isConfirmed) {
      return;
    }

    try {
      console.log('🔄 Activation de toutes les commissions...');
      
      const response = await apiService.activateAllCommissions();
      console.log('✅ Toutes les commissions activées:', response);
      
      // Mise à jour immédiate locale pour feedback visuel instantané
      console.log('🔄 Mise à jour locale des commissions...');
      setCommissionRates(prevRates => {
        const updatedRates = prevRates.map(rate => ({ ...rate, isActive: true }));
        console.log('📊 Commissions mises à jour localement:', updatedRates.length, 'commissions');
        return updatedRates;
      });
      
      // Recharger les données depuis le serveur pour confirmation
      console.log('🔄 Rechargement des données après activation...');
      await Promise.all([
        fetchCommissions(),
        fetchMobileStatus()
      ]);
      console.log('✅ Données rechargées avec succès');
      
      sweetAlert.success(
        'Commissions Activées',
        `${response.activated_count} commissions ont été activées avec succès !`
      );
    } catch (err: any) {
      console.error('❌ Erreur activation toutes commissions:', err);
      sweetAlert.error('Erreur', err.message || 'Impossible d\'activer les commissions');
    }
  };

  // Activer les commissions dans l'app mobile
  const handleActivateMobile = async () => {
    const result = await sweetAlert.confirm(
      'Activer les Commissions Mobile',
      'Voulez-vous synchroniser et activer toutes les commissions configurées dans l\'application mobile ?',
      {
        confirmButtonText: 'Oui, activer',
        cancelButtonText: 'Annuler',
        confirmButtonColor: '#10B981'
      }
    );
    
    if (!result.isConfirmed) {
      return;
    }

    try {
      setActivatingMobile(true);
      console.log('📱 Activation des commissions mobile...');
      
      const response = await apiService.activateCommissionsMobile();
      console.log('✅ Commissions mobile activées:', response);
      
      // Recharger le statut mobile
      await fetchMobileStatus();
      
      sweetAlert.success(
        'Commissions Activées',
        `${response.summary.total_commissions} commissions ont été activées avec succès dans l'application mobile !`
      );
    } catch (err: any) {
      console.error('❌ Erreur activation mobile:', err);
      sweetAlert.error('Erreur', err.message || 'Impossible d\'activer les commissions mobile');
    } finally {
      setActivatingMobile(false);
    }
  };

  useEffect(() => {
    fetchCommissions();
    fetchMobileStatus();
    // Calculer la répartition par défaut pour 10,000 FCFA
    calculateRechargeBreakdown(10000);
  }, []);

  const [newCommission, setNewCommission] = useState<Partial<CommissionRate>>({
    name: '',
    type: 'percentage',
    value: 0,
    category: 'recharge_cinetpay',
    description: '',
    isActive: true
  });

  const getCategoryLabel = (category: string) => {
    const labels = {
      'recharge_cinetpay': 'Recharge CinetPay',
      'recharge_business': 'Recharge TIM BUSINESS',
      'nfc_transaction': 'Transaction NFC',
      'loan_interest': 'Intérêts Prêt',
      'loan_penalty': 'Pénalité Prêt',
      'withdrawal': 'Retrait',
      'transfer': 'Transfert',
      'bank_transfer': 'Virement Bancaire'
    };
    return labels[category as keyof typeof labels] || category;
  };

  const getCategoryColor = (category: string) => {
    const colors = {
      'recharge_cinetpay': 'bg-green-100 text-green-800',
      'recharge_business': 'bg-purple-100 text-purple-800',
      'nfc_transaction': 'bg-blue-100 text-blue-800',
      'loan_interest': 'bg-orange-100 text-orange-800',
      'loan_penalty': 'bg-red-100 text-red-800',
      'withdrawal': 'bg-yellow-100 text-yellow-800',
      'transfer': 'bg-indigo-100 text-indigo-800',
      'bank_transfer': 'bg-gray-100 text-gray-800'
    };
    return colors[category as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const handleSave = async (id: string, updatedRate: Partial<CommissionRate>) => {
    try {
      console.log('💾 Mise à jour commission:', id, updatedRate);
      const response = await apiService.put(`/api/admin/commission-rates/id/${id}`, {
        name: updatedRate.name,
        rate_type: updatedRate.type,
        value: updatedRate.value,
        category: updatedRate.category,
        description: updatedRate.description,
        min_amount: updatedRate.minAmount,
        max_amount: updatedRate.maxAmount,
        is_active: updatedRate.isActive
      });
      console.log('✅ Commission mise à jour:', response);
      
      // Recharger les commissions
      await fetchCommissions();
      setIsEditing(null);
      sweetAlert.success('Commission Mise à Jour', 'La commission a été mise à jour avec succès !');
    } catch (err: any) {
      console.error('❌ Erreur mise à jour commission:', err);
      sweetAlert.error('Erreur', err.message || 'Impossible de mettre à jour la commission');
    }
  };

  const handleCreate = async () => {
    try {
      console.log('➕ Création commission:', newCommission);
      const response = await apiService.post('/api/admin/commission-rates', {
        name: newCommission.name,
        rate_type: newCommission.type,
        value: newCommission.value,
        category: newCommission.category,
        description: newCommission.description,
        min_amount: newCommission.minAmount,
        max_amount: newCommission.maxAmount,
        is_active: newCommission.isActive
      });
      console.log('✅ Commission créée:', response);
      
      // Recharger les commissions
      await fetchCommissions();
      setNewCommission({
        name: '',
        type: 'percentage',
        value: 0,
        category: 'recharge_cinetpay',
        description: '',
        isActive: true
      });
      setIsCreating(false);
      sweetAlert.success('Commission Créée', 'La commission a été créée avec succès !');
    } catch (err: any) {
      console.error('❌ Erreur création commission:', err);
      sweetAlert.error('Erreur', err.message || 'Impossible de créer la commission');
    }
  };

  const handleDelete = async (id: string) => {
    const result = await sweetAlert.confirm(
      'Supprimer la Commission',
      'Êtes-vous sûr de vouloir supprimer cette commission ? Cette action est irréversible.',
      {
        confirmButtonText: 'Oui, supprimer',
        cancelButtonText: 'Annuler',
        confirmButtonColor: '#EF4444'
      }
    );
    
    if (!result.isConfirmed) {
      return;
    }
    
    try {
      console.log('🗑️ Suppression commission:', id);
      await apiService.delete(`/api/admin/commission-rates/${id}`);
      console.log('✅ Commission supprimée');
      
      // Recharger les commissions
      await fetchCommissions();
      sweetAlert.success('Commission Supprimée', 'La commission a été supprimée avec succès !');
    } catch (err: any) {
      console.error('❌ Erreur suppression commission:', err);
      sweetAlert.error('Erreur', err.message || 'Impossible de supprimer la commission');
    }
  };

  const toggleActive = async (id: string, currentStatus: boolean) => {
    const rate = commissionRates.find(r => r.id === id);
    if (!rate) return;
    
    await handleSave(id, { ...rate, isActive: !currentStatus });
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600"></div>
        <p className="mt-4 text-amber-700 font-medium">Chargement des commissions...</p>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header avec thème marron doré */}
      <div className="bg-gradient-to-r from-amber-50/90 via-orange-50/90 to-red-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div className="p-3 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg shadow-md">
              <Settings className="w-6 h-6 text-white" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-amber-900">💰 Gestion des Commissions</h1>
              <p className="text-amber-700">Configurez et ajustez les taux de commission TIM CASH</p>
            </div>
          </div>
          <div className="flex space-x-3">
            <button
              onClick={handleActivateAll}
              disabled={loading || commissionRates.filter(r => !r.isActive).length === 0}
              className="flex items-center space-x-2 bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white px-4 py-2 rounded-lg transition-all shadow-md hover:shadow-lg transform hover:scale-105 disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none"
            >
              <CheckCircle className="w-4 h-4" />
              <span>Activer Toutes</span>
            </button>
            <button
              onClick={handleActivateMobile}
              disabled={activatingMobile || commissionRates.filter(r => r.isActive).length === 0}
              className="flex items-center space-x-2 bg-gradient-to-r from-green-600 to-emerald-600 hover:from-green-700 hover:to-emerald-700 text-white px-4 py-2 rounded-lg transition-all shadow-md hover:shadow-lg transform hover:scale-105 disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none"
            >
              {activatingMobile ? (
                <RefreshCw className="w-4 h-4 animate-spin" />
              ) : mobileStatus?.mobile_status?.is_synchronized ? (
                <Wifi className="w-4 h-4" />
              ) : (
                <Smartphone className="w-4 h-4" />
              )}
              <span>
                {activatingMobile 
                  ? 'Activation...' 
                  : mobileStatus?.mobile_status?.is_synchronized 
                    ? 'Resynchroniser Mobile'
                    : 'Activer Mobile'
                }
              </span>
            </button>
            <button
              onClick={() => setIsCreating(true)}
              className="flex items-center space-x-2 bg-gradient-to-r from-amber-600 via-orange-600 to-red-600 hover:from-amber-700 hover:via-orange-700 hover:to-red-700 text-white px-4 py-2 rounded-lg transition-all shadow-md hover:shadow-lg transform hover:scale-105"
            >
              <Plus className="w-4 h-4" />
              <span>Nouvelle Commission</span>
            </button>
          </div>
        </div>
      </div>

      {/* Statut Mobile */}
      {mobileStatus && (
        <div className="bg-gradient-to-r from-blue-50/90 via-indigo-50/90 to-purple-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-blue-200/30 p-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <div className="p-2 bg-green-100 rounded-lg">
                <Smartphone className="w-5 h-5 text-green-600" />
              </div>
              <div>
                <h3 className="text-lg font-semibold text-gray-900">📱 Statut Application Mobile</h3>
                <p className="text-sm text-gray-600">
                  {mobileStatus?.mobile_status?.total_commissions || 0} commissions actives
                </p>
              </div>
            </div>
            <div className="text-right">
              <span className={`inline-flex px-3 py-1 text-sm font-medium rounded-full ${
                mobileStatus?.mobile_status?.is_synchronized 
                  ? 'bg-green-100 text-green-800' 
                  : 'bg-yellow-100 text-yellow-800'
              }`}>
                {mobileStatus?.mobile_status?.is_synchronized ? 'Synchronisé' : 'Non synchronisé'}
              </span>
              {mobileStatus?.mobile_status?.last_sync && (
                <p className="text-xs text-gray-500 mt-1">
                  Dernière sync: {new Date(mobileStatus.mobile_status.last_sync).toLocaleString('fr-FR')}
                </p>
              )}
            </div>
          </div>
        </div>
      )}

      {/* Calculatrice Commission Recharge */}
      <div className="bg-white rounded-lg shadow-md p-6 mb-6">
        <div className="flex items-center space-x-3 mb-4">
          <div className="p-2 bg-blue-100 rounded-lg">
            <Calculator className="w-5 h-5 text-blue-600" />
          </div>
          <div>
            <h3 className="text-lg font-semibold text-gray-900">💳 Commission Recharge</h3>
            <p className="text-sm text-gray-600">
              Calculer la répartition des commissions pour Tim Mini, Tim Maxi, Tim Business
            </p>
          </div>
        </div>
        
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Calculatrice */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Montant de la recharge (FCFA)
            </label>
            <div className="flex space-x-3">
              <input
                type="number"
                value={rechargeAmount}
                onChange={(e) => setRechargeAmount(Number(e.target.value))}
                className="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                placeholder="10000"
                min="1"
              />
              <button
                onClick={() => calculateRechargeBreakdown(rechargeAmount)}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                Calculer
              </button>
            </div>
          </div>

          {/* Résultats */}
          {rechargeBreakdown && (
            <div className="bg-gray-50 rounded-lg p-4">
              <h4 className="font-medium text-gray-900 mb-3">Répartition des Commissions</h4>
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-gray-600">Montant total:</span>
                  <span className="font-medium">{rechargeBreakdown.breakdown?.total_amount?.toLocaleString()} FCFA</span>
                </div>
                <div className="flex justify-between text-red-600">
                  <span>Commission totale (3%):</span>
                  <span className="font-medium">-{rechargeBreakdown.breakdown?.total_commission?.toLocaleString()} FCFA</span>
                </div>
                <div className="flex justify-between text-orange-600 pl-4">
                  <span>• CinetPay (1%):</span>
                  <span>{rechargeBreakdown.breakdown?.cinetpay_commission?.toLocaleString()} FCFA</span>
                </div>
                <div className="flex justify-between text-blue-600 pl-4">
                  <span>• TimCash (2%):</span>
                  <span>{rechargeBreakdown.breakdown?.timcash_commission?.toLocaleString()} FCFA</span>
                </div>
                <hr className="my-2" />
                <div className="flex justify-between font-medium text-green-600">
                  <span>Montant net reçu:</span>
                  <span>{rechargeBreakdown.breakdown?.net_amount?.toLocaleString()} FCFA</span>
                </div>
              </div>
              <div className="mt-3 text-xs text-gray-500">
                Note: Toutes les commissions sont versées chez CinetPay
              </div>
            </div>
          )}
        </div>
      </div>

      {/* Statistiques avec thème marron doré */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-lg border border-amber-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-amber-700 font-medium">Total Commissions</p>
              <p className="text-2xl font-bold text-amber-900">{commissionRates.length}</p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-amber-400 to-orange-500 rounded-lg flex items-center justify-center shadow-md">
              <DollarSign className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-lg border border-green-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-green-700 font-medium">Actives</p>
              <p className="text-2xl font-bold text-green-900">
                {commissionRates.filter(r => r.isActive).length}
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-green-400 to-emerald-500 rounded-lg flex items-center justify-center shadow-md">
              <CheckCircle className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-xl shadow-lg border border-red-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-red-700 font-medium">Inactives</p>
              <p className="text-2xl font-bold text-red-900">
                {commissionRates.filter(r => !r.isActive).length}
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-red-400 to-rose-500 rounded-lg flex items-center justify-center shadow-md">
              <AlertCircle className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
        <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl shadow-lg border border-blue-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-blue-700 font-medium">Taux Moyen</p>
              <p className="text-2xl font-bold text-blue-900">
                {(commissionRates.filter(r => r.type === 'percentage')
                  .reduce((sum, r) => sum + r.value, 0) / 
                  commissionRates.filter(r => r.type === 'percentage').length || 0).toFixed(1)}%
              </p>
            </div>
            <div className="w-12 h-12 bg-gradient-to-br from-blue-400 to-indigo-500 rounded-lg flex items-center justify-center shadow-md">
              <Percent className="w-6 h-6 text-white" />
            </div>
          </div>
        </div>
      </div>

      {/* Formulaire de création */}
      {isCreating && (
        <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-lg border border-amber-200 p-6">
          <h3 className="text-lg font-semibold text-amber-900 mb-4">Nouvelle Commission</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Nom</label>
              <input
                type="text"
                value={newCommission.name}
                onChange={(e) => setNewCommission(prev => ({ ...prev, name: e.target.value }))}
                className="w-full px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
                placeholder="Nom de la commission"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Catégorie</label>
              <select
                value={newCommission.category}
                onChange={(e) => setNewCommission(prev => ({ ...prev, category: e.target.value as any }))}
                className="w-full px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              >
                <option value="recharge_cinetpay">Recharge CinetPay</option>
                <option value="recharge_business">Recharge TIM BUSINESS</option>
                <option value="nfc_transaction">Transaction NFC</option>
                <option value="loan_interest">Intérêts Prêt</option>
                <option value="loan_penalty">Pénalité Prêt</option>
                <option value="withdrawal">Retrait</option>
                <option value="transfer">Transfert</option>
                <option value="bank_transfer">Virement Bancaire</option>
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Type</label>
              <select
                value={newCommission.type}
                onChange={(e) => setNewCommission(prev => ({ ...prev, type: e.target.value as 'percentage' | 'fixed' }))}
                className="w-full px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              >
                <option value="percentage">Pourcentage</option>
                <option value="fixed">Montant fixe</option>
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Valeur {newCommission.type === 'percentage' ? '(%)' : '(FCFA)'}
              </label>
              <input
                type="number"
                step={newCommission.type === 'percentage' ? '0.1' : '1'}
                value={newCommission.value}
                onChange={(e) => setNewCommission(prev => ({ ...prev, value: parseFloat(e.target.value) }))}
                className="w-full px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
              />
            </div>
            <div className="md:col-span-2">
              <label className="block text-sm font-medium text-gray-700 mb-1">Description</label>
              <textarea
                value={newCommission.description}
                onChange={(e) => setNewCommission(prev => ({ ...prev, description: e.target.value }))}
                className="w-full px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500 bg-white"
                rows={2}
                placeholder="Description de la commission"
              />
            </div>
          </div>
          <div className="flex justify-end space-x-3 mt-4">
            <button
              onClick={() => setIsCreating(false)}
              className="px-4 py-2 text-gray-600 hover:text-gray-800 transition-colors"
            >
              Annuler
            </button>
            <button
              onClick={handleCreate}
              className="flex items-center space-x-2 bg-gradient-to-r from-amber-600 to-orange-600 text-white px-4 py-2 rounded-lg hover:from-amber-700 hover:to-orange-700 transition-all shadow-md"
            >
              <Save className="w-4 h-4" />
              <span>Créer</span>
            </button>
          </div>
        </div>
      )}

      {/* Liste des commissions avec thème marron doré */}
      <div className="bg-white rounded-xl shadow-lg border border-amber-200">
        <div className="p-6 border-b border-amber-200 bg-gradient-to-r from-amber-50/30 to-orange-50/30">
          <h3 className="text-lg font-semibold text-amber-900">Commissions Configurées</h3>
        </div>
        <div className="divide-y divide-gray-200">
          {commissionRates.length === 0 ? (
            <div className="p-12 text-center">
              <div className="inline-flex items-center justify-center w-16 h-16 bg-amber-100 rounded-full mb-4">
                <Settings className="w-8 h-8 text-amber-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Aucune commission configurée</h3>
              <p className="text-gray-600 mb-4">
                Commencez par créer votre première commission
              </p>
              <button
                onClick={() => setIsCreating(true)}
                className="inline-flex items-center space-x-2 bg-gradient-to-r from-amber-600 to-orange-600 text-white px-4 py-2 rounded-lg hover:from-amber-700 hover:to-orange-700 transition-all shadow-md"
              >
                <Plus className="w-4 h-4" />
                <span>Créer une commission</span>
              </button>
            </div>
          ) : (
            commissionRates.map((rate) => (
              <CommissionRow
                key={rate.id}
                rate={rate}
                isEditing={isEditing === rate.id}
                onEdit={() => setIsEditing(rate.id)}
                onSave={(updatedRate) => handleSave(rate.id, updatedRate)}
                onCancel={() => setIsEditing(null)}
                onDelete={() => handleDelete(rate.id)}
                onToggleActive={() => toggleActive(rate.id, rate.isActive)}
                getCategoryLabel={getCategoryLabel}
                getCategoryColor={getCategoryColor}
              />
            ))
          )}
        </div>
      </div>
    </div>
  );
};

// Composant pour une ligne de commission
interface CommissionRowProps {
  rate: CommissionRate;
  isEditing: boolean;
  onEdit: () => void;
  onSave: (updatedRate: Partial<CommissionRate>) => void;
  onCancel: () => void;
  onDelete: () => void;
  onToggleActive: () => void;
  getCategoryLabel: (category: string) => string;
  getCategoryColor: (category: string) => string;
}

const CommissionRow: React.FC<CommissionRowProps> = ({
  rate,
  isEditing,
  onEdit,
  onSave,
  onCancel,
  onDelete,
  onToggleActive,
  getCategoryLabel,
  getCategoryColor
}) => {
  const [editedRate, setEditedRate] = useState(rate);

  useEffect(() => {
    setEditedRate(rate);
  }, [rate]);

  if (isEditing) {
    return (
      <div className="p-6 bg-blue-50">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">Nom</label>
            <input
              type="text"
              value={editedRate.name}
              onChange={(e) => setEditedRate(prev => ({ ...prev, name: e.target.value }))}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">Type</label>
            <select
              value={editedRate.type}
              onChange={(e) => setEditedRate(prev => ({ ...prev, type: e.target.value as 'percentage' | 'fixed' }))}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              <option value="percentage">Pourcentage</option>
              <option value="fixed">Montant fixe</option>
            </select>
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Valeur {editedRate.type === 'percentage' ? '(%)' : '(FCFA)'}
            </label>
            <input
              type="number"
              step={editedRate.type === 'percentage' ? '0.1' : '1'}
              value={editedRate.value}
              onChange={(e) => setEditedRate(prev => ({ ...prev, value: parseFloat(e.target.value) }))}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div className="md:col-span-3">
            <label className="block text-sm font-medium text-gray-700 mb-1">Description</label>
            <textarea
              value={editedRate.description}
              onChange={(e) => setEditedRate(prev => ({ ...prev, description: e.target.value }))}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              rows={2}
            />
          </div>
        </div>
        <div className="flex justify-end space-x-3 mt-4">
          <button
            onClick={onCancel}
            className="flex items-center space-x-2 px-4 py-2 text-gray-600 hover:text-gray-800 transition-colors"
          >
            <X className="w-4 h-4" />
            <span>Annuler</span>
          </button>
          <button
            onClick={() => onSave(editedRate)}
            className="flex items-center space-x-2 bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors"
          >
            <Save className="w-4 h-4" />
            <span>Sauvegarder</span>
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6 hover:bg-gray-50 transition-colors">
      <div className="flex items-center justify-between">
        <div className="flex-1">
          <div className="flex items-center space-x-3 mb-2">
            <h4 className="font-medium text-gray-900">{rate.name}</h4>
            <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getCategoryColor(rate.category)}`}>
              {getCategoryLabel(rate.category)}
            </span>
            <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${
              rate.isActive ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
            }`}>
              {rate.isActive ? 'Actif' : 'Inactif'}
            </span>
          </div>
          <p className="text-sm text-gray-600 mb-2">{rate.description}</p>
          <div className="flex items-center space-x-4 text-sm text-gray-500">
            <span className="flex items-center space-x-1">
              {rate.type === 'percentage' ? <Percent className="w-4 h-4" /> : <DollarSign className="w-4 h-4" />}
              <span>
                {rate.value}{rate.type === 'percentage' ? '%' : ' FCFA'}
              </span>
            </span>
            {rate.minAmount && (
              <span>Min: {rate.minAmount.toLocaleString()} FCFA</span>
            )}
            {rate.maxAmount && (
              <span>Max: {rate.maxAmount.toLocaleString()} FCFA</span>
            )}
          </div>
        </div>
        <div className="flex items-center space-x-2">
          <button
            onClick={onToggleActive}
            className={`p-2 rounded-lg transition-colors ${
              rate.isActive 
                ? 'text-green-600 hover:bg-green-100' 
                : 'text-red-600 hover:bg-red-100'
            }`}
            title={rate.isActive ? 'Désactiver' : 'Activer'}
          >
            {rate.isActive ? <CheckCircle className="w-4 h-4" /> : <AlertCircle className="w-4 h-4" />}
          </button>
          <button
            onClick={onEdit}
            className="p-2 text-blue-600 hover:bg-blue-100 rounded-lg transition-colors"
            title="Modifier"
          >
            <Edit3 className="w-4 h-4" />
          </button>
          <button
            onClick={onDelete}
            className="p-2 text-red-600 hover:bg-red-100 rounded-lg transition-colors"
            title="Supprimer"
          >
            <Trash2 className="w-4 h-4" />
          </button>
        </div>
      </div>
    </div>
  );
};

export default CommissionManagement;
