import React, { useState, useEffect } from 'react';
import { 
  Search, 
  Shield, 
  ShieldCheck, 
  AlertTriangle, 
  Eye,
  RefreshCw,
  Filter,
  CreditCard,
  Users,
  Lock,
  Unlock,
  Plus,
  TrendingUp,
  Activity
} from 'lucide-react';
import { apiService } from '../../services/apiService';

interface CardData {
  id: string;
  uid: string;
  user_id: string;
  status: 'active' | 'blocked' | 'lost' | 'stolen';
  created_at: string;
  updated_at?: string;
  user_phone?: string;
  user_name?: string;
}

interface CardRechargeData {
  phone: string;
  amount: number;
  reason?: string;
}

const CardManagement: React.FC = () => {
  const [cards, setCards] = useState<CardData[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedCard, setSelectedCard] = useState<CardData | null>(null);
  const [showRechargeModal, setShowRechargeModal] = useState(false);
  const [rechargeData, setRechargeData] = useState<CardRechargeData>({
    phone: '',
    amount: 0,
    reason: ''
  });
  const [stats, setStats] = useState({
    total: 0,
    active: 0,
    blocked: 0,
    lost: 0,
    stolen: 0
  });

  useEffect(() => {
    fetchCards();
  }, []);

  const fetchCards = async () => {
    try {
      setLoading(true);
      // Utiliser le nouvel endpoint admin pour les cartes NFC
      const response = await apiService.get('/admin/nfc-cards?limit=1000');
      const cardsList = response.cards || [];
      setCards(cardsList);

      // Calculer les statistiques
      const total = response.total_count || cardsList.length;
      const active = cardsList.filter((c: CardData) => c.status === 'active').length;
      const blocked = cardsList.filter((c: CardData) => c.status === 'blocked').length;
      const lost = cardsList.filter((c: CardData) => c.status === 'lost').length;
      const stolen = cardsList.filter((c: CardData) => c.status === 'stolen').length;

      setStats({ total, active, blocked, lost, stolen });
    } catch (error) {
      console.error('Erreur lors du chargement des cartes:', error);
    } finally {
      setLoading(false);
    }
  };

  const searchCards = async () => {
    if (!searchQuery.trim()) {
      fetchCards();
      return;
    }

    try {
      setLoading(true);
      const response = await apiService.get(`/api/admin/cards/search?query=${encodeURIComponent(searchQuery)}`);
      setCards(response.cards || response || []);
    } catch (error) {
      console.error('Erreur lors de la recherche:', error);
    } finally {
      setLoading(false);
    }
  };

  const updateCardStatus = async (cardId: string, status: string, reason?: string) => {
    try {
      // Utiliser le nouvel endpoint admin pour mettre à jour le statut
      const response = await apiService.put(`/admin/nfc-cards/${cardId}/status`, {
        status,
        reason
      });

      console.log('✅ Statut de la carte mis à jour:', response);

      // Mettre à jour la liste locale
      setCards(prev => prev.map(card =>
        card.id === cardId
          ? { ...card, status: status as any, updated_at: new Date().toISOString() }
          : card
      ));

      fetchCards(); // Recharger pour avoir les stats à jour

      // Afficher un message de succès
      alert(`Carte ${status === 'active' ? 'activée' : 'désactivée'} avec succès. La modification sera synchronisée avec l'application mobile.`);
    } catch (error) {
      console.error('Erreur lors de la mise à jour:', error);
      alert('Erreur lors de la mise à jour du statut de la carte');
    }
  };

  const blockCard = async (cardId: string, reason: string, isLost: boolean = false, isStolen: boolean = false) => {
    try {
      await apiService.post(`/api/cards/${cardId}/block`, {
        reason,
        is_lost: isLost,
        is_stolen: isStolen
      });
      
      fetchCards();
    } catch (error) {
      console.error('Erreur lors du blocage:', error);
    }
  };

  const unblockCard = async (cardId: string) => {
    try {
      await apiService.post(`/api/cards/${cardId}/unblock`);
      fetchCards();
    } catch (error) {
      console.error('Erreur lors du déblocage:', error);
    }
  };

  const rechargeCard = async () => {
    try {
      await apiService.post('/api/cards/recharge', rechargeData);
      setShowRechargeModal(false);
      setRechargeData({ phone: '', amount: 0, reason: '' });
      alert('Recharge effectuée avec succès');
    } catch (error) {
      console.error('Erreur lors de la recharge:', error);
      alert('Erreur lors de la recharge');
    }
  };

  const getStatusBadge = (status: string) => {
    const statusConfig = {
      active: { 
        color: 'bg-gradient-to-r from-green-100 to-emerald-100 text-green-800 border border-green-200', 
        icon: ShieldCheck 
      },
      blocked: { 
        color: 'bg-gradient-to-r from-red-100 to-rose-100 text-red-800 border border-red-200', 
        icon: Lock 
      },
      lost: { 
        color: 'bg-gradient-to-r from-orange-100 to-amber-100 text-orange-800 border border-orange-200', 
        icon: AlertTriangle 
      },
      stolen: { 
        color: 'bg-gradient-to-r from-red-100 to-pink-100 text-red-800 border border-red-200', 
        icon: AlertTriangle 
      }
    } as const;

    const config = statusConfig[status as keyof typeof statusConfig] || statusConfig.blocked;
    const Icon = config.icon;

    return (
      <span className={`inline-flex items-center px-3 py-1.5 rounded-lg font-semibold text-sm shadow-sm ${config.color}`}>
        <Icon className="w-4 h-4 mr-1.5" />
        {status.toUpperCase()}
      </span>
    );
  };

  const filteredCards = cards.filter(card => 
    card.id.includes(searchQuery) ||
    card.uid.includes(searchQuery) ||
    card.user_phone?.includes(searchQuery) ||
    card.user_name?.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <div className="min-h-screen bg-gradient-to-br from-amber-50 via-yellow-50 to-orange-50">
      <div className="p-6 space-y-8">
        {/* Header avec design marron doré */}
        <div className="bg-gradient-to-r from-amber-600 to-yellow-600 rounded-2xl shadow-xl p-8 text-white">
          <div className="flex justify-between items-center">
            <div className="flex items-center space-x-4">
              <div className="bg-white/20 backdrop-blur-sm rounded-xl p-3">
                <CreditCard className="w-8 h-8" />
              </div>
              <div>
                <h1 className="text-3xl font-bold">Gestion des Cartes NFC</h1>
                <p className="text-amber-100 text-lg">Administration des cartes physiques</p>
              </div>
            </div>
            <div className="flex gap-3">
              <button
                onClick={() => setShowRechargeModal(true)}
                className="bg-white/20 backdrop-blur-sm hover:bg-white/30 text-white px-6 py-3 rounded-xl font-semibold transition-all duration-300 flex items-center space-x-2"
              >
                <Plus className="w-5 h-5" />
                <span>Recharger une carte</span>
              </button>
              <button
                onClick={fetchCards}
                disabled={loading}
                className="bg-white/20 backdrop-blur-sm hover:bg-white/30 text-white px-4 py-3 rounded-xl transition-all duration-300 disabled:opacity-50"
              >
                <RefreshCw className={`w-5 h-5 ${loading ? 'animate-spin' : ''}`} />
              </button>
            </div>
          </div>
        </div>

        {/* Statistiques avec design marron doré */}
        <div className="grid grid-cols-1 md:grid-cols-5 gap-6">
          <div className="bg-gradient-to-br from-amber-100 to-yellow-100 border border-amber-200 rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300">
            <div className="flex items-center justify-between">
              <div>
                <div className="text-3xl font-bold text-amber-800">{stats.total}</div>
                <p className="text-amber-600 font-medium">Total Cartes</p>
              </div>
              <div className="bg-amber-500 rounded-xl p-3">
                <CreditCard className="w-6 h-6 text-white" />
              </div>
            </div>
          </div>
          
          <div className="bg-gradient-to-br from-green-100 to-emerald-100 border border-green-200 rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300">
            <div className="flex items-center justify-between">
              <div>
                <div className="text-3xl font-bold text-green-800">{stats.active}</div>
                <p className="text-green-600 font-medium">Actives</p>
              </div>
              <div className="bg-green-500 rounded-xl p-3">
                <ShieldCheck className="w-6 h-6 text-white" />
              </div>
            </div>
          </div>
          
          <div className="bg-gradient-to-br from-red-100 to-rose-100 border border-red-200 rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300">
            <div className="flex items-center justify-between">
              <div>
                <div className="text-3xl font-bold text-red-800">{stats.blocked}</div>
                <p className="text-red-600 font-medium">Bloquées</p>
              </div>
              <div className="bg-red-500 rounded-xl p-3">
                <Lock className="w-6 h-6 text-white" />
              </div>
            </div>
          </div>
          
          <div className="bg-gradient-to-br from-orange-100 to-amber-100 border border-orange-200 rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300">
            <div className="flex items-center justify-between">
              <div>
                <div className="text-3xl font-bold text-orange-800">{stats.lost}</div>
                <p className="text-orange-600 font-medium">Perdues</p>
              </div>
              <div className="bg-orange-500 rounded-xl p-3">
                <AlertTriangle className="w-6 h-6 text-white" />
              </div>
            </div>
          </div>
          
          <div className="bg-gradient-to-br from-red-100 to-pink-100 border border-red-200 rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all duration-300">
            <div className="flex items-center justify-between">
              <div>
                <div className="text-3xl font-bold text-red-800">{stats.stolen}</div>
                <p className="text-red-600 font-medium">Volées</p>
              </div>
              <div className="bg-red-500 rounded-xl p-3">
                <AlertTriangle className="w-6 h-6 text-white" />
              </div>
            </div>
          </div>
        </div>

        {/* Barre de recherche avec design marron doré */}
        <div className="bg-white rounded-2xl shadow-lg border border-amber-200 p-6">
          <div className="flex gap-4">
            <div className="flex-1 relative">
              <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                <Search className="h-5 w-5 text-amber-400" />
              </div>
              <input
                type="text"
                placeholder="Rechercher par ID, UID, téléphone ou nom..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                onKeyPress={(e) => e.key === 'Enter' && searchCards()}
                className="w-full pl-10 pr-4 py-3 border border-amber-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-amber-500 focus:border-transparent transition-all duration-300"
              />
            </div>
            <button 
              onClick={searchCards}
              className="bg-gradient-to-r from-amber-500 to-yellow-500 hover:from-amber-600 hover:to-yellow-600 text-white px-6 py-3 rounded-xl font-semibold transition-all duration-300 flex items-center space-x-2 shadow-lg hover:shadow-xl"
            >
              <Search className="w-5 h-5" />
              <span>Rechercher</span>
            </button>
          </div>
        </div>

        {/* Table des cartes avec design marron doré */}
        <div className="bg-white rounded-2xl shadow-xl border border-amber-200 overflow-hidden">
          <div className="bg-gradient-to-r from-amber-50 to-yellow-50 px-6 py-4 border-b border-amber-200">
            <h2 className="text-xl font-bold text-amber-800 flex items-center space-x-2">
              <CreditCard className="w-6 h-6" />
              <span>Liste des Cartes</span>
            </h2>
          </div>
          <div className="overflow-x-auto">
            <table className="min-w-full">
              <thead className="bg-gradient-to-r from-amber-100 to-yellow-100">
                <tr>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-amber-800 uppercase tracking-wider">ID Carte</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-amber-800 uppercase tracking-wider">UID NFC</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-amber-800 uppercase tracking-wider">Utilisateur</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-amber-800 uppercase tracking-wider">Téléphone</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-amber-800 uppercase tracking-wider">Statut</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-amber-800 uppercase tracking-wider">Créée le</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-amber-800 uppercase tracking-wider">Actions</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-amber-100">
                {loading ? (
                  <tr>
                    <td colSpan={7} className="px-6 py-12 text-center">
                      <div className="flex flex-col items-center space-y-4">
                        <RefreshCw className="w-8 h-8 animate-spin text-amber-500" />
                        <p className="text-amber-600 font-medium">Chargement des cartes...</p>
                      </div>
                    </td>
                  </tr>
                ) : filteredCards.length === 0 ? (
                  <tr>
                    <td colSpan={7} className="px-6 py-12 text-center">
                      <div className="flex flex-col items-center space-y-4">
                        <CreditCard className="w-12 h-12 text-amber-300" />
                        <p className="text-amber-600 font-medium">Aucune carte trouvée</p>
                      </div>
                    </td>
                  </tr>
                ) : (
                  filteredCards.map((card) => (
                    <tr key={card.id} className="hover:bg-amber-50 transition-colors duration-200">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-mono font-bold text-amber-800 bg-amber-100 px-3 py-1 rounded-lg">
                          {card.id}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-mono text-gray-600 bg-gray-100 px-3 py-1 rounded-lg">
                          {card.uid}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">{card.user_name || 'N/A'}</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-600">{card.user_phone || 'N/A'}</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        {getStatusBadge(card.status)}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                        {new Date(card.created_at).toLocaleDateString()}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex space-x-2">
                          <button
                            onClick={() => setSelectedCard(card)}
                            className="bg-amber-100 hover:bg-amber-200 text-amber-700 px-3 py-2 rounded-lg transition-colors duration-200 flex items-center space-x-1"
                          >
                            <Eye className="w-4 h-4" />
                            <span className="text-sm">Voir</span>
                          </button>
                          {card.status === 'active' ? (
                            <button
                              onClick={() => {
                                const reason = prompt('Raison du blocage:');
                                if (reason) {
                                  const isLost = confirm('Carte perdue?');
                                  const isStolen = !isLost && confirm('Carte volée?');
                                  blockCard(card.id, reason, isLost, isStolen);
                                }
                              }}
                              className="bg-red-100 hover:bg-red-200 text-red-700 px-3 py-2 rounded-lg transition-colors duration-200 flex items-center space-x-1"
                            >
                              <Lock className="w-4 h-4" />
                              <span className="text-sm">Bloquer</span>
                            </button>
                          ) : (
                            <button
                              onClick={() => unblockCard(card.id)}
                              className="bg-green-100 hover:bg-green-200 text-green-700 px-3 py-2 rounded-lg transition-colors duration-200 flex items-center space-x-1"
                            >
                              <Unlock className="w-4 h-4" />
                              <span className="text-sm">Débloquer</span>
                            </button>
                          )}
                        </div>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>

        {/* Modal de recharge avec design marron doré */}
        {showRechargeModal && (
          <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
            <div className="w-full max-w-md bg-white rounded-2xl shadow-2xl border border-amber-200 overflow-hidden">
              <div className="bg-gradient-to-r from-amber-500 to-yellow-500 px-6 py-4">
                <h3 className="text-xl font-bold text-white flex items-center space-x-2">
                  <Plus className="w-6 h-6" />
                  <span>Recharger une carte</span>
                </h3>
              </div>
              <div className="p-6 space-y-6">
                <div>
                  <label className="block text-sm font-semibold text-amber-800 mb-2">
                    Numéro de téléphone
                  </label>
                  <input
                    value={rechargeData.phone}
                    onChange={(e) => setRechargeData({...rechargeData, phone: e.target.value})}
                    placeholder="Ex: +22512345678"
                    className="w-full px-4 py-3 border border-amber-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-amber-500 focus:border-transparent transition-all duration-300"
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold text-amber-800 mb-2">
                    Montant (FCFA)
                  </label>
                  <input
                    type="number"
                    value={rechargeData.amount}
                    onChange={(e) => setRechargeData({...rechargeData, amount: Number(e.target.value)})}
                    placeholder="0"
                    className="w-full px-4 py-3 border border-amber-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-amber-500 focus:border-transparent transition-all duration-300"
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold text-amber-800 mb-2">
                    Raison (optionnel)
                  </label>
                  <input
                    value={rechargeData.reason}
                    onChange={(e) => setRechargeData({...rechargeData, reason: e.target.value})}
                    placeholder="Ex: Recharge client"
                    className="w-full px-4 py-3 border border-amber-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-amber-500 focus:border-transparent transition-all duration-300"
                  />
                </div>
                <div className="flex gap-3 pt-4">
                  <button 
                    onClick={rechargeCard} 
                    className="flex-1 bg-gradient-to-r from-amber-500 to-yellow-500 hover:from-amber-600 hover:to-yellow-600 text-white px-6 py-3 rounded-xl font-semibold transition-all duration-300 shadow-lg hover:shadow-xl"
                  >
                    Recharger
                  </button>
                  <button 
                    onClick={() => setShowRechargeModal(false)}
                    className="flex-1 bg-gray-100 hover:bg-gray-200 text-gray-700 px-6 py-3 rounded-xl font-semibold transition-all duration-300"
                  >
                    Annuler
                  </button>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Modal de détails de la carte - Format réduit */}
        {selectedCard && (
          <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
            <div className="w-full max-w-lg bg-white rounded-xl shadow-2xl border border-amber-200 overflow-hidden">
              {/* Header */}
              <div className="bg-gradient-to-r from-amber-500 to-yellow-500 px-4 py-3 flex items-center justify-between">
                <h3 className="text-lg font-bold text-white flex items-center space-x-2">
                  <CreditCard className="w-5 h-5" />
                  <span>Détails Carte</span>
                </h3>
                <button
                  onClick={() => setSelectedCard(null)}
                  className="text-white hover:bg-white/20 rounded-lg p-1 transition-colors"
                >
                  ✕
                </button>
              </div>

              {/* Body */}
              <div className="p-4 space-y-3 max-h-[70vh] overflow-y-auto">
                {/* Carte & Utilisateur */}
                <div className="grid grid-cols-2 gap-2">
                  <div className="bg-amber-50 p-2 rounded-lg border border-amber-200">
                    <p className="text-xs text-amber-600 font-semibold">UID NFC</p>
                    <p className="text-xs text-gray-900 font-mono">{selectedCard.card_uid}</p>
                  </div>
                  <div className="bg-amber-50 p-2 rounded-lg border border-amber-200">
                    <p className="text-xs text-amber-600 font-semibold">N° Carte</p>
                    <p className="text-xs text-gray-900 font-mono">{selectedCard.card_number}</p>
                  </div>
                </div>

                {/* Utilisateur */}
                <div className="bg-gray-50 p-3 rounded-lg space-y-2">
                  <div className="flex items-center space-x-2 text-amber-700 mb-2">
                    <Users className="w-4 h-4" />
                    <span className="text-xs font-semibold">Utilisateur</span>
                  </div>
                  <div className="grid grid-cols-2 gap-2 text-xs">
                    <div>
                      <p className="text-gray-600">Nom</p>
                      <p className="text-gray-900 font-medium">{selectedCard.user_name || 'N/A'}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Téléphone</p>
                      <p className="text-gray-900 font-medium">{selectedCard.user_phone || 'N/A'}</p>
                    </div>
                  </div>
                </div>

                {/* Statut & Solde */}
                <div className="grid grid-cols-3 gap-2">
                  <div className="bg-gray-50 p-2 rounded-lg text-center">
                    <p className="text-xs text-gray-600 mb-1">Statut</p>
                    <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-semibold ${
                      selectedCard.status === 'active'
                        ? 'bg-green-100 text-green-700'
                        : selectedCard.status === 'blocked'
                        ? 'bg-red-100 text-red-700'
                        : 'bg-gray-100 text-gray-700'
                    }`}>
                      {selectedCard.status === 'active' ? 'Active' :
                       selectedCard.status === 'blocked' ? 'Bloquée' :
                       selectedCard.status === 'inactive' ? 'Inactive' :
                       selectedCard.status === 'lost' ? 'Perdue' :
                       selectedCard.status === 'stolen' ? 'Volée' : selectedCard.status}
                    </span>
                  </div>
                  <div className="bg-gray-50 p-2 rounded-lg text-center">
                    <p className="text-xs text-gray-600">Solde</p>
                    <p className="text-sm text-gray-900 font-bold">{selectedCard.balance || 0} F</p>
                  </div>
                  <div className="bg-gray-50 p-2 rounded-lg text-center">
                    <p className="text-xs text-gray-600">Max</p>
                    <p className="text-sm text-gray-900 font-bold">{selectedCard.max_balance || 50000} F</p>
                  </div>
                </div>

                {/* Dates */}
                <div className="bg-gray-50 p-2 rounded-lg">
                  <div className="grid grid-cols-2 gap-2 text-xs">
                    <div>
                      <p className="text-gray-600">Créée le</p>
                      <p className="text-gray-900">{new Date(selectedCard.created_at).toLocaleDateString('fr-FR')}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Dernière utilisation</p>
                      <p className="text-gray-900">
                        {selectedCard.last_used_at ? new Date(selectedCard.last_used_at).toLocaleDateString('fr-FR') : 'Jamais'}
                      </p>
                    </div>
                  </div>
                </div>
              </div>

              {/* Footer */}
              <div className="bg-gray-50 px-4 py-3 flex justify-end space-x-2">
                <button
                  onClick={() => setSelectedCard(null)}
                  className="bg-gray-200 hover:bg-gray-300 text-gray-700 px-4 py-2 rounded-lg text-sm font-semibold transition-colors"
                >
                  Fermer
                </button>
                {selectedCard.status === 'active' ? (
                  <button
                    onClick={() => {
                      const reason = prompt('Raison du blocage:');
                      if (reason) {
                        const isLost = confirm('Carte perdue?');
                        const isStolen = !isLost && confirm('Carte volée?');
                        blockCard(selectedCard.id, reason, isLost, isStolen);
                        setSelectedCard(null);
                      }
                    }}
                    className="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-lg text-sm font-semibold transition-colors flex items-center space-x-1"
                  >
                    <Lock className="w-4 h-4" />
                    <span>Bloquer</span>
                  </button>
                ) : (
                  <button
                    onClick={() => {
                      updateCardStatus(selectedCard.id, 'active');
                      setSelectedCard(null);
                    }}
                    className="bg-green-500 hover:bg-green-600 text-white px-4 py-2 rounded-lg text-sm font-semibold transition-colors flex items-center space-x-1"
                  >
                    <Unlock className="w-4 h-4" />
                    <span>Activer</span>
                  </button>
                )}
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default CardManagement;
