import { useState, useEffect } from 'react';
import { apiService } from '../services/apiService';

interface UserData {
  id: string;
  username: string;
  email: string;
  full_name: string;
  firstName?: string;
  lastName?: string;
  phone?: string;
  tim_account_type: string;
  role?: string;
  is_active: boolean;
  is_admin?: boolean;
  isVerified?: boolean;
  isBlocked?: boolean;
  wallet_balance: number;
  walletBalance?: number;
  total_transactions?: number;
  totalTransactions?: number;
  has_nfc_card?: boolean;
  last_activity?: string;
  created_at: string;
  createdAt?: string;
  lastLogin?: string;
}

interface UsersResponse {
  users: UserData[];
  total_count: number;
}

export const useUsers = (autoRefresh: boolean = false, refreshInterval: number = 60000) => {
  const [users, setUsers] = useState<UserData[]>([]);
  const [totalCount, setTotalCount] = useState(0);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchUsers = async () => {
    try {
      setLoading(true);
      setError(null);

      if (!apiService.isAuthenticated()) {
        console.warn('⚠️ useUsers: Utilisateur non authentifié');
        setUsers([]);
        return;
      }

      console.log('🔄 useUsers: Chargement des utilisateurs...');
      const data: UsersResponse = await apiService.get(`/admin/users?_t=${Date.now()}`);
      console.log('✅ useUsers: Utilisateurs chargés:', data);

      // Transformer les données pour correspondre au format attendu par le composant
      const transformedUsers = (data.users || []).map(user => ({
        ...user,
        firstName: user.full_name?.split(' ')[0] || '',
        lastName: user.full_name?.split(' ').slice(1).join(' ') || '',
        phone: user.username || '',
        role: user.is_admin ? 'admin' : 'user',
        isVerified: user.is_active,
        isBlocked: !user.is_active,
        walletBalance: user.wallet_balance,
        totalTransactions: user.total_transactions || 0,
        createdAt: user.created_at,
        lastLogin: user.last_activity || user.created_at
      }));

      setUsers(transformedUsers);
      setTotalCount(data.total_count || 0);
    } catch (err: any) {
      console.error('❌ useUsers: Erreur chargement utilisateurs:', err);
      setError(err.message || 'Erreur lors du chargement des utilisateurs');
      setUsers([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchUsers();

    if (autoRefresh) {
      const interval = setInterval(fetchUsers, refreshInterval);
      return () => clearInterval(interval);
    }
  }, [autoRefresh, refreshInterval]);

  return { users, totalCount, loading, error, refetch: fetchUsers };
};
