import { useState, useEffect } from 'react';
import { apiService } from '../services/apiService';

interface Transaction {
  id: string;
  reference: string;
  type: string;
  amount: number;
  commission_amount: number;
  status: string;
  description?: string;
  merchant_name?: string;
  location?: string;
  created_at: string;
  completed_at?: string | null;
}

interface TransactionsResponse {
  transactions: Transaction[];
  total: number;
}

export const useTransactions = (
  limit: number = 50,
  offset: number = 0,
  autoRefresh: boolean = false,
  refreshInterval: number = 30000
) => {
  const [transactions, setTransactions] = useState<Transaction[]>([]);
  const [total, setTotal] = useState(0);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchTransactions = async () => {
    try {
      setLoading(true);
      setError(null);
      
      if (!apiService.isAuthenticated()) {
        console.warn('⚠️ useTransactions: Utilisateur non authentifié');
        setTransactions([]);
        return;
      }

      console.log('🔄 useTransactions: Chargement des transactions...');
      const data = await apiService.getTransactions(limit, offset);
      console.log('✅ useTransactions: Transactions chargées:', data);
      
      setTransactions(data.transactions || []);
      setTotal(data.total_count || 0);
    } catch (err: any) {
      console.error('❌ useTransactions: Erreur chargement transactions:', err);
      setError(err.message || 'Erreur lors du chargement des transactions');
      setTransactions([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchTransactions();

    if (autoRefresh) {
      const interval = setInterval(fetchTransactions, refreshInterval);
      return () => clearInterval(interval);
    }
  }, [limit, offset, autoRefresh, refreshInterval]);

  return { transactions, total, loading, error, refetch: fetchTransactions };
};
