import { useState, useEffect } from 'react';
import { apiService } from '../services/apiService';

interface ChartDataItem {
  month: string;
  transactions: number;
  volume: number;
  nfc: number;
}

interface PieDataItem {
  label: string;
  value: number;
  color: string;
  amount: number;
  count: number;
}

interface ReportStats {
  total_transactions: number;
  total_volume: number;
  period_days: number;
  avg_daily_transactions: number;
  avg_transaction_amount: number;
}

interface ReportDataResponse {
  chart_data: ChartDataItem[];
  pie_data: PieDataItem[];
  stats: ReportStats;
  period: string;
  report_type: string;
  generated_at: string;
}

export const useReportData = (
  period: string = "30d",
  reportType: string = "financial"
) => {
  const [reportData, setReportData] = useState<ReportDataResponse | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const fetchReportData = async () => {
    try {
      setLoading(true);
      setError(null);
      
      if (!apiService.isAuthenticated()) {
        console.warn('⚠️ useReportData: Utilisateur non authentifié');
        setReportData(null);
        return;
      }

      console.log('🔄 useReportData: Chargement des données de rapport...');
      const data: ReportDataResponse = await apiService.get(
        `/admin/report-data?period=${period}&report_type=${reportType}&_t=${Date.now()}`
      );
      console.log('✅ useReportData: Données chargées:', data);
      
      setReportData(data);
    } catch (err: any) {
      console.error('❌ useReportData: Erreur chargement données:', err);
      setError(err.message || 'Erreur lors du chargement des données de rapport');
      setReportData(null);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchReportData();
  }, [period, reportType]);

  return { 
    reportData, 
    loading, 
    error, 
    refetch: fetchReportData,
    chartData: reportData?.chart_data || [],
    pieData: reportData?.pie_data || [],
    stats: reportData?.stats || {
      total_transactions: 0,
      total_volume: 0,
      period_days: 0,
      avg_daily_transactions: 0,
      avg_transaction_amount: 0
    }
  };
};
