import { useState, useEffect } from 'react';
import { apiService } from '../services/apiService';
import { Transaction } from '../types';

interface RecentTransactionsResponse {
  transactions: Transaction[];
  total_count: number;
}

export const useRecentTransactions = (
  limit: number = 10,
  autoRefresh: boolean = false, 
  refreshInterval: number = 30000
) => {
  const [transactions, setTransactions] = useState<Transaction[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchTransactions = async () => {
    try {
      setLoading(true);
      setError(null);
      
      if (!apiService.isAuthenticated()) {
        console.warn('⚠️ useRecentTransactions: Utilisateur non authentifié');
        setTransactions([]);
        return;
      }

      console.log('🔄 useRecentTransactions: Chargement des transactions récentes...');
      const data: RecentTransactionsResponse = await apiService.get(
        `/admin/recent-transactions?limit=${limit}&_t=${Date.now()}`
      );
      console.log('✅ useRecentTransactions: Transactions chargées:', data);
      
      setTransactions(data.transactions || []);
    } catch (err: any) {
      console.error('❌ useRecentTransactions: Erreur chargement transactions:', err);
      setError(err.message || 'Erreur lors du chargement des transactions');
      setTransactions([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchTransactions();

    if (autoRefresh) {
      const interval = setInterval(fetchTransactions, refreshInterval);
      return () => clearInterval(interval);
    }
  }, [limit, autoRefresh, refreshInterval]);

  return { transactions, loading, error, refetch: fetchTransactions };
};
