import { useState, useEffect } from 'react';
import { apiService } from '../services/apiService';

interface ChartData {
  label: string;
  value: number;
  change?: number;
}

interface PerformanceMetrics {
  nfc_transactions: number;
  mobile_loans: number;
  cinetpay_recharges: number;
  total_volume: number;
  total_transactions: number;
}

interface FinancialPerformanceResponse {
  chart_data: ChartData[];
  metrics: PerformanceMetrics;
  period: string;
}

export const useFinancialPerformance = (
  period: string = "12months",
  autoRefresh: boolean = false, 
  refreshInterval: number = 60000
) => {
  const [performanceData, setPerformanceData] = useState<FinancialPerformanceResponse | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchPerformanceData = async () => {
    try {
      setLoading(true);
      setError(null);
      
      if (!apiService.isAuthenticated()) {
        console.warn('⚠️ useFinancialPerformance: Utilisateur non authentifié');
        setPerformanceData(null);
        return;
      }

      console.log('🔄 useFinancialPerformance: Chargement des données de performance...');
      const data: FinancialPerformanceResponse = await apiService.get(
        `/admin/financial-performance?period=${period}&_t=${Date.now()}`
      );
      console.log('✅ useFinancialPerformance: Données chargées:', data);
      
      setPerformanceData(data);
    } catch (err: any) {
      console.error('❌ useFinancialPerformance: Erreur chargement données:', err);
      setError(err.message || 'Erreur lors du chargement des données de performance');
      setPerformanceData(null);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchPerformanceData();

    if (autoRefresh) {
      const interval = setInterval(fetchPerformanceData, refreshInterval);
      return () => clearInterval(interval);
    }
  }, [period, autoRefresh, refreshInterval]);

  return { 
    performanceData, 
    loading, 
    error, 
    refetch: fetchPerformanceData,
    chartData: performanceData?.chart_data || [],
    metrics: performanceData?.metrics || {
      nfc_transactions: 0,
      mobile_loans: 0,
      cinetpay_recharges: 0,
      total_volume: 0,
      total_transactions: 0
    }
  };
};
