import { useState, useEffect } from 'react';
import { apiService } from '../services/apiService';

interface DashboardStats {
  total_users: number;
  total_transactions: number;
  total_loans: number;
  active_loans: number;
  total_volume: number;
  admin_stock_balance: number;
  total_commissions: number;
  nfc_transactions: number;
}

export const useDashboardStats = (autoRefresh: boolean = false, refreshInterval: number = 60000) => {
  const [stats, setStats] = useState<DashboardStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchStats = async () => {
    try {
      setLoading(true);
      setError(null);
      
      if (!apiService.isAuthenticated()) {
        console.warn('⚠️ useDashboardStats: Utilisateur non authentifié');
        setStats(null);
        return;
      }

      console.log('🔄 useDashboardStats: Chargement des statistiques...');
      const data: DashboardStats = await apiService.get(`/admin/dashboard-stats?_t=${Date.now()}`);
      console.log('✅ useDashboardStats: Stats chargées:', data);
      
      setStats(data);
    } catch (err: any) {
      console.error('❌ useDashboardStats: Erreur chargement stats:', err);
      setError(err.message || 'Erreur lors du chargement des statistiques');
      setStats(null);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchStats();

    if (autoRefresh) {
      const interval = setInterval(fetchStats, refreshInterval);
      return () => clearInterval(interval);
    }
  }, [autoRefresh, refreshInterval]);

  return { stats, loading, error, refetch: fetchStats };
};
