import { useState, useEffect } from 'react';
import { apiService } from '../services/apiService';

interface AdminTransaction {
  id: string;
  user_id: string;
  user_email: string;
  user_name: string;
  transaction_type: string;
  amount: number;
  commission_amount: number;
  commission_rate: number;
  status: string;
  description?: string;
  reference: string;
  external_reference?: string;
  payment_method?: string;
  merchant_name?: string;
  location?: string;
  created_at: string;
  completed_at?: string;
}

interface TransactionStats {
  total_transactions: number;
  stats_by_status: Record<string, number>;
  stats_by_type: Record<string, number>;
  total_amount: number;
  total_commissions: number;
  recent_transactions_24h: number;
  generated_at: string;
}

export const useAdminTransactions = (
  limit: number = 100,
  offset: number = 0,
  statusFilter?: string,
  transactionType?: string,
  autoRefresh: boolean = false,
  refreshInterval: number = 30000
) => {
  const [transactions, setTransactions] = useState<AdminTransaction[]>([]);
  const [stats, setStats] = useState<TransactionStats | null>(null);
  const [total, setTotal] = useState(0);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchTransactions = async () => {
    try {
      setLoading(true);
      setError(null);
      
      if (!apiService.isAuthenticated()) {
        console.warn('⚠️ useAdminTransactions: Utilisateur non authentifié');
        setTransactions([]);
        return;
      }

      console.log('🔄 useAdminTransactions: Chargement des transactions admin...');
      
      // Récupérer les transactions et les stats en parallèle
      const [transactionsData, statsData] = await Promise.all([
        apiService.getAllTransactionsAdmin(limit, offset, statusFilter, transactionType),
        apiService.getTransactionsStatsAdmin()
      ]);
      
      console.log('✅ useAdminTransactions: Données chargées:', {
        transactions: transactionsData.transactions.length,
        stats: statsData
      });
      
      setTransactions(transactionsData.transactions || []);
      setTotal(transactionsData.total_count || 0);
      setStats(statsData);
      
    } catch (err: any) {
      console.error('❌ useAdminTransactions: Erreur chargement:', err);
      setError(err.message || 'Erreur lors du chargement des transactions');
      setTransactions([]);
      setStats(null);
    } finally {
      setLoading(false);
    }
  };

  const fetchStats = async () => {
    try {
      const statsData = await apiService.getTransactionsStatsAdmin();
      setStats(statsData);
    } catch (err: any) {
      console.error('❌ useAdminTransactions: Erreur chargement stats:', err);
    }
  };

  useEffect(() => {
    fetchTransactions();

    if (autoRefresh) {
      const interval = setInterval(fetchTransactions, refreshInterval);
      return () => clearInterval(interval);
    }
  }, [limit, offset, statusFilter, transactionType, autoRefresh, refreshInterval]);

  return { 
    transactions, 
    stats, 
    total, 
    loading, 
    error, 
    refetch: fetchTransactions,
    refetchStats: fetchStats
  };
};
