import { useState, useEffect } from 'react';
import { apiService } from '../services/apiService';

interface AdminStock {
  balance: number;
  total_credits: number;
  total_debits: number;
  total_commissions: number;
  variation_24h?: number;
  variation_percentage?: number;
  last_updated?: string;
  currency: string;
  total_users?: number;
  nfc_transactions?: number;
}

export const useAdminStock = (autoRefresh: boolean = false, refreshInterval: number = 30000) => {
  const [stock, setStock] = useState<AdminStock | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchStock = async () => {
    try {
      setLoading(true);
      setError(null);
      
      if (!apiService.isAuthenticated()) {
        console.warn('⚠️ useAdminStock: Utilisateur non authentifié');
        setStock(null);
        return;
      }

      console.log('🔄 useAdminStock: Chargement du stock admin...');
      const data = await apiService.get(`/admin/stock?_t=${Date.now()}`);
      console.log('✅ useAdminStock: Stock chargé:', data);
      setStock(data);
    } catch (err: any) {
      console.error('❌ useAdminStock: Erreur chargement stock admin:', err);
      setError(err.message || 'Erreur lors du chargement du stock');
      setStock(null);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchStock();

    // Auto-refresh si activé
    if (autoRefresh) {
      const interval = setInterval(fetchStock, refreshInterval);
      return () => clearInterval(interval);
    }
  }, [autoRefresh, refreshInterval]);

  return { stock, loading, error, refetch: fetchStock };
};
