import React, {
  createContext,
  useContext,
  useReducer,
  useEffect,
  ReactNode,
} from "react";
import { User, AuthState } from "../types";

interface AuthContextType extends AuthState {
  login: (email: string, password: string) => Promise<void>;
  logout: () => void;
  updateUser: (user: User) => void;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);
const API_BASE_URL = import.meta.env.VITE_API_URL || "${API_BASE_URL}";

type AuthAction =
  | { type: "LOGIN_SUCCESS"; payload: { user: User; token: string } }
  | { type: "LOGOUT" }
  | { type: "UPDATE_USER"; payload: User };

const authReducer = (state: AuthState, action: AuthAction): AuthState => {
  switch (action.type) {
    case "LOGIN_SUCCESS":
      return {
        user: action.payload.user,
        token: action.payload.token,
        isAuthenticated: true,
      };
    case "LOGOUT":
      return {
        user: null,
        token: null,
        isAuthenticated: false,
      };
    case "UPDATE_USER":
      return {
        ...state,
        user: action.payload,
      };
    default:
      return state;
  }
};

export const AuthProvider: React.FC<{ children: ReactNode }> = ({
  children,
}) => {
  const [state, dispatch] = useReducer(authReducer, {
    user: null,
    token: localStorage.getItem("tim_access_token"),
    isAuthenticated: false,
  });

  // Gestion de la session - Déconnexion après 10 minutes d'inactivité
  const SESSION_TIMEOUT = 10 * 60 * 1000; // 10 minutes en millisecondes
  let inactivityTimer: NodeJS.Timeout | null = null;

  // Fonction pour réinitialiser le timer d'inactivité
  const resetInactivityTimer = () => {
    if (inactivityTimer) {
      clearTimeout(inactivityTimer);
    }

    // Seulement si l'utilisateur est connecté
    if (state.isAuthenticated) {
      inactivityTimer = setTimeout(() => {
        console.log("⏰ Session expirée après 10 minutes d'inactivité");
        logout();
        alert(
          "Votre session a expiré après 10 minutes d'inactivité. Veuillez vous reconnecter."
        );
      }, SESSION_TIMEOUT);

      // Mettre à jour le timestamp de dernière activité
      localStorage.setItem("lastActivity", Date.now().toString());
    }
  };

  // Vérifier la session au chargement
  useEffect(() => {
    const token = localStorage.getItem("tim_access_token");
    const userData = localStorage.getItem("user");
    const lastActivity = localStorage.getItem("lastActivity");

    if (token && userData) {
      try {
        const user = JSON.parse(userData);

        // Vérifier si la session a expiré
        if (lastActivity) {
          const timeSinceLastActivity = Date.now() - parseInt(lastActivity);
          if (timeSinceLastActivity > SESSION_TIMEOUT) {
            console.log("⏰ Session expirée détectée au chargement");
            localStorage.removeItem("tim_access_token");
            localStorage.removeItem("tim_refresh_token");
            localStorage.removeItem("user");
            localStorage.removeItem("lastActivity");
            alert("Votre session a expiré. Veuillez vous reconnecter.");
            return;
          }
        }

        dispatch({ type: "LOGIN_SUCCESS", payload: { user, token } });
      } catch (error) {
        localStorage.removeItem("tim_access_token");
        localStorage.removeItem("tim_refresh_token");
        localStorage.removeItem("user");
        localStorage.removeItem("lastActivity");
      }
    }
  }, []);

  // Écouter les événements d'activité utilisateur
  useEffect(() => {
    if (!state.isAuthenticated) return;

    const events = ["mousedown", "keydown", "scroll", "touchstart", "click"];

    const handleActivity = () => {
      resetInactivityTimer();
    };

    // Ajouter les écouteurs d'événements
    events.forEach((event) => {
      window.addEventListener(event, handleActivity);
    });

    // Initialiser le timer
    resetInactivityTimer();

    // Nettoyer les écouteurs
    return () => {
      events.forEach((event) => {
        window.removeEventListener(event, handleActivity);
      });
      if (inactivityTimer) {
        clearTimeout(inactivityTimer);
      }
    };
  }, [state.isAuthenticated]);

  const login = async (emailOrPhone: string, password: string) => {
    try {
      // Payload à envoyer - l'API attend un numéro de téléphone
      const payload = {
        email: emailOrPhone, // Peut être email ou téléphone
        hashed_password: password, // Malgré le nom, envoyer le mot de passe en clair
      };

      console.log("🔐 Tentative de connexion avec:");
      console.log("   Phone/Email:", emailOrPhone);
      console.log("   Password length:", password.length);
      console.log("   Payload:", JSON.stringify(payload, null, 2));

      // Étape 1: Login pour obtenir le token
      const loginResponse = await fetch(
        `${API_BASE_URL}/api/auth/admin/login`,
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify(payload),
        }
      );

      console.log("   Response status:", loginResponse.status);

      if (!loginResponse.ok) {
        const errorData = await loginResponse
          .json()
          .catch(() => ({ detail: "Erreur inconnue" }));
        console.error("❌ Erreur serveur:", errorData);
        console.error("   Status:", loginResponse.status);
        console.error("   Detail:", errorData.detail);
        throw new Error(errorData.detail || "Identifiants invalides");
      }

      const loginData = await loginResponse.json();

      console.log("✅ Réponse login:", loginData);

      // Le backend retourne { success, data: { access_token, refresh_token, token_type } }
      const token = loginData.data?.access_token || loginData.access_token;
      const refreshToken =
        loginData.data?.refresh_token || loginData.refresh_token;

      console.log(
        "🔑 Token extrait:",
        token ? `${token.substring(0, 20)}...` : "TOKEN MANQUANT!"
      );

      if (!token) {
        throw new Error("Token non reçu du serveur");
      }

      // Étape 2: Récupérer les informations de l'utilisateur avec le token
      const meResponse = await fetch(`${API_BASE_URL}/api/auth/admin/profile`, {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "application/json",
        },
      });

      console.log("📥 Réponse /auth/me status:", meResponse.status);

      if (!meResponse.ok) {
        const errorText = await meResponse.text();
        console.error("❌ Erreur /auth/me:", errorText);
        throw new Error(
          `Impossible de récupérer les informations utilisateur: ${meResponse.status}`
        );
      }

      const meData = await meResponse.json();

      // Transformer les données en format User
      const userData: User = {
        id: meData.id,
        email: meData.email,
        full_name: meData.full_name || "Utilisateur",
        firstName: meData.full_name?.split(" ")[0] || "User",
        lastName: meData.full_name?.split(" ").slice(1).join(" ") || "",
        phone: meData.phone || "",
        role: meData.is_admin ? "admin" : "user",
        timType: meData.tim_account_type || "TIM_MINI",
        isVerified: meData.wallet?.is_active || true,
        isBlocked: false,
        createdAt: meData.created_at || new Date().toISOString(),
        updatedAt: new Date().toISOString(),
      };

      localStorage.setItem("tim_access_token", token);
      if (refreshToken) {
        localStorage.setItem("tim_refresh_token", refreshToken);
      }
      localStorage.setItem("user", JSON.stringify(userData));
      localStorage.setItem("lastActivity", Date.now().toString());

      dispatch({ type: "LOGIN_SUCCESS", payload: { user: userData, token } });
    } catch (error) {
      console.error("Login error:", error);
      throw new Error("Connexion échouée. Vérifiez vos identifiants.");
    }
  };

  const logout = () => {
    // Nettoyer le timer d'inactivité
    if (inactivityTimer) {
      clearTimeout(inactivityTimer);
    }

    localStorage.removeItem("tim_access_token");
    localStorage.removeItem("tim_refresh_token");
    localStorage.removeItem("user");
    localStorage.removeItem("lastActivity");
    dispatch({ type: "LOGOUT" });
  };

  const updateUser = (user: User) => {
    localStorage.setItem("user", JSON.stringify(user));
    dispatch({ type: "UPDATE_USER", payload: user });
  };

  return (
    <AuthContext.Provider value={{ ...state, login, logout, updateUser }}>
      {children}
    </AuthContext.Provider>
  );
};

export const useAuth = () => {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error("useAuth must be used within an AuthProvider");
  }
  return context;
};
