/**
 * Configuration API pour TIM CASH
 * Backend SQLAlchemy + SQLite
 */

// Configuration de base
const getBaseUrl = () => {
  return "http://localhost:8001";
};

export const API_CONFIG = {
  BASE_URL: getBaseUrl(),
  TIMEOUT: 10000,
  HEADERS: {
    "Content-Type": "application/json",
  },
};

// Endpoints API
export const API_ENDPOINTS = {
  // Authentification
  AUTH: {
    LOGIN: "/auth/login",
    REGISTER: "/auth/register",
    ME: "/auth/me",
    REFRESH: "/auth/refresh",
    LOGOUT: "/auth/logout",
  },

  // Portefeuille
  WALLET: {
    BALANCE: "/wallet/balance",
    RELOAD: "/wallet/reload",
    TRANSFER: "/wallet/transfer",
    HISTORY: "/wallet/history",
  },

  // Transactions
  TRANSACTIONS: {
    LIST: "/transactions",
    CREATE: "/wallet/transaction",
  },

  // Prêts
  LOANS: {
    CREATE: "/loans/create",
    ACTIVE: "/loans/active",
    HISTORY: "/loans/history",
  },

  // Administration
  ADMIN: {
    STOCK: "/admin/stock",
    USERS: "/admin/users",
    DASHBOARD_STATS: "/admin/dashboard-stats",
    COMMISSION_RATES: "/api/admin/commission-rates",
    COMMISSION_ACTIVATE_MOBILE: "/api/admin/commission-rates/activate-mobile",
    COMMISSION_MOBILE_STATUS: "/api/admin/commission-rates/mobile-status",
    COMMISSION_ACTIVATE_ALL: "/api/admin/commission-rates/activate-all",
    COMMISSION_RECHARGE_BREAKDOWN:
      "/api/admin/commission-rates/recharge-breakdown",
    DEVICE_ANALYTICS: "/admin/device-analytics",
    TRANSACTIONS: "/admin/transactions",
    TRANSACTIONS_STATS: "/admin/transactions/stats",
    NFC_CARDS: "/admin/nfc-cards",
    NFC_CARDS_CREATE: "/admin/nfc-cards/create",
    NFC_CARD_STATUS: (cardId: string) => `/admin/nfc-cards/${cardId}/status`,
    NFC_CARD_DETAILS: (cardId: string) => `/admin/nfc-cards/${cardId}`,
    NFC_CARD_DELETE: (cardId: string) => `/admin/nfc-cards/${cardId}`,
  },

  // Géographie
  GEOGRAPHY: {
    COUNTRIES: "/countries",
    CITIES: "/cities",
  },

  // CinetPay
  CINETPAY: {
    CALLBACK: "/wallet-paiements/callback",
  },
};

// Types d'utilisateurs TIM CASH
export const TIM_ACCOUNT_TYPES = {
  TIM_MINI: "TIM_MINI",
  TIM_MAXI: "TIM_MAXI",
  TIM_BUSINESS: "TIM_BUSINESS",
} as const;

// Limites par type de compte
export const ACCOUNT_LIMITS = {
  [TIM_ACCOUNT_TYPES.TIM_MINI]: {
    maxBalance: 2000000, // 2M FCFA
    maxLoan: 0, // Pas de prêt
    dailyLimit: 50000, // 50K FCFA/jour
  },
  [TIM_ACCOUNT_TYPES.TIM_MAXI]: {
    maxBalance: 2000000, // 2M FCFA
    maxLoan: 10000, // 10K FCFA
    dailyLimit: 500000, // 500K FCFA/jour
  },
  [TIM_ACCOUNT_TYPES.TIM_BUSINESS]: {
    maxBalance: null, // Illimité
    maxLoan: 50000, // 50K FCFA
    dailyLimit: null, // Illimité
  },
};

// Taux de commission
export const COMMISSION_RATES = {
  CINETPAY_RECHARGE: 0.005, // 0.5%
  TRANSFER: 0.01, // 1%
  WITHDRAWAL: 0.02, // 2%
  LOAN_INTEREST: 0.06, // 6%
  LOAN_PENALTY: 0.15, // 15%
};

// Statuts de transaction
export const TRANSACTION_STATUS = {
  PENDING: "pending",
  COMPLETED: "completed",
  FAILED: "failed",
  CANCELLED: "cancelled",
} as const;

// Types de transaction
export const TRANSACTION_TYPES = {
  RECHARGE_CINETPAY: "recharge_cinetpay",
  NFC_DEPOSIT: "nfc_deposit",
  NFC_WITHDRAWAL: "nfc_withdrawal",
  MOBILE_TRANSFER: "mobile_transfer",
  PRET_MOBILE: "pret_mobile",
  REMBOURSEMENT_AUTO: "remboursement_auto",
  TRANSFER_IN: "transfer_in",
  TRANSFER_OUT: "transfer_out",
} as const;

// Statuts de prêt
export const LOAN_STATUS = {
  PENDING: "pending",
  APPROVED: "approved",
  ACTIVE: "active",
  COMPLETED: "completed",
  OVERDUE: "overdue",
  REJECTED: "rejected",
} as const;

// Configuration des couleurs par type de compte
export const ACCOUNT_COLORS = {
  [TIM_ACCOUNT_TYPES.TIM_MINI]: {
    primary: "bg-blue-500",
    light: "bg-blue-100",
    text: "text-blue-700",
    border: "border-blue-200",
  },
  [TIM_ACCOUNT_TYPES.TIM_MAXI]: {
    primary: "bg-purple-500",
    light: "bg-purple-100",
    text: "text-purple-700",
    border: "border-purple-200",
  },
  [TIM_ACCOUNT_TYPES.TIM_BUSINESS]: {
    primary: "bg-orange-500",
    light: "bg-orange-100",
    text: "text-orange-700",
    border: "border-orange-200",
  },
};

// Messages d'erreur
export const ERROR_MESSAGES = {
  NETWORK_ERROR: "Erreur de connexion. Vérifiez votre connexion internet.",
  UNAUTHORIZED: "Session expirée. Veuillez vous reconnecter.",
  FORBIDDEN: "Accès non autorisé.",
  NOT_FOUND: "Ressource non trouvée.",
  SERVER_ERROR: "Erreur serveur. Veuillez réessayer plus tard.",
  VALIDATION_ERROR: "Données invalides. Vérifiez vos informations.",
  INSUFFICIENT_BALANCE: "Solde insuffisant.",
  ACCOUNT_LIMIT_EXCEEDED: "Limite de compte dépassée.",
  LOAN_ALREADY_ACTIVE: "Vous avez déjà un prêt actif.",
};

// Configuration des comptes de test
export const TEST_ACCOUNTS = {
  ADMIN: {
    email: "admin@timcash.ci",
    password: "admin123",
    type: TIM_ACCOUNT_TYPES.TIM_BUSINESS,
    isAdmin: true,
  },
  MARIE: {
    email: "marie.kouassi@timcash.ci",
    password: "password123",
    type: TIM_ACCOUNT_TYPES.TIM_MAXI,
    isAdmin: false,
  },
  JEAN: {
    email: "jean.baptiste@timcash.ci",
    password: "password123",
    type: TIM_ACCOUNT_TYPES.TIM_MINI,
    isAdmin: false,
  },
  FATOU: {
    email: "fatou.diallo@timcash.ci",
    password: "password123",
    type: TIM_ACCOUNT_TYPES.TIM_BUSINESS,
    isAdmin: false,
  },
};

export default API_CONFIG;
