import React, { useState, useEffect } from 'react';
import { Clock, AlertTriangle } from 'lucide-react';

interface SessionTimerProps {
  showWarning?: boolean;
}

const SessionTimer: React.FC<SessionTimerProps> = ({ showWarning = true }) => {
  const [timeRemaining, setTimeRemaining] = useState<number>(0);
  const [showAlert, setShowAlert] = useState(false);

  const SESSION_TIMEOUT = 10 * 60 * 1000; // 10 minutes
  const WARNING_THRESHOLD = 2 * 60 * 1000; // Avertir 2 minutes avant expiration

  useEffect(() => {
    const updateTimer = () => {
      const lastActivity = localStorage.getItem('lastActivity');
      if (lastActivity) {
        const elapsed = Date.now() - parseInt(lastActivity);
        const remaining = SESSION_TIMEOUT - elapsed;
        
        if (remaining > 0) {
          setTimeRemaining(remaining);
          
          // Afficher l'alerte si moins de 2 minutes restantes
          if (remaining < WARNING_THRESHOLD && showWarning) {
            setShowAlert(true);
          } else {
            setShowAlert(false);
          }
        } else {
          setTimeRemaining(0);
        }
      }
    };

    // Mettre à jour toutes les secondes
    const interval = setInterval(updateTimer, 1000);
    updateTimer(); // Appel initial

    return () => clearInterval(interval);
  }, [showWarning]);

  const formatTime = (ms: number): string => {
    const minutes = Math.floor(ms / 60000);
    const seconds = Math.floor((ms % 60000) / 1000);
    return `${minutes}:${seconds.toString().padStart(2, '0')}`;
  };

  const extendSession = () => {
    localStorage.setItem('lastActivity', Date.now().toString());
    setShowAlert(false);
  };

  if (timeRemaining === 0) return null;

  return (
    <>
      {/* Indicateur de temps restant */}
      <div className="flex items-center space-x-2 text-xs">
        <Clock className={`w-3 h-3 ${showAlert ? 'text-red-600' : 'text-amber-600'}`} />
        <span className={`font-medium ${showAlert ? 'text-red-600' : 'text-amber-600'}`}>
          Session: {formatTime(timeRemaining)}
        </span>
      </div>

      {/* Alerte d'expiration imminente */}
      {showAlert && (
        <div className="fixed bottom-4 right-4 z-50 animate-fadeIn">
          <div className="bg-gradient-to-r from-red-500 to-orange-500 text-white rounded-lg shadow-2xl p-4 max-w-sm">
            <div className="flex items-start space-x-3">
              <AlertTriangle className="w-5 h-5 flex-shrink-0 mt-0.5" />
              <div className="flex-1">
                <h4 className="font-bold text-sm mb-1">Session bientôt expirée</h4>
                <p className="text-xs mb-3">
                  Votre session expirera dans {formatTime(timeRemaining)}. 
                  Cliquez pour prolonger.
                </p>
                <div className="flex space-x-2">
                  <button
                    onClick={extendSession}
                    className="bg-white text-red-600 px-3 py-1 rounded text-xs font-medium hover:bg-red-50 transition-colors"
                  >
                    Prolonger la session
                  </button>
                  <button
                    onClick={() => setShowAlert(false)}
                    className="bg-white/20 text-white px-3 py-1 rounded text-xs font-medium hover:bg-white/30 transition-colors"
                  >
                    Fermer
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
    </>
  );
};

export default SessionTimer;
