import React, { useState } from 'react';
import { NavLink, useLocation } from 'react-router-dom';
import { 
  Home, 
  Users, 
  Wallet, 
  CreditCard, 
  BarChart3, 
  Settings,
  DollarSign,
  FileText,
  Bell,
  ArrowLeftRight,
  PlusCircle,
  History,
  Send,
  Shield,
  Building2,
  ChevronDown,
  ChevronRight,
  LogOut,
  User,
  Menu,
  X,
  Activity,
  Clock,
  AlertTriangle,
  Percent,
  Globe,
  Image,
  Banknote,
  HandCoins,
  TrendingUp
} from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';
import { timClasses } from '../../styles/colors';
import { useAdminStock } from '../../hooks/useAdminStock';

interface MenuItem {
  name: string;
  href?: string;
  icon: React.ComponentType<any>;
  badge?: string | number;
  children?: MenuItem[];
  roles: string[];
}

const Sidebar: React.FC = () => {
  const { user, logout } = useAuth();
  const location = useLocation();
  const [expandedItems, setExpandedItems] = useState<string[]>(['Portefeuille']);
  const [isCollapsed, setIsCollapsed] = useState(false);
  const [isMobileOpen, setIsMobileOpen] = useState(false);

  const isAdmin = user?.role === 'admin';
  
  // Récupérer le solde du stock admin avec auto-refresh toutes les 30 secondes
  const { stock } = useAdminStock(isAdmin, 30000);

  const menuItems: MenuItem[] = [
    {
      name: 'Dashboard Admin',
      icon: Home,
      href: '/dashboard',
      roles: ['admin', 'manager', 'user']
    },
    {
      name: isAdmin ? 'Gestion Portefeuilles' : 'Mon TIM Wallet',
      icon: Wallet,
      roles: ['admin', 'manager', 'user'],
      children: isAdmin ? [
        { name: 'Compte Stock Admin', icon: Shield, href: '/stock-account', roles: ['admin'] },
        { name: 'Portefeuilles Mobiles', icon: Wallet, href: '/wallets', roles: ['admin', 'manager'] },
        { name: 'Recharges CinetPay', icon: PlusCircle, href: '/recharges', roles: ['admin', 'manager'] },
        { name: 'Historique Transferts', icon: ArrowLeftRight, href: '/transfers', roles: ['admin', 'manager'] },
        { name: 'Flux Financiers', icon: ArrowLeftRight, href: '/flows', roles: ['admin'] }
      ] : [
        { name: 'Mon TIM MAXI', icon: Wallet, href: '/wallet', roles: ['user'] },
        { name: 'Recharge CinetPay', icon: PlusCircle, href: '/wallet/reload', roles: ['user'] },
        { name: 'Mes Transferts', icon: ArrowLeftRight, href: '/transfers', roles: ['user'] },
        { name: 'Paiement NFC', icon: Activity, href: '/nfc-payment', roles: ['user'] },
        { name: 'Transfert Mobile', icon: Send, href: '/wallet/transfer', roles: ['user'] }
      ]
    },
    {
      name: isAdmin ? 'Supervision Prêts' : 'Mes Prêts TIM',
      icon: CreditCard,
      roles: ['admin', 'manager', 'user'],
      children: isAdmin ? [
        { name: 'Prêts en Cours', icon: CreditCard, href: '/loans/active', roles: ['admin', 'manager'] },
        { name: 'Demandes Pending', icon: Clock, href: '/loans/pending', roles: ['admin', 'manager'] },
        { name: 'Remboursements Auto', icon: ArrowLeftRight, href: '/loans/repayments', roles: ['admin'] },
        { name: 'Prêts en Retard', icon: AlertTriangle, href: '/loans/overdue', roles: ['admin', 'manager'] }
      ] : [
        { name: 'Mes Prêts Actifs', icon: CreditCard, href: '/loans', roles: ['user'] },
        { name: 'Demander Prêt', icon: PlusCircle, href: '/loans/apply', roles: ['user'] },
        { name: 'Historique Prêts', icon: History, href: '/loans/history', roles: ['user'] }
      ]
    },
    {
      name: isAdmin ? 'Transactions Mobiles' : 'Mes Transactions',
      icon: Activity,
      href: '/transactions',
      roles: ['admin', 'manager', 'user']
    },
    {
      name: 'Administration',
      icon: Building2,
      roles: ['admin', 'manager'],
      children: [
        { name: 'Utilisateurs Mobiles', icon: Users, href: '/users', roles: ['admin', 'manager'] },
        { name: 'Recharges TIM BUSINESS', icon: TrendingUp, href: '/tim-business-recharges', roles: ['admin', 'manager'] },
        { name: 'Gestion Commissions', icon: Percent, href: '/admin/commissions', roles: ['admin'] },
        { name: 'Cartes NFC', icon: CreditCard, href: '/admin/cards', roles: ['admin', 'manager'] },
        { name: 'Cartes Virtuelles', icon: CreditCard, href: '/admin/virtual-cards', roles: ['admin', 'manager'] },
        { name: 'Pays et Villes', icon: Globe, href: '/countries', roles: ['admin'] },
        { name: 'Publicités', icon: Image, href: '/advertising', roles: ['admin'] },
        { name: 'Compte Stock', icon: Banknote, href: '/stock-account', roles: ['admin'] },
        { name: 'Gestion Prêts', icon: HandCoins, href: '/loans-management', roles: ['admin'] },
        { name: 'Analytics NFC', icon: BarChart3, href: '/analytics/nfc', roles: ['admin'] },
        { name: 'Rapports CinetPay', icon: FileText, href: '/reports/cinetpay', roles: ['admin', 'manager'] },
        { name: 'Monitoring App', icon: Activity, href: '/monitoring', roles: ['admin'] }
      ]
    },
    {
      name: 'Notifications',
      icon: Bell,
      href: '/notifications',
      badge: '3',
      roles: ['admin', 'manager', 'user']
    },
    {
      name: 'Paramètres',
      icon: Settings,
      href: '/settings',
      roles: ['admin', 'manager', 'user']
    }
  ];

  const toggleExpanded = (itemName: string) => {
    setExpandedItems(prev => 
      prev.includes(itemName) 
        ? prev.filter(name => name !== itemName)
        : [...prev, itemName]
    );
  };

  const isActive = (href: string) => {
    return location.pathname === href || location.pathname.startsWith(href + '/');
  };

  const hasAccess = (roles: string[]) => {
    return roles.includes(user?.role || 'user');
  };

  const renderMenuItem = (item: MenuItem, level = 0) => {
    if (!hasAccess(item.roles)) {
      return null;
    }

    const hasChildren = item.children && item.children.length > 0;
    const isExpanded = expandedItems.includes(item.name);
    const isItemActive = item.href ? isActive(item.href) : false;

    if (hasChildren) {
      return (
        <div key={item.name} className="mb-1">
          <button
            onClick={() => toggleExpanded(item.name)}
            className={`w-full flex items-center justify-between px-4 py-3 text-left rounded-xl transition-all duration-200 group ${
              level === 0 
                ? 'text-amber-200 hover:bg-amber-800/30 hover:text-amber-100'
                : 'text-amber-300 hover:bg-amber-800/20'
            }`}
          >
            <div className="flex items-center space-x-2">
              <item.icon className={`h-4 w-4 ${
                level === 0 ? 'text-amber-400 group-hover:text-yellow-400' : 'text-amber-500'
              }`} />
              {!isCollapsed && (
                <span className="font-medium text-sm">{item.name}</span>
              )}
            </div>
            {!isCollapsed && (
              <div className="flex items-center space-x-2">
                {item.badge && (
                  <span className="bg-gradient-to-r from-amber-500 to-orange-600 text-white text-xs px-2 py-1 rounded-full font-medium shadow-lg">
                    {item.badge}
                  </span>
                )}
                {isExpanded ? (
                  <ChevronDown className="h-4 w-4 text-amber-400" />
                ) : (
                  <ChevronRight className="h-4 w-4 text-amber-400" />
                )}
              </div>
            )}
          </button>
          {isExpanded && !isCollapsed && item.children && (
            <div className="ml-4 mt-1 space-y-1">
              {item.children.map(child => renderMenuItem(child, level + 1))}
            </div>
          )}
        </div>
      );
    }

    return (
      <NavLink
        key={item.name}
        to={item.href!}
        className={({ isActive }) => `
          flex items-center justify-between px-4 py-3 rounded-xl transition-all duration-200 group mb-1
          ${
            isActive || isItemActive
              ? 'bg-gradient-to-r from-amber-600 to-orange-700 text-white shadow-lg transform scale-[1.02] shadow-amber-500/30'
              : level === 0
              ? 'text-amber-200 hover:bg-amber-800/30 hover:text-amber-100'
              : 'text-amber-300 hover:bg-amber-800/20 hover:text-amber-200'
          }
        `}
      >
        <div className="flex items-center space-x-2">
          <item.icon className={`h-4 w-4 ${
            isItemActive
              ? 'text-white'
              : level === 0
              ? 'text-amber-400 group-hover:text-yellow-400'
              : 'text-amber-500 group-hover:text-amber-300'
          }`} />
          {!isCollapsed && (
            <span className="font-medium text-sm">{item.name}</span>
          )}
        </div>
        {!isCollapsed && item.badge && (
          <span className={`text-xs px-2 py-1 rounded-full font-medium ${
            isItemActive
              ? 'bg-yellow-500 text-amber-900'
              : 'bg-gradient-to-r from-amber-500 to-orange-600 text-white shadow-lg'
          }`}>
            {item.badge}
          </span>
        )}
      </NavLink>
    );
  };

  return (
    <>
      {/* Mobile Menu Button */}
      <button
        onClick={() => setIsMobileOpen(!isMobileOpen)}
        className="lg:hidden fixed top-4 left-4 z-50 p-2 bg-slate-900 text-white rounded-lg shadow-lg"
      >
        {isMobileOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}
      </button>

      {/* Mobile Overlay */}
      {isMobileOpen && (
        <div 
          className="lg:hidden fixed inset-0 z-40 bg-black bg-opacity-50" 
          onClick={() => setIsMobileOpen(false)}
        />
      )}

      {/* Sidebar */}
      <aside className={`
        fixed inset-y-0 left-0 z-50 bg-gradient-to-b from-amber-900 via-orange-900 to-red-900 text-white shadow-2xl transition-all duration-300
        ${isCollapsed ? 'w-16' : 'w-64'}
        ${isMobileOpen ? 'translate-x-0' : '-translate-x-full lg:translate-x-0'}
      `}>
        {/* Header avec Logo TIM CASH */}
        <div className="flex items-center justify-between p-4 border-b border-amber-700/50">
          {!isCollapsed && (
            <div className="flex items-center space-x-3">
              {/* Logo TIM CASH */}
              <div className="relative">
                <img 
                  src="/assets/logoTim-2.png" 
                  alt="TIM CASH Logo" 
                  className="w-10 h-10 object-contain"
                />
                <div className="absolute inset-0 bg-gradient-to-br from-amber-400/20 to-orange-500/20 rounded-lg"></div>
              </div>
              <div>
                <h1 className="text-lg font-bold bg-gradient-to-r from-amber-300 via-yellow-400 to-orange-400 bg-clip-text text-transparent">
                  TIM CASH
                </h1>
                <p className="text-amber-200 text-xs font-medium">Plateforme Mobile</p>
              </div>
            </div>
          )}
          {isCollapsed && (
            <div className="flex justify-center w-full">
              <img 
                src="/assets/logo_4.png" 
                alt="TIM CASH" 
                className="w-8 h-8 object-contain"
              />
            </div>
          )}
          <button
            onClick={() => setIsCollapsed(!isCollapsed)}
            className="hidden lg:block p-1.5 rounded-lg hover:bg-amber-800/50 transition-colors"
          >
            <Menu className="h-4 w-4 text-amber-300" />
          </button>
        </div>

        {/* User Profile */}
        <div className="p-4 border-b border-amber-700/50">
          <div className="flex items-center space-x-2">
            <div className="w-10 h-10 bg-gradient-to-br from-amber-500 via-orange-500 to-red-500 rounded-lg flex items-center justify-center shadow-lg">
              <User className="h-5 w-5 text-white" />
            </div>
            {!isCollapsed && (
              <div className="flex-1 min-w-0">
                <p className="text-xs font-semibold text-amber-100 truncate">
                  {user?.firstName} {user?.lastName}
                </p>
                <p className="text-xs text-amber-300 truncate">{user?.email}</p>
                <span className={`inline-block px-2 py-0.5 text-xs rounded-full mt-1 font-medium ${
                  user?.role === 'admin' 
                    ? 'bg-gradient-to-r from-amber-500 to-orange-600 text-white'
                    : user?.role === 'manager'
                    ? 'bg-gradient-to-r from-blue-500 to-cyan-500 text-white'
                    : 'bg-gradient-to-r from-green-500 to-emerald-500 text-white'
                }`}>
                  {user?.role === 'admin' ? 'Administrateur' : 
                   user?.role === 'manager' ? 'Gestionnaire' : 'Utilisateur'}
                </span>
              </div>
            )}
          </div>
        </div>

        {/* Navigation Menu avec scroll */}
        <nav className="flex-1 px-3 py-4 space-y-1.5 overflow-y-auto custom-scrollbar" style={{ maxHeight: 'calc(100vh - 400px)' }}>
          {menuItems.map(item => renderMenuItem(item))}
        </nav>

        {/* Quick Stats (for non-collapsed) */}
        {!isCollapsed && (
          <div className="p-3 mx-3 mb-3 bg-gradient-to-r from-amber-800/50 to-orange-800/50 rounded-lg border border-amber-600/30 backdrop-blur-sm">
            <h3 className="text-xs font-semibold text-amber-100 mb-2">
              {isAdmin ? 'Admin TIM CASH' : 'Mon TIM CASH'}
            </h3>
            <div className="space-y-1.5">
              <div className="flex justify-between items-center">
                <span className="text-xs text-amber-300">
                  {isAdmin ? 'Stock Admin' : 'Portefeuille'}
                </span>
                <span className="text-xs font-bold text-yellow-400">
                  {isAdmin 
                    ? `${(stock?.balance || 0).toLocaleString('fr-FR')} FCFA` 
                    : '0 FCFA'}
                </span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-xs text-amber-300">
                  {isAdmin ? 'Utilisateurs' : 'Prêts'}
                </span>
                <span className="text-xs font-bold text-orange-400">
                  {isAdmin ? (stock?.total_users || 0) : '0'}
                </span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-xs text-amber-300">
                  {isAdmin ? 'Paiements NFC' : 'NFC ce mois'}
                </span>
                <span className="text-xs font-bold text-amber-400">
                  {isAdmin ? (stock?.nfc_transactions || 0) : '0'}
                </span>
              </div>
            </div>
          </div>
        )}

        {/* Logout Button */}
        <div className="p-4 border-t border-amber-700/50">
          <button
            onClick={logout}
            className="w-full flex items-center space-x-3 px-4 py-3 text-red-300 hover:bg-red-900/30 hover:text-red-200 rounded-xl transition-all duration-200 group"
          >
            <LogOut className="h-5 w-5 group-hover:scale-110 transition-transform" />
            {!isCollapsed && <span className="font-medium">Déconnexion</span>}
          </button>
        </div>
      </aside>

    </>
  );
};

export default Sidebar;