import React, { useState, useEffect } from 'react';
import { 
  CreditCard, 
  LogOut, 
  Settings, 
  User, 
  Wallet,
  HelpCircle,
  TrendingUp,
  Activity,
  Filter,
  MoreVertical,
  Maximize2,
  Search
} from 'lucide-react';
import { useAuth } from '../../contexts/AuthContext';

const Header: React.FC = () => {
  const { user, logout } = useAuth();
  const [showUserMenu, setShowUserMenu] = useState(false);
  // États pour la gestion des notifications
  const [showNotifications, setShowNotifications] = useState(false);
  const [notifications, setNotifications] = useState<Array<{
    id: string;
    title: string;
    message: string;
    time: string;
    type: 'success' | 'info' | 'warning' | 'error';
    icon: any;
  }>>([]);
  const [isNotificationsLoading, setIsNotificationsLoading] = useState(true);

  // États pour les actions rapides
  const [showQuickActions, setShowQuickActions] = useState(false);
  
  // États pour la recherche
  const [searchQuery, setSearchQuery] = useState('');
  
  // États pour le solde du portefeuille
  const [walletBalance, setWalletBalance] = useState<{ current: number; pending: number } | null>(null);
  const [error, setError] = useState<string | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  
  // État pour l'heure actuelle
  const [currentTime, setCurrentTime] = useState(new Date());

  useEffect(() => {
    const timer = setInterval(() => setCurrentTime(new Date()), 1000);
    return () => clearInterval(timer);
  }, []);

  useEffect(() => {
    const fetchWalletBalance = async () => {
      try {
        setIsLoading(true);
        // Remplacez par votre appel API réel
        // const response = await api.get('/api/wallet/balance');
        // setWalletBalance(response.data);
        
        // Simulation de chargement
        setTimeout(() => {
          setWalletBalance({
            current: 2450750, // Remplacé par la valeur de l'API
            pending: 12500
          });
          setIsLoading(false);
        }, 500);
      } catch (err) {
        console.error('Erreur lors du chargement du solde:', err);
        setError('Impossible de charger le solde');
        setIsLoading(false);
      }
    };

    fetchWalletBalance();
  }, []);

  useEffect(() => {
    const fetchNotifications = async () => {
      try {
        setIsNotificationsLoading(true);
        // Remplacer par un appel API réel
        // const response = await api.get('/api/notifications');
        // setNotifications(response.data);
        
        // Simulation de chargement
        setTimeout(() => {
          setNotifications([]); // Tableau vide par défaut
          setIsNotificationsLoading(false);
        }, 500);
      } catch (err) {
        console.error('Erreur lors du chargement des notifications:', err);
        setNotifications([]);
        setIsNotificationsLoading(false);
      }
    };

    fetchNotifications();
  }, []);

  // Fonction pour basculer en mode plein écran
  const toggleFullscreen = async () => {
    try {
      if (!document.fullscreenElement) {
        await document.documentElement.requestFullscreen();
      } else if (document.exitFullscreen) {
        await document.exitFullscreen();
      }
    } catch (err) {
      console.error('Erreur lors du changement de mode plein écran:', err);
    }
  };

  const handleLogout = () => {
    logout();
    setShowUserMenu(false);
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    if (!searchQuery.trim()) return;
    
    // Implémenter la logique de recherche
    // Exemple: router.push(`/search?q=${encodeURIComponent(searchQuery)}`);
    console.log('Recherche:', searchQuery);
  };

  /**
   * Formate une date en français
   * @param date - Date à formater
   * @returns Date formatée (ex: "lundi 12 décembre")
   */
  const formatDate = (date: Date): string => {
    return new Intl.DateTimeFormat('fr-FR', { 
      weekday: 'long', 
      day: 'numeric', 
      month: 'long' 
    }).format(date);
  };

  /**
   * Formate une heure au format 24h
   * @param date - Date contenant l'heure à formater
   * @returns Heure formatée (ex: "14:30")
   */
  const formatTime = (date: Date): string => {
    return new Intl.DateTimeFormat('fr-FR', { 
      hour: '2-digit', 
      minute: '2-digit',
      hour12: false
    }).format(date);
  };

  // Rendu du composant
  return (
    <header className="bg-white shadow-sm border-b border-gray-200 sticky top-0 z-40">
      <div className="flex items-center justify-between px-4 py-3">
        {/* Left Section - Search & Quick Info */}
        <div className="flex items-center space-x-6">
          {/* Enhanced Search */}
          <form onSubmit={handleSearch} className="relative">
            <Search className="w-5 h-5 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
            <input
              type="text"
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              placeholder="Rechercher..."
              className="pl-10 pr-12 py-3 w-96 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent focus:bg-white transition-all duration-200"
            />
            <button
              type="button"
              className="absolute right-3 top-1/2 transform -translate-y-1/2 p-1 hover:bg-gray-100 rounded-md transition-colors"
            >
              <Filter className="w-4 h-4 text-gray-400" />
            </button>
          </form>

          {/* Quick Stats */}
          <div className="hidden xl:flex items-center space-x-4">
            <div className="bg-gradient-to-r from-blue-50 to-blue-100 px-4 py-2 rounded-lg border border-blue-200">
              <div className="flex items-center space-x-2 min-w-[120px]">
                <Wallet className="w-4 h-4 text-blue-600" />
                <div>
                  <p className="text-xs text-blue-600 font-medium">Solde</p>
                  {isLoading ? (
                    <div className="h-4 w-20 bg-gray-200 rounded animate-pulse"></div>
                  ) : error ? (
                    <p className="text-xs text-red-500">Erreur</p>
                  ) : (
                    <p className="text-sm font-bold text-blue-700">
                      {walletBalance?.current.toLocaleString('fr-FR')} FCFA
                    </p>
                  )}
                </div>
              </div>
            </div>
            
            {walletBalance?.pending !== undefined && (
              <div className="bg-gradient-to-r from-blue-50 to-cyan-50 px-4 py-2 rounded-lg border border-blue-200">
                <div className="flex items-center space-x-2">
                  <TrendingUp className="w-4 h-4 text-blue-600" />
                  <div>
                    <p className="text-xs text-blue-600 font-medium">En attente</p>
                    <p className="text-sm font-bold text-blue-700">
                      +{walletBalance.pending.toLocaleString('fr-FR')} FCFA
                    </p>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
        
        {/* Right Section - Actions & User */}
        <div className="flex items-center space-x-3">
          {/* Date & Time */}
          <div className="hidden lg:block text-right">
            <p className="text-xs text-gray-500 font-medium">{formatDate(currentTime)}</p>
            <p className="text-sm font-bold text-gray-700">{formatTime(currentTime)}</p>
          </div>

          {/* Notifications */}
          <div className="relative">
            <button
              onClick={() => setShowNotifications(!showNotifications)}
              className="p-2 rounded-full hover:bg-gray-100 transition-colors relative"
              disabled={isNotificationsLoading}
            >
              <Activity className="w-5 h-5 text-gray-600" />
              {!isNotificationsLoading && notifications.length > 0 && (
                <span className="absolute top-0 right-0 w-2 h-2 bg-red-500 rounded-full"></span>
              )}
              {isNotificationsLoading && (
                <span className="absolute top-0 right-0 w-2 h-2 bg-blue-500 rounded-full animate-pulse"></span>
              )}
            </button>
            {showNotifications && (
              <div className="absolute right-0 mt-2 w-80 bg-white rounded-lg shadow-xl border border-gray-200 overflow-hidden z-50">
                <div className="p-3 border-b border-gray-200 bg-gray-50">
                  <div className="flex justify-between items-center">
                    <h3 className="font-semibold text-gray-800">
                      {isNotificationsLoading ? 'Chargement...' : 'Notifications'}
                    </h3>
                    {!isNotificationsLoading && notifications.length > 0 && (
                      <button className="text-xs text-blue-600 hover:text-blue-800">
                        Tout marquer comme lu
                      </button>
                    )}
                  </div>
                </div>
                <div className="max-h-96 overflow-y-auto">
                  {isNotificationsLoading ? (
                    <div className="p-4 flex justify-center">
                      <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500"></div>
                    </div>
                  ) : notifications.length > 0 ? (
                    notifications.map((notification) => (
                      <div
                        key={notification.id}
                        className="p-3 border-b border-gray-100 hover:bg-gray-50 cursor-pointer transition-colors"
                      >
                        <div className="flex items-start space-x-3">
                          <div className={`p-2 rounded-full ${
                            notification.type === 'success' ? 'bg-green-100 text-green-600' :
                            notification.type === 'warning' ? 'bg-yellow-100 text-yellow-600' :
                            notification.type === 'error' ? 'bg-red-100 text-red-600' :
                            'bg-blue-100 text-blue-600'
                          }`}>
                            <notification.icon className="w-4 h-4" />
                          </div>
                          <div className="flex-1 min-w-0">
                            <p className="text-sm font-medium text-gray-900 truncate">{notification.title}</p>
                            <p className="text-sm text-gray-500 mt-1">{notification.message}</p>
                            <p className="text-xs text-gray-400 mt-1">{notification.time}</p>
                          </div>
                        </div>
                      </div>
                    ))
                  ) : (
                    <div className="p-4 text-center text-gray-500">
                      <p>Aucune notification récente</p>
                    </div>
                  )}
                </div>
                <div className="p-3 border-t border-gray-200 bg-gray-50 text-center">
                  <a href="/notifications" className="text-sm font-medium text-blue-600 hover:text-blue-800">
                    Voir toutes les notifications
                  </a>
                </div>
              </div>
            )}
          </div>

          {/* Quick Actions */}
          <div className="relative">
            <button
              onClick={() => setShowQuickActions(!showQuickActions)}
              className="p-3 text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-xl transition-all duration-200 group"
            >
              <MoreVertical className="w-5 h-5 group-hover:scale-110 transition-transform" />
            </button>
            
            {showQuickActions && (
              <div className="absolute right-0 mt-2 w-64 bg-white rounded-xl shadow-xl border border-gray-200 py-3 z-50">
                <div className="px-4 py-2 border-b border-gray-100">
                  <h3 className="font-semibold text-gray-900">Actions Rapides</h3>
                </div>
                <div className="p-2">
                  {/* Actions par défaut */}
                  <button
                    className="w-full text-left px-3 py-3 text-sm hover:bg-blue-50 hover:text-blue-700 flex items-center transition-colors group"
                  >
                    <div className="p-2 bg-blue-100 text-blue-600 rounded-lg mr-3 group-hover:scale-110 transition-transform">
                      <TrendingUp className="w-4 h-4" />
                    </div>
                    <span className="font-medium text-gray-700">Nouveau Transfert</span>
                  </button>
                  <button
                    className="w-full text-left px-3 py-3 text-sm hover:bg-green-50 hover:text-green-700 flex items-center transition-colors group"
                  >
                    <div className="p-2 bg-green-100 text-green-600 rounded-lg mr-3 group-hover:scale-110 transition-transform">
                      <CreditCard className="w-4 h-4" />
                    </div>
                    <span className="font-medium text-gray-700">Demander un Prêt</span>
                  </button>
                  <button
                    className="w-full text-left px-3 py-3 text-sm hover:bg-purple-50 hover:text-purple-700 flex items-center transition-colors group"
                  >
                    <div className="p-2 bg-purple-100 text-purple-600 rounded-lg mr-3 group-hover:scale-110 transition-transform">
                      <Wallet className="w-4 h-4" />
                    </div>
                    <span className="font-medium text-gray-700">Recharger</span>
                  </button>
                  <button
                    className="w-full text-left px-3 py-3 text-sm hover:bg-orange-50 hover:text-orange-700 flex items-center transition-colors group"
                  >
                    <div className="p-2 bg-orange-100 text-orange-600 rounded-lg mr-3 group-hover:scale-110 transition-transform">
                      <Activity className="w-4 h-4" />
                    </div>
                    <span className="font-medium text-gray-700">Voir Rapports</span>
                  </button>
                </div>
              </div>
            )}
          </div>

          {/* Fullscreen Toggle */}
          <button
            onClick={toggleFullscreen}
            className="p-2 rounded-full hover:bg-gray-100 transition-colors"
            aria-label="Basculer en plein écran"
          >
            <Maximize2 className="w-5 h-5 text-gray-600" />
          </button>
          
          {/* Enhanced User Menu */}
          <div className="relative">
            <button
              onClick={() => setShowUserMenu(!showUserMenu)}
              className="flex items-center space-x-3 p-2 text-gray-600 hover:text-gray-900 hover:bg-gray-100 rounded-xl transition-all duration-200 group"
            >
              <div className="w-10 h-10 bg-gradient-to-br from-blue-500 via-purple-500 to-pink-500 rounded-xl flex items-center justify-center shadow-lg group-hover:scale-105 transition-transform">
                <User className="w-5 h-5 text-white" />
              </div>
              <div className="hidden md:block text-left">
                <p className="text-sm font-semibold text-gray-900">
                  {user?.firstName} {user?.lastName}
                </p>
                <p className="text-xs text-gray-500 capitalize flex items-center">
                  <div className="relative">
                    <span className={`inline-block w-2 h-2 rounded-full mr-2 ${
                      user?.role === 'admin' ? 'bg-purple-400' :
                      user?.role === 'manager' ? 'bg-blue-400' : 'bg-green-400'
                    }`}></span>
                    {user?.role === 'admin' ? 'Administrateur' : 
                     user?.role === 'manager' ? 'Gestionnaire' : 'Utilisateur'}
                  </div>
                </p>
              </div>
            </button>
            
            {showUserMenu && (
              <div className="absolute right-0 mt-2 w-64 bg-white rounded-xl shadow-2xl border border-gray-200 py-3 z-50">
                <div className="px-6 py-4 border-b border-gray-100">
                  <div className="flex items-center space-x-3">
                    <div className="w-12 h-12 bg-gradient-to-br from-blue-500 via-purple-500 to-pink-500 rounded-xl flex items-center justify-center shadow-lg">
                      <User className="w-6 h-6 text-white" />
                    </div>
                    <div>
                      <p className="text-sm font-semibold text-gray-900">
                        {user?.firstName} {user?.lastName}
                      </p>
                      <p className="text-xs text-gray-500">{user?.email}</p>
                      <span className={`inline-block px-2 py-1 text-xs rounded-full mt-1 font-medium ${
                        user?.role === 'admin' ? 'bg-purple-100 text-purple-600' :
                        user?.role === 'manager' ? 'bg-blue-100 text-blue-600' : 'bg-green-100 text-green-600'
                      }`}>
                        {user?.role === 'admin' ? 'Administrateur' : 
                         user?.role === 'manager' ? 'Gestionnaire' : 'Utilisateur'}
                      </span>
                    </div>
                  </div>
                </div>
                
                <div className="py-2">
                  <button className="w-full text-left px-6 py-3 text-sm text-gray-700 hover:bg-blue-50 hover:text-blue-700 flex items-center transition-colors group">
                    <div className="p-2 bg-blue-100 text-blue-600 rounded-lg mr-3 group-hover:scale-110 transition-transform">
                      <Wallet className="w-4 h-4" />
                    </div>
                    <div>
                      <p className="font-medium">Mon Portefeuille</p>
                      <p className="text-xs text-gray-500">Gérer mes finances</p>
                    </div>
                  </button>
                  
                  <button className="w-full text-left px-6 py-3 text-sm text-gray-700 hover:bg-green-50 hover:text-green-700 flex items-center transition-colors group">
                    <div className="p-2 bg-green-100 text-green-600 rounded-lg mr-3 group-hover:scale-110 transition-transform">
                      <Activity className="w-4 h-4" />
                    </div>
                    <div>
                      <p className="font-medium">Activité</p>
                      <p className="text-xs text-gray-500">Voir l'historique</p>
                    </div>
                  </button>
                  
                  <button className="w-full text-left px-6 py-3 text-sm text-gray-700 hover:bg-purple-50 hover:text-purple-700 flex items-center transition-colors group">
                    <div className="p-2 bg-purple-100 text-purple-600 rounded-lg mr-3 group-hover:scale-110 transition-transform">
                      <Settings className="w-4 h-4" />
                    </div>
                    <div>
                      <p className="font-medium">Paramètres</p>
                      <p className="text-xs text-gray-500">Configuration</p>
                    </div>
                  </button>
                  
                  <button className="w-full text-left px-6 py-3 text-sm text-gray-700 hover:bg-orange-50 hover:text-orange-700 flex items-center transition-colors group">
                    <div className="p-2 bg-orange-100 text-orange-600 rounded-lg mr-3 group-hover:scale-110 transition-transform">
                      <HelpCircle className="w-4 h-4" />
                    </div>
                    <div>
                      <p className="font-medium">Aide & Support</p>
                      <p className="text-xs text-gray-500">Besoin d'aide ?</p>
                    </div>
                  </button>
                </div>
                
                <div className="border-t border-gray-100 pt-2">
                  <button
                    onClick={handleLogout}
                    className="w-full text-left px-6 py-3 text-sm text-red-600 hover:bg-red-50 hover:text-red-700 flex items-center transition-colors group"
                  >
                    <div className="p-2 bg-red-100 text-red-600 rounded-lg mr-3 group-hover:scale-110 transition-transform">
                      <LogOut className="w-4 h-4" />
                    </div>
                    <div>
                      <p className="font-medium">Déconnexion</p>
                      <p className="text-xs text-red-400">Quitter la session</p>
                    </div>
                  </button>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
      
      {/* Progress Bar for Loading States */}
      <div className="h-1 bg-gradient-to-r from-blue-500 via-purple-500 to-pink-500 opacity-0 transition-opacity duration-300"></div>
    </header>
  );
};

export default Header;