import React from 'react';
import { LucideIcon } from 'lucide-react';

interface WidgetProps {
  title: string;
  value: string | number;
  change?: number;
  changeType?: 'increase' | 'decrease';
  icon: LucideIcon;
  color: 'blue' | 'green' | 'purple' | 'red' | 'orange' | 'indigo' | 'amber' | 'gold' | 'copper';
  description?: string;
  trend?: number[];
  className?: string;
}

const Widget: React.FC<WidgetProps> = ({
  title,
  value,
  change,
  changeType,
  icon: Icon,
  color,
  description,
  trend,
  className = ''
}) => {
  const colorClasses = {
    blue: {
      bg: 'from-blue-500 to-blue-600',
      text: 'text-blue-600',
      light: 'from-blue-50 to-blue-100',
      trend: 'from-blue-200 to-blue-400'
    },
    green: {
      bg: 'from-green-500 to-green-600',
      text: 'text-green-600',
      light: 'from-green-50 to-green-100',
      trend: 'from-green-200 to-green-400'
    },
    purple: {
      bg: 'from-purple-500 to-purple-600',
      text: 'text-purple-600',
      light: 'from-purple-50 to-purple-100',
      trend: 'from-purple-200 to-purple-400'
    },
    red: {
      bg: 'from-red-500 to-red-600',
      text: 'text-red-600',
      light: 'from-red-50 to-red-100',
      trend: 'from-red-200 to-red-400'
    },
    orange: {
      bg: 'from-orange-500 to-orange-600',
      text: 'text-orange-600',
      light: 'from-orange-50 to-orange-100',
      trend: 'from-orange-200 to-orange-400'
    },
    indigo: {
      bg: 'from-indigo-500 to-indigo-600',
      text: 'text-indigo-600',
      light: 'from-indigo-50 to-indigo-100',
      trend: 'from-indigo-200 to-indigo-400'
    },
    amber: {
      bg: 'from-amber-500 via-amber-600 to-orange-600',
      text: 'text-amber-600',
      light: 'from-amber-50 to-orange-50',
      trend: 'from-amber-200 to-orange-400'
    },
    gold: {
      bg: 'from-yellow-400 via-amber-500 to-orange-600',
      text: 'text-amber-700',
      light: 'from-yellow-50 to-amber-50',
      trend: 'from-yellow-200 to-amber-400'
    },
    copper: {
      bg: 'from-orange-500 via-red-500 to-orange-600',
      text: 'text-orange-600',
      light: 'from-orange-50 to-red-50',
      trend: 'from-orange-200 to-red-400'
    }
  };

  const colors = colorClasses[color];

  return (
    <div className={`group bg-gradient-to-br from-amber-50/90 to-orange-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-4 hover:shadow-xl transition-all duration-300 transform hover:scale-105 ${className}`}>
      {/* Header */}
      <div className="flex items-center justify-between mb-3">
        <div className={`p-2 rounded-lg bg-gradient-to-br ${colors.bg} shadow-md group-hover:scale-110 transition-transform`}>
          <Icon className="w-5 h-5 text-white" />
        </div>
        {change !== undefined && (
          <div className={`flex items-center space-x-1 px-3 py-1 rounded-full text-sm font-semibold ${
            changeType === 'increase' ? 'bg-amber-100 text-amber-700' : 'bg-orange-100 text-orange-700'
          }`}>
            <span className={changeType === 'increase' ? '↗' : '↘'}>
              {changeType === 'increase' ? '↗' : '↘'}
            </span>
            <span>{Math.abs(change)}%</span>
          </div>
        )}
      </div>
      
      {/* Content */}
      <div className="mb-3">
        <h3 className="text-xs font-medium text-amber-600 mb-1">{title}</h3>
        <p className="text-xl font-bold text-amber-900 mb-1">{value}</p>
        {description && (
          <p className="text-xs text-amber-700">{description}</p>
        )}
      </div>
      
      {/* Trend Chart */}
      {trend && trend.length > 0 && (
        <div className="h-8 flex items-end space-x-1">
          {trend.map((trendValue, idx) => {
            const height = (trendValue / Math.max(...trend)) * 100;
            return (
              <div
                key={idx}
                className={`flex-1 rounded-t bg-gradient-to-t opacity-70 ${colors.trend} transition-all duration-500 hover:opacity-90`}
                style={{ height: `${height}%` }}
              />
            );
          })}
        </div>
      )}
    </div>
  );
};

export default Widget;
