import React, { useState } from 'react';
import { 
  X, 
  Download, 
  FileText, 
  BarChart3, 
  PieChart, 
  Calendar,
  Filter,
  Eye
} from 'lucide-react';
import { useReportData } from '../../hooks/useReportData';

interface ReportGeneratorProps {
  isOpen: boolean;
  onClose: () => void;
  isAdmin: boolean;
}

const ReportGenerator: React.FC<ReportGeneratorProps> = ({
  isOpen,
  onClose,
  isAdmin
}) => {
  const [activeTab, setActiveTab] = useState<'chart' | 'pie' | 'table'>('chart');
  const [dateRange, setDateRange] = useState('30d');
  const [reportType, setReportType] = useState('financial');

  // Récupérer les données de rapport avec le hook
  const { chartData: realChartData, pieData: realPieData, loading, stats } = useReportData(dateRange, reportType);

  if (!isOpen) return null;

  // Utiliser les vraies données ou les données par défaut
  const chartData = realChartData.length > 0 ? realChartData : [
    { month: 'Jan', transactions: 0, volume: 0, nfc: 0 },
    { month: 'Fév', transactions: 0, volume: 0, nfc: 0 },
    { month: 'Mar', transactions: 0, volume: 0, nfc: 0 },
    { month: 'Avr', transactions: 0, volume: 0, nfc: 0 },
    { month: 'Mai', transactions: 0, volume: 0, nfc: 0 },
    { month: 'Jun', transactions: 0, volume: 0, nfc: 0 }
  ];

  const pieData = realPieData.length > 0 ? realPieData : [
    { label: 'Paiements NFC', value: 0, color: '#D97706', amount: 0 },
    { label: 'Recharges CinetPay', value: 0, color: '#F59E0B', amount: 0 },
    { label: 'Prêts SOS', value: 0, color: '#FBBF24', amount: 0 },
    { label: 'Virements', value: 0, color: '#EF4444', amount: 0 }
  ];

  const maxValue = Math.max(...chartData.map(d => d.transactions));

  const handleExport = (format: 'pdf' | 'excel' | 'csv') => {
    console.log(`Exporting report as ${format}...`);
    // Simulation d'export
    setTimeout(() => {
      alert(`Rapport exporté en ${format.toUpperCase()} avec succès !`);
    }, 1000);
  };

  const renderChart = () => (
    <div className="h-80 p-4">
      <div className="flex items-center justify-between mb-6">
        <h4 className="text-lg font-semibold text-amber-900">
          {isAdmin ? 'Évolution des Transactions' : 'Mon Activité Financière'}
        </h4>
        <div className="flex items-center space-x-4">
          <div className="flex items-center space-x-2">
            <div className="w-3 h-3 bg-amber-500 rounded-full"></div>
            <span className="text-sm text-amber-600">Transactions</span>
          </div>
          <div className="flex items-center space-x-2">
            <div className="w-3 h-3 bg-orange-500 rounded-full"></div>
            <span className="text-sm text-amber-600">Volume FCFA</span>
          </div>
        </div>
      </div>
      
      <div className="relative h-64">
        {/* Axes */}
        <div className="absolute left-0 top-0 h-full flex flex-col justify-between text-xs text-amber-500 pr-2">
          <span>{(maxValue / 1000).toFixed(0)}K</span>
          <span>{(maxValue * 0.75 / 1000).toFixed(0)}K</span>
          <span>{(maxValue * 0.5 / 1000).toFixed(0)}K</span>
          <span>{(maxValue * 0.25 / 1000).toFixed(0)}K</span>
          <span>0</span>
        </div>
        
        {/* Barres */}
        <div className="ml-8 h-full flex items-end justify-between space-x-2">
          {chartData.map((item, index) => (
            <div key={index} className="flex flex-col items-center flex-1">
              <div className="relative w-full max-w-12 h-48 flex flex-col justify-end space-y-1">
                {/* Barre transactions */}
                <div 
                  className="bg-amber-500 rounded-t-sm transition-all duration-500 ease-out hover:bg-amber-600"
                  style={{ 
                    height: `${(item.transactions / maxValue) * 100}%`,
                    minHeight: '4px'
                  }}
                  title={`${item.transactions.toLocaleString()} transactions`}
                ></div>
                
                {/* Barre NFC */}
                <div 
                  className="bg-orange-500 rounded-t-sm transition-all duration-500 ease-out hover:bg-orange-600"
                  style={{ 
                    height: `${(item.nfc / maxValue) * 100}%`,
                    minHeight: '4px'
                  }}
                  title={`${item.nfc.toLocaleString()} paiements NFC`}
                ></div>
              </div>
              
              <span className="text-xs text-gray-500 mt-2">{item.month}</span>
            </div>
          ))}
        </div>
      </div>
    </div>
  );

  const renderPieChart = () => {
    const size = 200;
    const strokeWidth = 30;
    const radius = (size - strokeWidth) / 2;
    const circumference = radius * 2 * Math.PI;
    let cumulativePercentage = 0;

    // Calculer le total réel à partir des données
    const totalAmount = pieData.reduce((sum, item) => sum + item.amount, 0);

    return (
      <div className="h-80 p-4">
        <div className="flex items-center justify-between mb-6">
          <h4 className="text-lg font-semibold text-gray-900">
            Répartition par Type de Transaction
          </h4>
          <span className="text-sm text-gray-600">
            Total: {(stats?.total_volume || totalAmount).toLocaleString('fr-FR')} FCFA
          </span>
        </div>
        
        <div className="flex items-center justify-center space-x-8">
          {/* Graphique circulaire */}
          <div className="relative">
            <svg width={size} height={size} className="transform -rotate-90">
              <circle
                cx={size / 2}
                cy={size / 2}
                r={radius}
                fill="none"
                stroke="#f3f4f6"
                strokeWidth={strokeWidth}
              />
              
              {pieData.map((item, index) => {
                const strokeDasharray = `${item.value * circumference / 100} ${circumference}`;
                const strokeDashoffset = -cumulativePercentage * circumference / 100;
                cumulativePercentage += item.value;
                
                return (
                  <circle
                    key={index}
                    cx={size / 2}
                    cy={size / 2}
                    r={radius}
                    fill="none"
                    stroke={item.color}
                    strokeWidth={strokeWidth}
                    strokeDasharray={strokeDasharray}
                    strokeDashoffset={strokeDashoffset}
                    strokeLinecap="round"
                    className="transition-all duration-500 ease-out hover:opacity-80"
                  />
                );
              })}
            </svg>
            
            <div className="absolute inset-0 flex flex-col items-center justify-center">
              <span className="text-sm text-gray-600">Total</span>
              <span className="text-lg font-bold text-gray-900">100%</span>
            </div>
          </div>

          {/* Légende */}
          <div className="space-y-3">
            {pieData.map((item, index) => (
              <div key={index} className="flex items-center justify-between min-w-48">
                <div className="flex items-center space-x-3">
                  <div 
                    className="w-4 h-4 rounded-full"
                    style={{ backgroundColor: item.color }}
                  ></div>
                  <span className="text-sm text-gray-700">{item.label}</span>
                </div>
                <div className="text-right">
                  <div className="text-sm font-semibold text-gray-900">
                    {item.amount.toLocaleString()} FCFA
                  </div>
                  <div className="text-xs text-gray-500">
                    {item.value}%
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>
    );
  };

  const renderTable = () => (
    <div className="h-80 p-4 overflow-y-auto">
      <div className="flex items-center justify-between mb-6">
        <h4 className="text-lg font-semibold text-gray-900">
          Données Détaillées
        </h4>
        <span className="text-sm text-gray-600">6 derniers mois</span>
      </div>
      
      <div className="overflow-x-auto">
        <table className="w-full text-sm">
          <thead className="bg-gray-50">
            <tr>
              <th className="text-left py-3 px-4 font-medium text-gray-900">Mois</th>
              <th className="text-right py-3 px-4 font-medium text-gray-900">Transactions</th>
              <th className="text-right py-3 px-4 font-medium text-gray-900">Volume (FCFA)</th>
              <th className="text-right py-3 px-4 font-medium text-gray-900">NFC</th>
              <th className="text-right py-3 px-4 font-medium text-gray-900">Croissance</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-gray-200">
            {chartData.map((item, index) => {
              const prevItem = chartData[index - 1];
              const growth = prevItem 
                ? ((item.transactions - prevItem.transactions) / prevItem.transactions * 100)
                : 0;
              
              return (
                <tr key={index} className="hover:bg-gray-50">
                  <td className="py-3 px-4 font-medium text-gray-900">{item.month}</td>
                  <td className="py-3 px-4 text-right text-gray-700">
                    {item.transactions.toLocaleString()}
                  </td>
                  <td className="py-3 px-4 text-right text-gray-700">
                    {item.volume.toLocaleString()}
                  </td>
                  <td className="py-3 px-4 text-right text-gray-700">
                    {item.nfc.toLocaleString()}
                  </td>
                  <td className={`py-3 px-4 text-right font-medium ${
                    growth >= 0 ? 'text-green-600' : 'text-red-600'
                  }`}>
                    {growth >= 0 ? '+' : ''}{growth.toFixed(1)}%
                  </td>
                </tr>
              );
            })}
          </tbody>
        </table>
      </div>
    </div>
  );

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-gradient-to-br from-amber-50 to-orange-50 rounded-xl shadow-2xl w-full max-w-4xl max-h-[90vh] overflow-hidden">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-amber-200">
          <div className="flex items-center space-x-3">
            <FileText className="w-6 h-6 text-amber-600" />
            <div>
              <h3 className="text-xl font-bold text-amber-900">
                {isAdmin ? 'Rapport Plateforme TIM CASH' : 'Mon Rapport Personnel'}
              </h3>
              <p className="text-sm text-amber-600">
                Analyse interactive des données financières
              </p>
            </div>
          </div>
          <button
            onClick={onClose}
            className="p-2 text-amber-400 hover:text-amber-600 hover:bg-amber-100 rounded-lg transition-colors"
          >
            <X className="w-5 h-5" />
          </button>
        </div>

        {/* Filters */}
        <div className="p-6 border-b border-amber-200 bg-amber-50">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <div className="flex items-center space-x-2">
                <Calendar className="w-4 h-4 text-amber-500" />
                <select
                  value={dateRange}
                  onChange={(e) => setDateRange(e.target.value)}
                  className="px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                >
                  <option value="7d">7 derniers jours</option>
                  <option value="30d">30 derniers jours</option>
                  <option value="90d">90 derniers jours</option>
                  <option value="1y">1 année</option>
                </select>
              </div>
              
              <div className="flex items-center space-x-2">
                <Filter className="w-4 h-4 text-amber-500" />
                <select
                  value={reportType}
                  onChange={(e) => setReportType(e.target.value)}
                  className="px-3 py-2 border border-amber-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-amber-500"
                >
                  <option value="financial">Rapport Financier</option>
                  <option value="nfc">Transactions NFC</option>
                  <option value="loans">Prêts SOS</option>
                  <option value="recharges">Recharges</option>
                </select>
              </div>
            </div>

            <div className="flex items-center space-x-2">
              <button
                onClick={() => handleExport('pdf')}
                className="flex items-center space-x-2 px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700 transition-colors"
              >
                <Download className="w-4 h-4" />
                <span>PDF</span>
              </button>
              <button
                onClick={() => handleExport('excel')}
                className="flex items-center space-x-2 px-4 py-2 bg-orange-600 text-white rounded-lg hover:bg-orange-700 transition-colors"
              >
                <Download className="w-4 h-4" />
                <span>Excel</span>
              </button>
              <button
                onClick={() => handleExport('csv')}
                className="flex items-center space-x-2 px-4 py-2 bg-yellow-600 text-white rounded-lg hover:bg-yellow-700 transition-colors"
              >
                <Download className="w-4 h-4" />
                <span>CSV</span>
              </button>
            </div>
          </div>
        </div>

        {/* Tabs */}
        <div className="flex border-b border-amber-200">
          <button
            onClick={() => setActiveTab('chart')}
            className={`flex items-center space-x-2 px-6 py-4 font-medium transition-colors ${
              activeTab === 'chart'
                ? 'text-amber-600 border-b-2 border-amber-600 bg-amber-50'
                : 'text-amber-600 hover:text-amber-900 hover:bg-amber-50'
            }`}
          >
            <BarChart3 className="w-4 h-4" />
            <span>Graphique</span>
          </button>
          <button
            onClick={() => setActiveTab('pie')}
            className={`flex items-center space-x-2 px-6 py-4 font-medium transition-colors ${
              activeTab === 'pie'
                ? 'text-amber-600 border-b-2 border-amber-600 bg-amber-50'
                : 'text-amber-600 hover:text-amber-900 hover:bg-amber-50'
            }`}
          >
            <PieChart className="w-4 h-4" />
            <span>Diagramme</span>
          </button>
          <button
            onClick={() => setActiveTab('table')}
            className={`flex items-center space-x-2 px-6 py-4 font-medium transition-colors ${
              activeTab === 'table'
                ? 'text-amber-600 border-b-2 border-amber-600 bg-amber-50'
                : 'text-amber-600 hover:text-amber-900 hover:bg-amber-50'
            }`}
          >
            <Eye className="w-4 h-4" />
            <span>Tableau</span>
          </button>
        </div>

        {/* Content */}
        <div className="bg-white">
          {loading ? (
            <div className="h-80 flex items-center justify-center">
              <div className="text-center">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-amber-600 mx-auto mb-4"></div>
                <p className="text-amber-600">Chargement des données...</p>
              </div>
            </div>
          ) : (
            <>
              {activeTab === 'chart' && renderChart()}
              {activeTab === 'pie' && renderPieChart()}
              {activeTab === 'table' && renderTable()}
            </>
          )}
        </div>

        {/* Footer */}
        <div className="p-6 border-t border-gray-200 bg-gray-50">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4 text-sm text-gray-600">
              <span>Généré le {new Date().toLocaleDateString('fr-FR')}</span>
              <span>•</span>
              <span>TIM CASH Dashboard</span>
              {stats && (
                <>
                  <span>•</span>
                  <span>{stats.total_transactions} transactions</span>
                  <span>•</span>
                  <span>{stats.total_volume.toLocaleString('fr-FR')} FCFA</span>
                </>
              )}
            </div>
            <button
              onClick={onClose}
              className="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors"
            >
              Fermer
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ReportGenerator;
